/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.
*/

/*
   Benchmark ZKP of knowledge of factoring.
 */

#include "bench.h"
#include "amcl/mta.h"

#define MIN_TIME 5.0
#define MIN_ITERS 10

// Primes for Paillier key
char *P_hex = "94f689d07ba20cf7c7ca7ccbed22ae6b40c426db74eaee4ce0ced2b6f52a5e136663f5f1ef379cdbb0c4fdd6e4074d6cff21082d4803d43d89e42fd8dfa82b135aa31a8844ffea25f255f956cbc1b9d8631d01baf1010d028a190b94ce40f3b72897e8196df19edf1ff62e6556f2701d52cef1442e3301db7608ecbdcca703db";
char *Q_hex = "9a9ad73f246df853e129c589925fdad9df05606a61081e62e72be4fb33f6e5ec492cc734f28bfb71fbe2ba9a11e4c02e2c0d103a5cbb0a9d6402c07de63b1b995dd72ac8f29825d66923a088b421fb4d52b0b855d2f5dde2be9b0ca0cee6f7a94e5566735fe6cff1fcad3199602f88528d19aa8d0263adff8f5053c38254a2a3";

// BC setup
char *PT_hex    = "CA5F37B7C0DDF6530B30A41116588218DE95F1F36B807FD7C28E4C467EE3F35967BC01D28B71F8A627A353675A81C86A1FF03DCECAF1686891183FA317BA34A4A1148D40A89F1F3AC0C200511C6CFE02342CD75354C25A2E069886DD4FB73BD365660D163F1282B143119AB8F375A73875EC16B634F52593B73BC6D875F2D3EF";
char *QT_hex    = "C2FC545C1C803F6C7625FBC4ECF9355734D6B6058FD714816D3ECFB93F1F705C9CE90D4F8796A05148AB5ABC201F90889231CC6BF5F68ED15EE4D901F603930A280EEABF10C613BFCB67A816363C839EB902B02607EB48AB8325E2B72620D4D294A232803217090DFB50AF8C620D4679E77CE3053437ED518F4F68840DCF1AA3";
char *ALPHA_hex = "1128dc85f9bbdde2826244bcefd0ec6668c19ee254b81bbbfc7575ec45922fc573567d45dc27fc659ec29e8909548a94f1d1ed280cfa49d75192c8cb04925884fa2e7ee9cce71bf5f699f73c07a9bcfbeed87aa4446099a940a03b6828a292319f3a4a71206bd902e9f99f6d6226344a14a0eb2b127b0e8925db779c21fa15ef 465212e8b5c0a8bd2fb3d171bfdad345d15676ad65f20447d8d28d9f7a3be092903966725054e94d95f7aff0ff854efeae993e9b97a2942fa7426cd1bfb843cd635c1058fb73d21ab7f9cc2319a307129f4f84369c01f0e29ea3716dfa692c56a3e4aae1437e9110464003afcb5a654661984f80eadefe04b511f2acd09a7ac5";
char *B0_hex    = "544c8b0766c7490f7c6abfe0517709f3ab2c9b81fa8455cd8f99302dc58efa8d73318b078b31e49336d05caae1be491e620ec4893dfd50153c75d99d81970995c48b73cbb379097f69d55d4fb07de6124388b30c5718ccc5bd251945a1a51de335a7ebc4e226d7a60d82a7afc485845e849228de10211b2b8d7a759dd24ec4a4 57fdae3380b96fa8f3e12ba112a2ea07c1a74484ae7938e80afd4f17e17dddb7257fdcddfbcf2d2c51f350fb0c30a4eed76625039e5310da553ceaad1f9993c3b25bff1a657800308d4864199baeec8036945a9ac2bb429bd92d568b500f65268743179451623d45e7e25234812de34c9c1b1db6ab2184800b97b7117d8247a7";

// Paillier ciphertext and plaintext
char* X_hex  = "0000000000000000000000000000000000000000000000000000000000000003";
char* Y_hex  = "0000000000000000000000000000000000000000000000000000000000000004";
char* C1_hex = "19c8b725dbd74b7dcaf72bd9ff2cd207b47cb1095393685906171af9e2f2959e7f68729e0e40f97a22bbca93373d618ad51dd077c0d102938598a8ecc8a656e978ebd14007da99db8e691d85fc18a428097ee8a63dcf95b84b660294474a20ed2edcf2b1b4f305c1cc25860a08d1348c2a4d24cc1a97b51f920e2985b8108b3392a5eafc443cf3449e288eb49dbde2228a56233afa5a6643e5ae6ec6aa8937a666ef74a30625c35bb22c3cc57b700f8eae7690f8d37edbfd27ccb2e882f70d0d85e0cc825347453a28e98e877ab1eeaa6efa09f034bc8976bffb86420106978066ff52221b315f71eb32cbf608d2b72cfa4c88e43282598f175b48ba3b5c14d72b2d90baabc00025450740ac89fc0dcd7d2f80cf12c721b6ec493c2025d7adc683b78f1d711b639a1b0dd043b9defa7ff928e257599dd95525bc8b45e1b88470311e11feb72749e5fc98f69051ddd1101b1bcc92f649681bd7ae316575444625d9d73d3684789142650951321e17f6b2f92103f36dbbd004cd66cda366e80faa4f57b71b9abb042f6cc932716fa3e6fdf50674e3d1e6d871f723d3f4f672c1270b41e7cdd5930a2572ddfc8ce370576a7a75ee6924f53122d717146c74eb6167811a2488bb899cc2da9dc2e29df66b5a03ed986fdad6ef177151ddd2698055050709c475b4ed5a2ab0be00c8b03e24193fb79f91cfd81fbcb838e45c25f8ba05";
char* C2_hex = "1f1f087e749c85aacdacaace8659a33b53baad5eec1e56628435d335a8b150f96865d6e090f53146e120e7089b6f4a91c762622b24d0d2fba0e703301170a0b826a1336d4d6bb83dccd29ad9ef0936614bf14e992ea4daa202c63ace9bd3f95b9a8a6edd7949e89ec165541e7c01bd41395baf3e2fe7f3a9611af8b5ed8639c02a2bfc236c17a136bef6d09f966db718f3df9d6f4f40b618b4b6058b4e4ec241e6c2424404d0aee0ef5cd666e5c4253a62ae9deb09289fb84657109e0b933f58871ba7ea77190d6ea45a04be68360478adf43a85851cf583c5575543578635996d2dcd020aeceabf18be6ff8b45e4ecd63c899cbfe353bc6be246aa421f54bb1f6aad797b36e435e2f33a3a049aeab894b851c5ce1076aa6e19316e3da6f539197e00e17e7a3025b53490a9d1210b900c1cac32a3bdc31d4c4866e7499a2858942e057be2840cf8ad4b1dcd914c64ac7d4b89e3f1b1a010096ecb62bb5837d9e79018870002962838bc46d7a70c23494985c300b4f8a7352a412bfc4134378d23343b3c8a77f65c234c8586e5fb0731881cb756e994c82773261f2a2321e45df45a08830e67b6c983e3f01a464b9ca6cc78ec7f170782748d114889656377e86a1e3b3c28616f3b4b73693867fefd7459fe99e9892435f0902ad74ceebac99c4f67340344f128b1f55fdb85acdc64891b77d9961653361f5264d5f1e0b67173b";
char* R_hex  = "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000018c5947fda2edea04c1f87c207e0bab17aff5f77ac21d04cb194631efd1f7256dc37de9473fc86009df36206974859c09023ac8179b02aacea8d89a01f4de161db955d450cef55ce959897636973b952371e349778e67c61ef6fae5f73fd728d423a594b6a76d5faca97d59d6ae40c53f3bd42dfccc93183e355422ba7af308a87d32c0352d478156275f98bc74e9ed4f2c7a9853c9f35b996fafe765b56c7f2e83771c6b676b75436e5c1697b838b3908aee92001cbccf3bf6cfb7aaea27a358a12cfe1ddde886b975ae14517e5912eba3ff9792e46403a998edd371020bbc5fbd6a705e669383303030ef79653ce16e13122233c626bb101ee8dd27bf4ff86";

// DLOG public ECP
char *ECPX_hex = "02f9308a019258c31049344f85f89d5229b531c845836f99b08601f113bce036f9";

// RV for Range Proof
char *ZK_ALPHA_hex = "000000000000000000000000000000000000000000000000000000000000000076f5e997c4867a05c7997d84a283473a8c611ed65a8cf0acfaf17d8930700af9bd20b9f17545d87ab92d9820be15f922d452027cabba1a66ccb1d816daaf84adda09ffbaa981b6985d55e3bbf440089a405f90fca71b0767ea7b7d3125965c8b";
char *BETA_hex     = "89d4ea9f023d28e5a69b21995668c4c12abc92bd7f6b2b446c37bb3d2db4e1235f4a213dcc3bdb054ddd6287fcc4987235916985505bfb16c81dab556ee8f8470e714a646600f532eaf2d6709cf93df9afcf4f0a97569e75a53a903b3b323dc38bd7db5231bef89ea6ebc46c364d21ad2fd547699e59625b98200d0f7375fb8d39b4ae6fba9c300fe6f565cf4d5625d127afd11da71b97db9c36c34221b70625307d91d0c3e447cbbfe4db3677a1bccd60f077e7cf4d1c3c51df8e359536676f0ad4a804f408fd37348f6b45102a75069812a7cd62c1a2aac267ea44e3b1cc499de47baab2450d97d16fb95879d1e82fd199ba05b8447cc4f8e75fd579ef0881";
char *GAMMA_hex    = "b5f19c6877cb4e4adf42b68c9bc70912bdaaa783f7c59bb03698095c964b84d2d453a2ad6f9948dc6fa0d2ce38115b96e74fddeaa33e2838ad7303eacab69816318d74d200269178971879e27e6c31a3dcbc0288b2adbd749f5249e03046ce73eb55ec7abc738b837d7973247c8fc4992232672ef58a9e423e3ed8ad55e9ce29b2b8f9839a6d61f04a4de0fdcd87f4bd3e36976b86257f275158b318b5ebe5e8e9098fc99213b1cd1266db9169c61170e19cd7c9f99224111e5d8dc4aca32def232bc236bdae32c5bff081abc92d7963de20258a75a13db67d3f5fd1567d836cc083f9e4bb251301f7de0604cca37956f06b5d3ce655b520bddecc910bb4ddb1";
char *RHO_hex      = "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000021d7fe4be0364f97b605cbc60fb1d081220fe45635b245f1f88ff35c0b9a00d7253df086e791a13a085752ae0ecc47bda62c98bf7f0dcd428eb75242c32277f41e23f1d41046464c89e9fe0a527e2196fee85564c2b2107388b6069a6f44b8f4354067fcebca5d0671b17146658aa4372a855984b1f9a102f35b1764dd1bb43dc3c14b08bd6230413cf54a428b6f24ffaa28120f38b8467bb574b32b7cce43044a0be2f9f0273f4cfc7032123b1507abb93d63a015380a0ccb4d543c9844b1a9d2c9a82a33da4c21e9aefd625475dcdab23472bd065a8f4da376c7b4a5380db7623bd5b69625eb06535e05fc07f64495508f7e52930cf5a66ba276c9518f5afebb8789fed64fd1d2bd2970dc2ed4d47e69a3387c19c7ab4f8fd62e1ceb14c0d7";
char *RHO1_hex     = "000000000000000000000000000000000000000000000000000000000000000058c0c9f5ddbf7bd65d5878ca82a4ea6e6f4887e0f400b326213a076685004ca34e7d6a56ef83c0aa9c95b5fcc44db71bb14cb4faf1ff717da92f57ddc1dd8e318ccd1b7de38bdbda1a193d5dc92f78092a90da77ec295817bc4129bd689ca860ce3f8dbb11065cdec047a2f8fe25fc2fc2c13b82347bee6b96b1444dcb4af664e8be05634e338f2a8c6931030f24f35e44aa4976a82312d46675e67afd30c505103050aedb1c15c8e613ebd81bc3301920254f91226afe3972bfdb569fe6a70938397e047afb83a853e2b2b1c090dbb89357424db8b8825ca6be59ca1a9fc8d9c02b88bd88d0d3b839649e6486aa0d4ac9399eadfb8fffedf39a3c1f2d3fb59efc62b4b220e69dfbd1dacfa45c561234f1b70fc192e705de2742b5456a7748b82469ed2ce8d11ce5e15b71df9f9b7350a0139f84b60965b1c5993927d02beb229fe6a8cf0c87100d7ab4d0996ecd09c8a4a6eed357d6eddd347296a705e00a52";
char *SIGMA_hex    = "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004b407580fb3003f6ee0e91502d0fe5bc85804356c143afa7e9c27254a19806619283e7cea7ad9791ebdb12a0901968560d422f274e833f9d5bcdf14238df3caf39a16ac84868e31a338f89966a9c17b608d6efbbc34c33c6a09e6f5752cf4bd34a6f3eb5170689c8393c07a828d8b3541a7c6132bf217f1797450fe38818c838e8ec10656a4e336d770df84e8e01b917001dbd0ca1e2255dd77f137f1dc481798bd4e7790b887c225d02cebaa052f285704fabc26dd235cfb84f25076c654a1d554facd6c0d075b2733280ffcb96b7837d263cd906af3312af58037f6c3ecd98019846a894830f3f9ed3880f040560fb9d5113e4c272a68583d8d19c3750c73b80cbd437952afec998671ddc655701b7d8ef63169d2dc847d382fb14a40f1a8b";
char *TAU_hex      = "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000765cade56daf04af46039c9b3ea3f64faac275c0d7472bcfe9ccb4f624c828876573a69a8588b18612feb9ab30aaf68358862f8da2a12625a2dc70740dfe6a54e86a1187184d5c0381d2d14a476e908bc27aba81fb190d9a938f1f1e7ddf01044fed18239b6b0248492c50f3568128d19387e59491537fc3c0bbe727247f0e8166384d7140788fd0f8d04eb06e41b15d5e2cca49908e15a1b048c06096a183cd4eea523c3ba745cd33532d2c8b56e60dcd085c346fa0a8847849e5856a9ceebe6e314c7b8348486fd898c2284d9fd00964135947100b15a77858a604f69b9f8caaae482c0ecbca2a0b98ce4da00abee65e86f3b689602c263027a1c79a9675835d0ee953e2f84e70aebbefa00c5b6b7666c508ea2d539f4f3dbaf0c9f3a17582";

int main()
{
    int rc;

    int iterations;
    clock_t start;
    double elapsed;

    PAILLIER_private_key priv_key;
    PAILLIER_public_key pub_key;
    COMMITMENTS_BC_priv_modulus priv_mod;
    COMMITMENTS_BC_pub_modulus pub_mod;

    MTA_ZKWC_commitment c;
    MTA_ZKWC_commitment_rv rv;
    MTA_ZKWC_proof proof;

    char c1[2*FS_2048];
    octet C1 = {0, sizeof(c1), c1};

    char c2[2*FS_2048];
    octet C2 = {0, sizeof(c2), c2};

    char r[2*FS_2048];
    octet R = {0, sizeof(r), r};

    char x[MODBYTES_256_56];
    octet X = {0, sizeof(x), x};

    char ecpx[EFS_SECP256K1 + 1];
    octet ECPX = {0, sizeof(ecpx), ecpx};

    char y[MODBYTES_256_56];
    octet Y = {0, sizeof(y), y};

    char e[MODBYTES_256_56];
    octet E = {0, sizeof(e), e};

    char p[HFS_2048];
    octet P = {0, sizeof(p), p};

    char q[HFS_2048];
    octet Q = {0, sizeof(q), q};

    char alpha[FS_2048];
    octet ALPHA = {0, sizeof(alpha), alpha};

    char b0[FS_2048];
    octet B0 = {0, sizeof(b0), b0};

    char oct[FS_2048 + HFS_2048];
    octet OCT = {0, sizeof(oct), oct};

    // Load paillier key
    OCT_fromHex(&P, P_hex);
    OCT_fromHex(&Q, Q_hex);
    PAILLIER_KEY_PAIR(NULL, &P, &Q, &pub_key, &priv_key);

    // Load DLOG ECP
    OCT_fromHex(&ECPX, ECPX_hex);

    // Generate BC commitment modulus
    OCT_fromHex(&P,     PT_hex);
    OCT_fromHex(&Q,     QT_hex);
    OCT_fromHex(&ALPHA, ALPHA_hex);
    OCT_fromHex(&B0,    B0_hex);
    COMMITMENTS_BC_setup(NULL, &priv_mod, &P, &Q, &ALPHA, &B0);
    COMMITMENTS_BC_export_public_modulus(&pub_mod, &priv_mod);

    // Load Paillier encryption values
    OCT_fromHex(&X,  X_hex);
    OCT_fromHex(&Y,  Y_hex);
    OCT_fromHex(&R,  R_hex);
    OCT_fromHex(&C1, C1_hex);
    OCT_fromHex(&C2, C2_hex);

    // Load Random Values for Range Proof
    OCT_fromHex(&OCT, ZK_ALPHA_hex);
    OCT_pad(&OCT, FS_2048);
    FF_2048_fromOctet(rv.alpha, &OCT, FFLEN_2048);

    OCT_fromHex(&OCT, BETA_hex);
    FF_2048_fromOctet(rv.beta, &OCT, FFLEN_2048);

    OCT_fromHex(&OCT, GAMMA_hex);
    FF_2048_fromOctet(rv.gamma, &OCT, FFLEN_2048);

    OCT_fromHex(&OCT, RHO_hex);
    FF_2048_fromOctet(rv.rho, &OCT, FFLEN_2048 + HFLEN_2048);

    OCT_fromHex(&OCT, RHO1_hex);
    FF_2048_fromOctet(rv.rho1, &OCT, FFLEN_2048 + HFLEN_2048);

    OCT_fromHex(&OCT, SIGMA_hex);
    FF_2048_fromOctet(rv.sigma, &OCT, FFLEN_2048 + HFLEN_2048);

    OCT_fromHex(&OCT, TAU_hex);
    FF_2048_fromOctet(rv.tau, &OCT, FFLEN_2048 + HFLEN_2048);

    print_system_info();

    printf("Timing info\n");
    printf("===========\n");

    iterations = 0;
    start = clock();
    do
    {
        MTA_ZKWC_commit(NULL, &pub_key, &pub_mod, &X, &Y, &C1, &c, &rv);
        iterations++;
        elapsed = (clock() - start) / (double)CLOCKS_PER_SEC;
    }
    while (elapsed < MIN_TIME || iterations < MIN_ITERS);

    elapsed = MILLISECOND * elapsed / iterations;
    printf("\tMTA_ZKWC_commit\t\t%8d iterations\t", iterations);
    printf("%8.2lf ms per iteration\n", elapsed);

    iterations = 0;
    start = clock();
    do
    {
        MTA_ZKWC_challenge(&pub_key, &pub_mod, &C1, &C2, &ECPX, &c, &E);
        iterations++;
        elapsed = (clock() - start) / (double)CLOCKS_PER_SEC;
    }
    while (elapsed < MIN_TIME || iterations < MIN_ITERS);

    elapsed = MICROSECOND * elapsed / iterations;
    printf("\tMTA_ZKWC_challenge\t%8d iterations\t", iterations);
    printf("%8.2lf us per iteration\n", elapsed);

    iterations = 0;
    start = clock();
    do
    {
        MTA_ZKWC_prove(&pub_key, &rv, &X, &Y, &R, &E, &proof);
        iterations++;
        elapsed = (clock() - start) / (double)CLOCKS_PER_SEC;
    }
    while (elapsed < MIN_TIME || iterations < MIN_ITERS);

    elapsed = MILLISECOND * elapsed / iterations;
    printf("\tMTA_ZKWC_prove\t\t%8d iterations\t", iterations);
    printf("%8.2lf ms per iteration\n", elapsed);

    iterations = 0;
    start = clock();
    do
    {
        rc = MTA_ZKWC_verify(&priv_key, &priv_mod, &C1, &C2, &ECPX, &E, &c, &proof);
        iterations++;
        elapsed = (clock() - start) / (double)CLOCKS_PER_SEC;
    }
    while (elapsed < MIN_TIME || iterations < MIN_ITERS);

    if (rc != MTA_OK)
    {
        printf("FAILURE MTA_ZKWC_verify: %d\n", rc);
        exit(EXIT_FAILURE);
    }

    elapsed = MILLISECOND * elapsed / iterations;
    printf("\tMTA_ZKWC_verify\t\t%8d iterations\t", iterations);
    printf("%8.2lf ms per iteration\n", elapsed);

    exit(EXIT_SUCCESS);
}
