/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.inlong.sort.singletenant.flink.serialization;

import org.apache.commons.lang3.StringUtils;
import org.apache.flink.api.common.serialization.SerializationSchema;
import org.apache.flink.formats.json.debezium.DebeziumJsonSerializationSchema;
import org.apache.flink.table.types.logical.RowType;
import org.apache.flink.types.Row;
import org.apache.inlong.sort.protocol.FieldInfo;
import org.apache.inlong.sort.protocol.serialization.DebeziumSerializationInfo;

import java.io.IOException;

import static org.apache.inlong.sort.singletenant.flink.serialization.SerializationSchemaFactory.MAP_NULL_KEY_LITERAL_DEFAULT;
import static org.apache.inlong.sort.singletenant.flink.serialization.SerializationSchemaFactory.getMapNullKeyMode;
import static org.apache.inlong.sort.singletenant.flink.utils.CommonUtils.convertDateToStringFormatInfo;
import static org.apache.inlong.sort.singletenant.flink.utils.CommonUtils.convertFieldInfosToRowType;
import static org.apache.inlong.sort.singletenant.flink.utils.CommonUtils.extractFormatInfos;
import static org.apache.inlong.sort.singletenant.flink.utils.CommonUtils.getTimestampFormatStandard;

public class DebeziumSerializationSchemaBuilder {

    public static SerializationSchema<Row> build(
            FieldInfo[] fieldInfos,
            DebeziumSerializationInfo serializationInfo
    ) throws IOException, ClassNotFoundException {
        String mapNullKeyLiteral = serializationInfo.getMapNullKeyLiteral();
        if (StringUtils.isEmpty(mapNullKeyLiteral)) {
            mapNullKeyLiteral = MAP_NULL_KEY_LITERAL_DEFAULT;
        }

        FieldInfo[] convertedFieldInfos = convertDateToStringFormatInfo(fieldInfos);
        RowType convertedRowType = convertFieldInfosToRowType(convertedFieldInfos);
        DebeziumJsonSerializationSchema debeziumSchema = new DebeziumJsonSerializationSchema(
                convertedRowType,
                getTimestampFormatStandard(serializationInfo.getTimestampFormatStandard()),
                getMapNullKeyMode(serializationInfo.getMapNullKeyMod()),
                mapNullKeyLiteral,
                serializationInfo.isEncodeDecimalAsPlainNumber()
        );

        RowToRowDataSerializationSchemaWrapper rowToRowDataSchema
                = new RowToRowDataSerializationSchemaWrapper(debeziumSchema, convertedFieldInfos);

        return new CustomDateFormatSerializationSchemaWrapper(rowToRowDataSchema, extractFormatInfos(fieldInfos));
    }

}
