/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.inlong.manager.web.controller;

import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.inlong.manager.common.beans.Response;
import org.apache.inlong.manager.common.enums.OperationType;
import org.apache.inlong.manager.common.pojo.source.SourceListResponse;
import org.apache.inlong.manager.common.pojo.source.SourcePageRequest;
import org.apache.inlong.manager.common.pojo.source.SourceRequest;
import org.apache.inlong.manager.common.pojo.source.SourceResponse;
import org.apache.inlong.manager.common.util.LoginUserUtils;
import org.apache.inlong.manager.service.core.operationlog.OperationLog;
import org.apache.inlong.manager.service.source.StreamSourceService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * Stream source control layer
 */
@RestController
@RequestMapping("/source")
@Api(tags = "Stream source config")
public class StreamSourceController {

    @Autowired
    StreamSourceService sourceService;

    @RequestMapping(value = "/save", method = RequestMethod.POST)
    @OperationLog(operation = OperationType.CREATE)
    @ApiOperation(value = "Save stream source")
    public Response<Integer> save(@Validated @RequestBody SourceRequest request) {
        return Response.success(sourceService.save(request, LoginUserUtils.getLoginUserDetail().getUserName()));
    }

    @RequestMapping(value = "/get/{id}", method = RequestMethod.GET)
    @ApiOperation(value = "Query stream source")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", dataTypeClass = Integer.class, required = true),
            @ApiImplicitParam(name = "sourceType", dataTypeClass = String.class, required = true)
    })
    public Response<SourceResponse> get(@PathVariable Integer id, @RequestParam String sourceType) {
        return Response.success(sourceService.get(id, sourceType));
    }

    @RequestMapping(value = "/list", method = RequestMethod.GET)
    @ApiOperation(value = "Query stream source list")
    public Response<PageInfo<? extends SourceListResponse>> listByCondition(SourcePageRequest request) {
        return Response.success(sourceService.listByCondition(request));
    }

    @RequestMapping(value = "/update", method = RequestMethod.POST)
    @OperationLog(operation = OperationType.UPDATE)
    @ApiOperation(value = "Modify stream source")
    public Response<Boolean> update(@Validated @RequestBody SourceRequest request) {
        return Response.success(sourceService.update(request, LoginUserUtils.getLoginUserDetail().getUserName()));
    }

    @RequestMapping(value = "/delete/{id}", method = RequestMethod.DELETE)
    @OperationLog(operation = OperationType.DELETE)
    @ApiOperation(value = "Delete stream source")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", dataTypeClass = Integer.class, required = true),
            @ApiImplicitParam(name = "sourceType", dataTypeClass = String.class, required = true)
    })
    public Response<Boolean> delete(@PathVariable Integer id, @RequestParam String sourceType) {
        boolean result = sourceService.delete(id, sourceType, LoginUserUtils.getLoginUserDetail().getUserName());
        return Response.success(result);
    }

}