/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.extensions.gcp.util;

import com.google.api.client.googleapis.batch.BatchRequest;
import com.google.api.client.googleapis.batch.json.JsonBatchCallback;
import com.google.api.client.googleapis.json.GoogleJsonError;
import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.api.client.http.HttpHeaders;
import com.google.api.client.http.HttpRequestInitializer;
import com.google.api.client.util.BackOff;
import com.google.api.client.util.Sleeper;
import com.google.api.services.storage.Storage;
import com.google.api.services.storage.model.Bucket;
import com.google.api.services.storage.model.Objects;
import com.google.api.services.storage.model.RewriteResponse;
import com.google.api.services.storage.model.StorageObject;
import com.google.auth.Credentials;
import com.google.auto.value.AutoValue;
import com.google.cloud.hadoop.gcsio.CreateObjectOptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorage;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageImpl;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageOptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.util.ApiErrorExtractor;
import com.google.cloud.hadoop.util.AsyncWriteChannelOptions;
import com.google.cloud.hadoop.util.ResilientOperation;
import com.google.cloud.hadoop.util.RetryDeterminer;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.AccessDeniedException;
import java.nio.file.FileAlreadyExistsException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.beam.sdk.extensions.gcp.options.GcsOptions;
import org.apache.beam.sdk.extensions.gcp.util.AutoValue_GcsUtil_StorageObjectOrIOException;
import org.apache.beam.sdk.extensions.gcp.util.BackOffAdapter;
import org.apache.beam.sdk.extensions.gcp.util.Transport;
import org.apache.beam.sdk.extensions.gcp.util.gcsfs.GcsPath;
import org.apache.beam.sdk.io.FileSystemUtils;
import org.apache.beam.sdk.options.DefaultValueFactory;
import org.apache.beam.sdk.options.ExperimentalOptions;
import org.apache.beam.sdk.options.PipelineOptions;
import org.apache.beam.sdk.util.FluentBackoff;
import org.apache.beam.sdk.util.MoreFutures;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.annotations.VisibleForTesting;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.base.Preconditions;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.collect.ImmutableList;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.collect.Lists;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.util.concurrent.ListeningExecutorService;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.util.concurrent.MoreExecutors;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GcsUtil {
    private static final Logger LOG = LoggerFactory.getLogger(GcsUtil.class);
    private static final long MAX_LIST_ITEMS_PER_CALL = 1024L;
    private static final Pattern GLOB_PREFIX = Pattern.compile("(?<PREFIX>[^\\[*?]*)[\\[*?].*");
    private static final int MAX_REQUESTS_PER_BATCH = 100;
    private static final int MAX_CONCURRENT_BATCHES = 256;
    private static final FluentBackoff BACKOFF_FACTORY = FluentBackoff.DEFAULT.withMaxRetries(10).withInitialBackoff(Duration.standardSeconds((long)1L));
    private Storage storageClient;
    private final HttpRequestInitializer httpRequestInitializer;
    private final @Nullable Integer uploadBufferSizeBytes;
    private final ApiErrorExtractor errorExtractor = new ApiErrorExtractor();
    final ExecutorService executorService;
    private Credentials credentials;
    private GoogleCloudStorage googleCloudStorage;
    private GoogleCloudStorageOptions googleCloudStorageOptions;
    @VisibleForTesting
    @Nullable Long maxBytesRewrittenPerCall;
    @VisibleForTesting
    @Nullable AtomicInteger numRewriteTokensUsed;

    public static String getNonWildcardPrefix(String globExp) {
        Matcher m = GLOB_PREFIX.matcher(globExp);
        Preconditions.checkArgument((boolean)m.matches(), (Object)String.format("Glob expression: [%s] is not expandable.", globExp));
        return m.group("PREFIX");
    }

    public static boolean isWildcard(GcsPath spec) {
        return GLOB_PREFIX.matcher(spec.getObject()).matches();
    }

    private GcsUtil(Storage storageClient, HttpRequestInitializer httpRequestInitializer, ExecutorService executorService, Boolean shouldUseGrpc, Credentials credentials, @Nullable Integer uploadBufferSizeBytes) {
        this.storageClient = storageClient;
        this.httpRequestInitializer = httpRequestInitializer;
        this.uploadBufferSizeBytes = uploadBufferSizeBytes;
        this.executorService = executorService;
        this.credentials = credentials;
        this.maxBytesRewrittenPerCall = null;
        this.numRewriteTokensUsed = null;
        this.googleCloudStorageOptions = GoogleCloudStorageOptions.builder().setAppName("Beam").setGrpcEnabled(shouldUseGrpc.booleanValue()).build();
        this.googleCloudStorage = new GoogleCloudStorageImpl(this.googleCloudStorageOptions, storageClient, credentials);
    }

    protected void setStorageClient(Storage storageClient) {
        this.storageClient = storageClient;
    }

    public List<GcsPath> expand(GcsPath gcsPattern) throws IOException {
        Objects objects;
        Pattern p = null;
        String prefix = null;
        if (!GcsUtil.isWildcard(gcsPattern)) {
            try {
                this.getObject(gcsPattern);
                return ImmutableList.of((Object)gcsPattern);
            }
            catch (FileNotFoundException e) {
                return ImmutableList.of();
            }
        }
        prefix = GcsUtil.getNonWildcardPrefix(gcsPattern.getObject());
        p = Pattern.compile(FileSystemUtils.wildcardToRegexp((String)gcsPattern.getObject()));
        LOG.debug("matching files in bucket {}, prefix {} against pattern {}", new Object[]{gcsPattern.getBucket(), prefix, p.toString()});
        String pageToken = null;
        ArrayList<GcsPath> results = new ArrayList<GcsPath>();
        while ((objects = this.listObjects(gcsPattern.getBucket(), prefix, pageToken)).getItems() != null) {
            for (StorageObject o : objects.getItems()) {
                String name = o.getName();
                if (!p.matcher(name).matches() || name.endsWith("/")) continue;
                LOG.debug("Matched object: {}", (Object)name);
                results.add(GcsPath.fromObject(o));
            }
            pageToken = objects.getNextPageToken();
            if (pageToken != null) continue;
        }
        return results;
    }

    @VisibleForTesting
    @Nullable Integer getUploadBufferSizeBytes() {
        return this.uploadBufferSizeBytes;
    }

    private static BackOff createBackOff() {
        return BackOffAdapter.toGcpBackOff(BACKOFF_FACTORY.backoff());
    }

    public long fileSize(GcsPath path) throws IOException {
        return this.getObject(path).getSize().longValue();
    }

    public StorageObject getObject(GcsPath gcsPath) throws IOException {
        return this.getObject(gcsPath, GcsUtil.createBackOff(), Sleeper.DEFAULT);
    }

    @VisibleForTesting
    StorageObject getObject(GcsPath gcsPath, BackOff backoff, Sleeper sleeper) throws IOException {
        Storage.Objects.Get getObject = this.storageClient.objects().get(gcsPath.getBucket(), gcsPath.getObject());
        try {
            return (StorageObject)ResilientOperation.retry(() -> ((Storage.Objects.Get)getObject).execute(), (BackOff)backoff, (RetryDeterminer)RetryDeterminer.SOCKET_ERRORS, IOException.class, (Sleeper)sleeper);
        }
        catch (IOException | InterruptedException e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            if (e instanceof IOException && this.errorExtractor.itemNotFound((IOException)e)) {
                throw new FileNotFoundException(gcsPath.toString());
            }
            throw new IOException(String.format("Unable to get the file object for path %s.", gcsPath), e);
        }
    }

    public List<StorageObjectOrIOException> getObjects(List<GcsPath> gcsPaths) throws IOException {
        ArrayList<StorageObjectOrIOException[]> results = new ArrayList<StorageObjectOrIOException[]>();
        GcsUtil.executeBatches(this.makeGetBatches(gcsPaths, results));
        ImmutableList.Builder ret = ImmutableList.builder();
        for (StorageObjectOrIOException[] result : results) {
            ret.add((Object)result[0]);
        }
        return ret.build();
    }

    public Objects listObjects(String bucket, String prefix, @Nullable String pageToken) throws IOException {
        return this.listObjects(bucket, prefix, pageToken, null);
    }

    public Objects listObjects(String bucket, String prefix, @Nullable String pageToken, @Nullable String delimiter) throws IOException {
        Storage.Objects.List listObject = this.storageClient.objects().list(bucket);
        listObject.setMaxResults(Long.valueOf(1024L));
        listObject.setPrefix(prefix);
        listObject.setDelimiter(delimiter);
        if (pageToken != null) {
            listObject.setPageToken(pageToken);
        }
        try {
            return (Objects)ResilientOperation.retry(() -> ((Storage.Objects.List)listObject).execute(), (BackOff)GcsUtil.createBackOff(), (RetryDeterminer)RetryDeterminer.SOCKET_ERRORS, IOException.class);
        }
        catch (Exception e) {
            throw new IOException(String.format("Unable to match files in bucket %s, prefix %s.", bucket, prefix), e);
        }
    }

    @VisibleForTesting
    List<Long> fileSizes(List<GcsPath> paths) throws IOException {
        List<StorageObjectOrIOException> results = this.getObjects(paths);
        ImmutableList.Builder ret = ImmutableList.builder();
        for (StorageObjectOrIOException result : results) {
            ret.add((Object)this.toFileSize(result));
        }
        return ret.build();
    }

    private Long toFileSize(StorageObjectOrIOException storageObjectOrIOException) throws IOException {
        if (storageObjectOrIOException.ioException() != null) {
            throw storageObjectOrIOException.ioException();
        }
        return storageObjectOrIOException.storageObject().getSize().longValue();
    }

    @VisibleForTesting
    void setCloudStorageImpl(GoogleCloudStorage g) {
        this.googleCloudStorage = g;
    }

    @VisibleForTesting
    void setCloudStorageImpl(GoogleCloudStorageOptions g) {
        this.googleCloudStorageOptions = g;
    }

    public SeekableByteChannel open(GcsPath path) throws IOException {
        return this.googleCloudStorage.open(new StorageResourceId(path.getBucket(), path.getObject()));
    }

    @VisibleForTesting
    SeekableByteChannel open(GcsPath path, GoogleCloudStorageReadOptions readOptions) throws IOException {
        return this.googleCloudStorage.open(new StorageResourceId(path.getBucket(), path.getObject()), readOptions);
    }

    public WritableByteChannel create(GcsPath path, String type) throws IOException {
        return this.create(path, type, this.uploadBufferSizeBytes);
    }

    public WritableByteChannel create(GcsPath path, String type, Integer uploadBufferSizeBytes) throws IOException {
        AsyncWriteChannelOptions wcOptions = this.googleCloudStorageOptions.getWriteChannelOptions();
        int uploadChunkSize = uploadBufferSizeBytes == null ? wcOptions.getUploadChunkSize() : uploadBufferSizeBytes.intValue();
        AsyncWriteChannelOptions newOptions = wcOptions.toBuilder().setUploadChunkSize(uploadChunkSize).build();
        GoogleCloudStorageOptions newGoogleCloudStorageOptions = this.googleCloudStorageOptions.toBuilder().setWriteChannelOptions(newOptions).build();
        GoogleCloudStorageImpl gcpStorage = new GoogleCloudStorageImpl(newGoogleCloudStorageOptions, this.storageClient, this.credentials);
        return gcpStorage.create(new StorageResourceId(path.getBucket(), path.getObject()), CreateObjectOptions.builder().setOverwriteExisting(true).setContentType(type).build());
    }

    public boolean bucketAccessible(GcsPath path) throws IOException {
        return this.bucketAccessible(path, GcsUtil.createBackOff(), Sleeper.DEFAULT);
    }

    public long bucketOwner(GcsPath path) throws IOException {
        return this.getBucket(path, GcsUtil.createBackOff(), Sleeper.DEFAULT).getProjectNumber().longValue();
    }

    public void createBucket(String projectId, Bucket bucket) throws IOException {
        this.createBucket(projectId, bucket, GcsUtil.createBackOff(), Sleeper.DEFAULT);
    }

    @VisibleForTesting
    boolean bucketAccessible(GcsPath path, BackOff backoff, Sleeper sleeper) throws IOException {
        try {
            return this.getBucket(path, backoff, sleeper) != null;
        }
        catch (FileNotFoundException | AccessDeniedException e) {
            return false;
        }
    }

    @VisibleForTesting
    @Nullable Bucket getBucket(GcsPath path, BackOff backoff, Sleeper sleeper) throws IOException {
        Storage.Buckets.Get getBucket = this.storageClient.buckets().get(path.getBucket());
        try {
            return (Bucket)ResilientOperation.retry(() -> ((Storage.Buckets.Get)getBucket).execute(), (BackOff)backoff, (RetryDeterminer)new RetryDeterminer<IOException>(){

                public boolean shouldRetry(IOException e) {
                    if (GcsUtil.this.errorExtractor.itemNotFound(e) || GcsUtil.this.errorExtractor.accessDenied(e)) {
                        return false;
                    }
                    return RetryDeterminer.SOCKET_ERRORS.shouldRetry((Exception)e);
                }
            }, IOException.class, (Sleeper)sleeper);
        }
        catch (GoogleJsonResponseException e) {
            if (this.errorExtractor.accessDenied((IOException)((Object)e))) {
                throw new AccessDeniedException(path.toString(), null, e.getMessage());
            }
            if (this.errorExtractor.itemNotFound((IOException)((Object)e))) {
                throw new FileNotFoundException(e.getMessage());
            }
            throw e;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException(String.format("Error while attempting to verify existence of bucket gs://%s", path.getBucket()), e);
        }
    }

    @VisibleForTesting
    void createBucket(String projectId, Bucket bucket, BackOff backoff, Sleeper sleeper) throws IOException {
        Storage.Buckets.Insert insertBucket = this.storageClient.buckets().insert(projectId, bucket);
        insertBucket.setPredefinedAcl("projectPrivate");
        insertBucket.setPredefinedDefaultObjectAcl("projectPrivate");
        try {
            ResilientOperation.retry(() -> ((Storage.Buckets.Insert)insertBucket).execute(), (BackOff)backoff, (RetryDeterminer)new RetryDeterminer<IOException>(){

                public boolean shouldRetry(IOException e) {
                    if (GcsUtil.this.errorExtractor.itemAlreadyExists(e) || GcsUtil.this.errorExtractor.accessDenied(e)) {
                        return false;
                    }
                    return RetryDeterminer.SOCKET_ERRORS.shouldRetry((Exception)e);
                }
            }, IOException.class, (Sleeper)sleeper);
            return;
        }
        catch (GoogleJsonResponseException e) {
            if (this.errorExtractor.accessDenied((IOException)((Object)e))) {
                throw new AccessDeniedException(bucket.getName(), null, e.getMessage());
            }
            if (this.errorExtractor.itemAlreadyExists((IOException)((Object)e))) {
                throw new FileAlreadyExistsException(bucket.getName(), null, e.getMessage());
            }
            throw e;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException(String.format("Error while attempting to create bucket gs://%s for rproject %s", bucket.getName(), projectId), e);
        }
    }

    private static void executeBatches(List<BatchRequest> batches) throws IOException {
        ListeningExecutorService executor = MoreExecutors.listeningDecorator((ExecutorService)new ThreadPoolExecutor(256, 256, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>()));
        ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>();
        for (BatchRequest batch : batches) {
            futures.add(MoreFutures.runAsync(() -> batch.execute(), (ExecutorService)executor));
        }
        try {
            MoreFutures.get((CompletionStage)MoreFutures.allAsList(futures));
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Interrupted while executing batch GCS request", e);
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof FileNotFoundException) {
                throw (FileNotFoundException)e.getCause();
            }
            throw new IOException("Error executing batch GCS request", e);
        }
        finally {
            executor.shutdown();
        }
    }

    @VisibleForTesting
    List<BatchRequest> makeGetBatches(Collection<GcsPath> paths, List<StorageObjectOrIOException[]> results) throws IOException {
        ArrayList<BatchRequest> batches = new ArrayList<BatchRequest>();
        for (List filesToGet : Lists.partition((List)Lists.newArrayList(paths), (int)100)) {
            BatchRequest batch = this.createBatchRequest();
            for (GcsPath path : filesToGet) {
                results.add(this.enqueueGetFileSize(path, batch));
            }
            batches.add(batch);
        }
        return batches;
    }

    public void copy(Iterable<String> srcFilenames, Iterable<String> destFilenames) throws IOException {
        LinkedList<RewriteOp> rewrites = this.makeRewriteOps(srcFilenames, destFilenames);
        while (rewrites.size() > 0) {
            GcsUtil.executeBatches(this.makeCopyBatches(rewrites));
        }
    }

    LinkedList<RewriteOp> makeRewriteOps(Iterable<String> srcFilenames, Iterable<String> destFilenames) throws IOException {
        ArrayList srcList = Lists.newArrayList(srcFilenames);
        ArrayList destList = Lists.newArrayList(destFilenames);
        Preconditions.checkArgument((srcList.size() == destList.size() ? 1 : 0) != 0, (String)"Number of source files %s must equal number of destination files %s", (int)srcList.size(), (int)destList.size());
        LinkedList rewrites = Lists.newLinkedList();
        for (int i = 0; i < srcList.size(); ++i) {
            GcsPath sourcePath = GcsPath.fromUri((String)srcList.get(i));
            GcsPath destPath = GcsPath.fromUri((String)destList.get(i));
            rewrites.addLast(new RewriteOp(sourcePath, destPath));
        }
        return rewrites;
    }

    List<BatchRequest> makeCopyBatches(LinkedList<RewriteOp> rewrites) throws IOException {
        ArrayList<BatchRequest> batches = new ArrayList<BatchRequest>();
        BatchRequest batch = this.createBatchRequest();
        Iterator it = rewrites.iterator();
        while (it.hasNext()) {
            RewriteOp rewrite = (RewriteOp)((Object)it.next());
            if (!rewrite.getReadyToEnqueue()) {
                it.remove();
                continue;
            }
            rewrite.enqueue(batch);
            if (batch.size() < 100) continue;
            batches.add(batch);
            batch = this.createBatchRequest();
        }
        if (batch.size() > 0) {
            batches.add(batch);
        }
        return batches;
    }

    List<BatchRequest> makeRemoveBatches(Collection<String> filenames) throws IOException {
        ArrayList<BatchRequest> batches = new ArrayList<BatchRequest>();
        for (List filesToDelete : Lists.partition((List)Lists.newArrayList(filenames), (int)100)) {
            BatchRequest batch = this.createBatchRequest();
            for (String file : filesToDelete) {
                this.enqueueDelete(GcsPath.fromUri(file), batch);
            }
            batches.add(batch);
        }
        return batches;
    }

    public void remove(Collection<String> filenames) throws IOException {
        GcsUtil.executeBatches(this.makeRemoveBatches(filenames));
    }

    private StorageObjectOrIOException[] enqueueGetFileSize(final GcsPath path, BatchRequest batch) throws IOException {
        final StorageObjectOrIOException[] ret = new StorageObjectOrIOException[1];
        Storage.Objects.Get getRequest = this.storageClient.objects().get(path.getBucket(), path.getObject());
        getRequest.queue(batch, (JsonBatchCallback)new JsonBatchCallback<StorageObject>(){

            public void onSuccess(StorageObject response, HttpHeaders httpHeaders) throws IOException {
                ret[0] = StorageObjectOrIOException.create(response);
            }

            public void onFailure(GoogleJsonError e, HttpHeaders httpHeaders) throws IOException {
                IOException ioException = e.getCode() == 404 ? new FileNotFoundException(path.toString()) : new IOException(String.format("Error trying to get %s: %s", path, e));
                ret[0] = StorageObjectOrIOException.create(ioException);
            }
        });
        return ret;
    }

    private void enqueueDelete(final GcsPath file, BatchRequest batch) throws IOException {
        Storage.Objects.Delete deleteRequest = this.storageClient.objects().delete(file.getBucket(), file.getObject());
        deleteRequest.queue(batch, (JsonBatchCallback)new JsonBatchCallback<Void>(){

            public void onSuccess(Void obj, HttpHeaders responseHeaders) {
                LOG.debug("Successfully deleted {}", (Object)file);
            }

            public void onFailure(GoogleJsonError e, HttpHeaders responseHeaders) throws IOException {
                if (e.getCode() != 404) {
                    throw new IOException(String.format("Error trying to delete %s: %s", file, e));
                }
                LOG.info("Ignoring failed deletion of file {} which already does not exist: {}", (Object)file, (Object)e);
            }
        });
    }

    private BatchRequest createBatchRequest() {
        return this.storageClient.batch(this.httpRequestInitializer);
    }

    @SuppressFBWarnings(value={"NM_CLASS_NOT_EXCEPTION"})
    @AutoValue
    public static abstract class StorageObjectOrIOException {
        public abstract @Nullable StorageObject storageObject();

        public abstract @Nullable IOException ioException();

        @VisibleForTesting
        public static StorageObjectOrIOException create(StorageObject storageObject) {
            return new AutoValue_GcsUtil_StorageObjectOrIOException((StorageObject)Preconditions.checkNotNull((Object)storageObject, (Object)"storageObject"), null);
        }

        @VisibleForTesting
        public static StorageObjectOrIOException create(IOException ioException) {
            return new AutoValue_GcsUtil_StorageObjectOrIOException(null, (IOException)Preconditions.checkNotNull((Object)ioException, (Object)"ioException"));
        }
    }

    class RewriteOp
    extends JsonBatchCallback<RewriteResponse> {
        private GcsPath from;
        private GcsPath to;
        private boolean readyToEnqueue;
        @VisibleForTesting
        Storage.Objects.Rewrite rewriteRequest;

        public boolean getReadyToEnqueue() {
            return this.readyToEnqueue;
        }

        public void enqueue(BatchRequest batch) throws IOException {
            if (!this.readyToEnqueue) {
                throw new IOException(String.format("Invalid state for Rewrite, from=%s, to=%s, readyToEnqueue=%s", this.from, this.to, this.readyToEnqueue));
            }
            this.rewriteRequest.queue(batch, (JsonBatchCallback)this);
            this.readyToEnqueue = false;
        }

        public RewriteOp(GcsPath from, GcsPath to) throws IOException {
            this.from = from;
            this.to = to;
            this.rewriteRequest = GcsUtil.this.storageClient.objects().rewrite(from.getBucket(), from.getObject(), to.getBucket(), to.getObject(), null);
            if (GcsUtil.this.maxBytesRewrittenPerCall != null) {
                this.rewriteRequest.setMaxBytesRewrittenPerCall(GcsUtil.this.maxBytesRewrittenPerCall);
            }
            this.readyToEnqueue = true;
        }

        public void onSuccess(RewriteResponse rewriteResponse, HttpHeaders responseHeaders) throws IOException {
            if (rewriteResponse.getDone().booleanValue()) {
                LOG.debug("Rewrite done: {} to {}", (Object)this.from, (Object)this.to);
                this.readyToEnqueue = false;
            } else {
                LOG.debug("Rewrite progress: {} of {} bytes, {} to {}", new Object[]{rewriteResponse.getTotalBytesRewritten(), rewriteResponse.getObjectSize(), this.from, this.to});
                this.rewriteRequest.setRewriteToken(rewriteResponse.getRewriteToken());
                this.readyToEnqueue = true;
                if (GcsUtil.this.numRewriteTokensUsed != null) {
                    GcsUtil.this.numRewriteTokensUsed.incrementAndGet();
                }
            }
        }

        public void onFailure(GoogleJsonError e, HttpHeaders responseHeaders) throws IOException {
            this.readyToEnqueue = false;
            throw new IOException(String.format("Error trying to rewrite %s to %s: %s", this.from, this.to, e));
        }
    }

    public static class GcsUtilFactory
    implements DefaultValueFactory<GcsUtil> {
        public GcsUtil create(PipelineOptions options) {
            LOG.debug("Creating new GcsUtil");
            GcsOptions gcsOptions = (GcsOptions)options.as(GcsOptions.class);
            Storage.Builder storageBuilder = Transport.newStorageClient(gcsOptions);
            return new GcsUtil(storageBuilder.build(), storageBuilder.getHttpRequestInitializer(), gcsOptions.getExecutorService(), ExperimentalOptions.hasExperiment((PipelineOptions)options, (String)"use_grpc_for_gcs"), gcsOptions.getGcpCredential(), gcsOptions.getGcsUploadBufferSizeBytes());
        }

        public static GcsUtil create(PipelineOptions options, Storage storageClient, HttpRequestInitializer httpRequestInitializer, ExecutorService executorService, Credentials credentials, @Nullable Integer uploadBufferSizeBytes) {
            return new GcsUtil(storageClient, httpRequestInitializer, executorService, ExperimentalOptions.hasExperiment((PipelineOptions)options, (String)"use_grpc_for_gcs"), credentials, uploadBufferSizeBytes);
        }
    }
}

