/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hop.pipeline.transforms.ssh;

import com.trilead.ssh2.Connection;
import com.trilead.ssh2.HTTPProxyData;
import com.trilead.ssh2.ServerHostKeyVerifier;
import org.apache.commons.vfs2.FileContent;
import org.apache.commons.vfs2.FileObject;
import org.apache.hop.core.exception.HopException;
import org.apache.hop.core.variables.IVariables;
import org.apache.hop.core.vfs.HopVfs;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Matchers;
import org.mockito.Mock;
import org.powermock.api.mockito.PowerMockito;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.junit4.PowerMockRunner;

import java.io.ByteArrayInputStream;

import static org.junit.Assert.assertNotNull;
import static org.mockito.Matchers.*;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@RunWith(PowerMockRunner.class)
@PrepareForTest({SSHData.class, HopVfs.class})
public class SSHDataTest {

  @Mock Connection connection;
  @Mock FileObject fileObject;
  @Mock FileContent fileContent;
  @Mock IVariables variables;

  String server = "testServerUrl";
  String keyFilePath = "keyFilePath";
  String passPhrase = "passPhrase";
  String username = "username";
  String password = "password";
  String proxyUsername = "proxyUsername";
  String proxyPassword = "proxyPassword";
  String proxyHost = "proxyHost";
  int port = 22;
  int proxyPort = 23;

  @Before
  public void setup() throws Exception {
    PowerMockito.mockStatic(SSHData.class);
    PowerMockito.mockStatic(HopVfs.class);
    when(SSHData.createConnection(server, port)).thenReturn(connection);
    when(SSHData.OpenConnection(
            any(),
            anyInt(),
            any(),
            any(),
            anyBoolean(),
            any(),
            any(),
            anyInt(),
            anyObject(),
            any(),
            anyInt(),
            any(),
            any()))
        .thenCallRealMethod();
    when(HopVfs.getFileObject(keyFilePath)).thenReturn(fileObject);
  }

  @Test
  public void testOpenConnection_1() throws Exception {
    when(connection.authenticateWithPassword(username, password)).thenReturn(true);
    assertNotNull(
        SSHData.OpenConnection(
            server, port, username, password, false, null, null, 0, null, null, 0, null, null));
    verify(connection).connect();
    verify(connection).authenticateWithPassword(username, password);
  }

  @Test(expected = HopException.class)
  public void testOpenConnection_2() throws Exception {
    when(connection.authenticateWithPassword(username, password)).thenReturn(false);
    SSHData.OpenConnection(
        server, port, username, password, false, null, null, 0, null, null, 0, null, null);
    verify(connection).connect();
    verify(connection).authenticateWithPassword(username, password);
  }

  @Test(expected = HopException.class)
  public void testOpenConnectionUseKey_1() throws Exception {
    when(fileObject.exists()).thenReturn(false);
    SSHData.OpenConnection(
        server, port, null, null, true, null, null, 0, null, null, 0, null, null);
    verify(fileObject).exists();
  }

  @Test
  public void testOpenConnectionUseKey_2() throws Exception {
    when(fileObject.exists()).thenReturn(true);
    when(fileObject.getContent()).thenReturn(fileContent);
    when(fileContent.getSize()).thenReturn(1000L);
    when(fileContent.getInputStream())
        .thenReturn(new ByteArrayInputStream(new byte[] {1, 2, 3, 4, 5}));
    when(variables.resolve(passPhrase)).thenReturn(passPhrase);
    when(connection.authenticateWithPublicKey(eq(username), Matchers.<char[]>any(), eq(passPhrase)))
        .thenReturn(true);
    SSHData.OpenConnection(
        server,
        port,
        username,
        null,
        true,
        keyFilePath,
        passPhrase,
        0,
        variables,
        null,
        0,
        null,
        null);
    verify(connection).connect();
    verify(connection)
        .authenticateWithPublicKey(eq(username), Matchers.<char[]>any(), eq(passPhrase));
  }

  @Test
  public void testOpenConnectionProxy() throws Exception {
    when(connection.authenticateWithPassword(username, password)).thenReturn(true);
    assertNotNull(
        SSHData.OpenConnection(
            server,
            port,
            username,
            password,
            false,
            null,
            null,
            0,
            null,
            proxyHost,
            proxyPort,
            proxyUsername,
            proxyPassword));
    verify(connection).connect();
    verify(connection).authenticateWithPassword(username, password);
    verify(connection).setProxyData(any(HTTPProxyData.class));
  }

  @Test
  public void testOpenConnectionTimeOut() throws Exception {
    when(connection.authenticateWithPassword(username, password)).thenReturn(true);
    assertNotNull(
        SSHData.OpenConnection(
            server,
            port,
            username,
            password,
            false,
            null,
            null,
            100,
            null,
            null,
            proxyPort,
            proxyUsername,
            proxyPassword));
    verify(connection).connect(isNull(ServerHostKeyVerifier.class), eq(0), eq(100 * 1000));
  }
}
