/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.pipeline.transforms.databaselookup.readallcache;

import org.apache.hop.core.row.IValueMeta;

import java.util.BitSet;

/** @author Andrey Khayrutdinov */
class LtIndex extends Index {

  static Index greaterOrEqualCache(int column, IValueMeta valueMeta, int rowsAmount) {
    return new LtIndex(column, valueMeta, rowsAmount, true);
  }

  private final boolean isMatchingGreaterOrEqual;

  LtIndex(int column, IValueMeta valueMeta, int rowsAmount) {
    this(column, valueMeta, rowsAmount, false);
  }

  private LtIndex(
      int column, IValueMeta valueMeta, int rowsAmount, boolean isMatchingGreaterOrEqual) {
    super(column, valueMeta, rowsAmount);
    this.isMatchingGreaterOrEqual = isMatchingGreaterOrEqual;
  }

  @Override
  void doApply(SearchingContext context, IValueMeta lookupMeta, Object lookupValue) {
    int firstValue = findInsertionPointOf(new IndexedValue(lookupValue, -1));
    if (firstValue == 0) {
      // everything is greater than lookupValue
      if (isMatchingGreaterOrEqual) {
        // then do nothing
        return;
      }
      context.setEmpty();
    } else {
      BitSet bitSet = context.getWorkingSet();

      int start, end;
      if (firstValue < values.length / 2) {
        start = 0;
        end = firstValue;
      } else {
        start = firstValue;
        end = values.length;
      }

      for (int i = start; i < end; i++) {
        bitSet.set(values[i].row, true);
      }

      context.intersect(bitSet, (start != 0) ^ isMatchingGreaterOrEqual);
    }
  }

  @Override
  int getRestrictionPower() {
    return -1000;
  }
}
