/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package hivemall.tools.array;

import hivemall.TestUtils;
import hivemall.utils.hadoop.WritableUtils;

import java.io.IOException;
import java.util.List;

import org.apache.hadoop.hive.ql.metadata.HiveException;
import org.apache.hadoop.hive.ql.udf.generic.GenericUDF;
import org.apache.hadoop.hive.serde2.io.DoubleWritable;
import org.apache.hadoop.hive.serde2.objectinspector.ObjectInspector;
import org.apache.hadoop.hive.serde2.objectinspector.ObjectInspectorFactory;
import org.apache.hadoop.hive.serde2.objectinspector.ObjectInspectorUtils;
import org.apache.hadoop.hive.serde2.objectinspector.primitive.PrimitiveObjectInspectorFactory;
import org.junit.Assert;
import org.junit.Test;

public class SelectKBestUDFTest {

    @Test
    public void test() throws Exception {
        final SelectKBestUDF selectKBest = new SelectKBestUDF();
        final int k = 2;
        final double[] data =
                new double[] {250.29999999999998, 170.90000000000003, 73.2, 12.199999999999996};
        final double[] importanceList = new double[] {292.1666753739119, 152.70000455081467,
                187.93333893418327, 59.93333511948589};

        final GenericUDF.DeferredObject[] dObjs = new GenericUDF.DeferredObject[] {
                new GenericUDF.DeferredJavaObject(WritableUtils.toWritableList(data)),
                new GenericUDF.DeferredJavaObject(WritableUtils.toWritableList(importanceList)),
                new GenericUDF.DeferredJavaObject(k)};

        selectKBest.initialize(new ObjectInspector[] {
                ObjectInspectorFactory.getStandardListObjectInspector(
                    PrimitiveObjectInspectorFactory.writableDoubleObjectInspector),
                ObjectInspectorFactory.getStandardListObjectInspector(
                    PrimitiveObjectInspectorFactory.writableDoubleObjectInspector),
                ObjectInspectorUtils.getConstantObjectInspector(
                    PrimitiveObjectInspectorFactory.javaIntObjectInspector, k)});
        final List<DoubleWritable> resultObj = selectKBest.evaluate(dObjs);

        Assert.assertEquals(resultObj.size(), k);

        final double[] result = new double[k];
        for (int i = 0; i < k; i++) {
            result[i] = resultObj.get(i).get();
        }

        final double[] answer = new double[] {250.29999999999998, 73.2};

        Assert.assertArrayEquals(answer, result, 0.d);
        selectKBest.close();
    }

    @Test
    public void testSerialization() throws HiveException, IOException {
        final int k = 2;
        final double[] data =
                new double[] {250.29999999999998, 170.90000000000003, 73.2, 12.199999999999996};
        final double[] importanceList = new double[] {292.1666753739119, 152.70000455081467,
                187.93333893418327, 59.93333511948589};

        TestUtils.testGenericUDFSerialization(SelectKBestUDF.class,
            new ObjectInspector[] {
                    ObjectInspectorFactory.getStandardListObjectInspector(
                        PrimitiveObjectInspectorFactory.writableDoubleObjectInspector),
                    ObjectInspectorFactory.getStandardListObjectInspector(
                        PrimitiveObjectInspectorFactory.writableDoubleObjectInspector),
                    ObjectInspectorUtils.getConstantObjectInspector(
                        PrimitiveObjectInspectorFactory.javaIntObjectInspector, k)},
            new Object[] {WritableUtils.toWritableList(data),
                    WritableUtils.toWritableList(importanceList), k});
    }

}
