/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hertzbeat.collector.collect.common.cache;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.googlecode.concurrentlinkedhashmap.ConcurrentLinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.hertzbeat.collector.collect.common.cache.AbstractConnection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionCommonCache<T, C extends AbstractConnection<?>> {
    private static final Logger log = LoggerFactory.getLogger(ConnectionCommonCache.class);
    private static final long DEFAULT_CACHE_TIMEOUT = 600000L;
    private static final int CACHE_TIME_LENGTH = 2;
    private Map<T, Long[]> timeoutMap;
    private ConcurrentLinkedHashMap<T, C> cacheMap;

    public ConnectionCommonCache() {
        this.initCache();
    }

    private void initCache() {
        this.cacheMap = new ConcurrentLinkedHashMap.Builder().maximumWeightedCapacity(Integer.MAX_VALUE).listener((key, value) -> {
            this.timeoutMap.remove(key);
            try {
                value.close();
            }
            catch (Exception e) {
                log.error("connection close error: {}.", (Object)e.getMessage(), (Object)e);
            }
            log.info("connection common cache discard key: {}, value: {}.", key, value);
        }).build();
        this.timeoutMap = new ConcurrentHashMap<T, Long[]>(16);
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("connection-cache-timout-detector-%d").setDaemon(true).build();
        ScheduledThreadPoolExecutor scheduledExecutor = new ScheduledThreadPoolExecutor(1, threadFactory);
        scheduledExecutor.scheduleWithFixedDelay(this::cleanTimeoutOrUnHealthCache, 2L, 100L, TimeUnit.SECONDS);
    }

    private void cleanTimeoutOrUnHealthCache() {
        try {
            this.cacheMap.forEach((key, value) -> {
                Long[] cacheTime = this.timeoutMap.get(key);
                long currentTime = System.currentTimeMillis();
                if (cacheTime == null || cacheTime.length != 2 || cacheTime[0] + cacheTime[1] < currentTime) {
                    log.warn("[connection common cache] clean the timeout cache, key {}", key);
                    this.timeoutMap.remove(key);
                    this.cacheMap.remove(key);
                    try {
                        value.close();
                    }
                    catch (Exception e) {
                        log.error("clean connection close error: {}.", (Object)e.getMessage(), (Object)e);
                    }
                }
            });
        }
        catch (Exception e) {
            log.error("[connection common cache] clean timeout cache error: {}.", (Object)e.getMessage(), (Object)e);
        }
    }

    public void addCache(T key, C value, Long timeDiff) {
        this.removeCache(key);
        if (timeDiff == null) {
            timeDiff = 600000L;
        }
        this.cacheMap.put(key, value);
        this.timeoutMap.put(key, new Long[]{System.currentTimeMillis(), timeDiff});
    }

    public void addCache(T key, C value) {
        this.addCache(key, value, 600000L);
    }

    public Optional<C> getCache(T key, boolean refreshCache) {
        Long[] cacheTime = this.timeoutMap.get(key);
        if (cacheTime == null || cacheTime.length != 2) {
            log.info("[connection common cache] not hit the cache, key {}.", key);
            return Optional.empty();
        }
        if (cacheTime[0] + cacheTime[1] < System.currentTimeMillis()) {
            log.warn("[connection common cache] is timeout, remove it, key {}.", key);
            this.timeoutMap.remove(key);
            this.cacheMap.remove(key);
            return Optional.empty();
        }
        AbstractConnection value = (AbstractConnection)this.cacheMap.compute(key, (k, v) -> {
            if (v == null) {
                log.error("[connection common cache] value is null, remove it, key {}.", key);
                this.timeoutMap.remove(key);
                return null;
            }
            if (refreshCache) {
                cacheTime[0] = System.currentTimeMillis();
                this.timeoutMap.put(key, cacheTime);
            }
            return v;
        });
        return Optional.ofNullable(value);
    }

    public void removeCache(T key) {
        this.timeoutMap.remove(key);
        AbstractConnection value = (AbstractConnection)this.cacheMap.remove(key);
        try {
            if (value == null) {
                return;
            }
            value.close();
        }
        catch (Exception e) {
            log.error("connection close error: {}.", (Object)e.getMessage(), (Object)e);
        }
    }
}

