/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hertzbeat.collector.collect.ssh;

import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.hertzbeat.collector.collect.AbstractCollect;
import org.apache.hertzbeat.collector.collect.common.cache.CacheIdentifier;
import org.apache.hertzbeat.collector.collect.common.cache.ConnectionCommonCache;
import org.apache.hertzbeat.collector.collect.common.cache.SshConnect;
import org.apache.hertzbeat.collector.collect.common.ssh.CommonSshClient;
import org.apache.hertzbeat.collector.util.CollectUtil;
import org.apache.hertzbeat.collector.util.PrivateKeyUtils;
import org.apache.hertzbeat.common.entity.job.Metrics;
import org.apache.hertzbeat.common.entity.job.protocol.SshProtocol;
import org.apache.hertzbeat.common.entity.message.CollectRep;
import org.apache.hertzbeat.common.util.CommonUtil;
import org.apache.sshd.client.SshClient;
import org.apache.sshd.client.channel.ClientChannelEvent;
import org.apache.sshd.client.future.AuthFuture;
import org.apache.sshd.client.future.ConnectFuture;
import org.apache.sshd.client.session.ClientSession;
import org.apache.sshd.common.Closeable;
import org.apache.sshd.common.SshException;
import org.apache.sshd.common.channel.exception.SshChannelOpenException;
import org.apache.sshd.common.util.io.output.NoCloseOutputStream;
import org.apache.sshd.common.util.security.SecurityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

public class SshCollectImpl
extends AbstractCollect {
    private static final Logger log = LoggerFactory.getLogger(SshCollectImpl.class);
    private static final String PARSE_TYPE_ONE_ROW = "oneRow";
    private static final String PARSE_TYPE_MULTI_ROW = "multiRow";
    private static final String PARSE_TYPE_NETCAT = "netcat";
    private static final String PARSE_TYPE_LOG = "log";
    private static final int DEFAULT_TIMEOUT = 10000;
    private final ConnectionCommonCache<CacheIdentifier, SshConnect> connectionCommonCache = new ConnectionCommonCache();

    @Override
    public void preCheck(Metrics metrics) throws IllegalArgumentException {
        if (metrics == null || metrics.getSsh() == null) {
            throw new IllegalArgumentException("ssh collect must has ssh params");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void collect(CollectRep.MetricsData.Builder builder, long monitorId, String app, Metrics metrics) {
        String errorMsg;
        long startTime = System.currentTimeMillis();
        SshProtocol sshProtocol = metrics.getSsh();
        boolean reuseConnection = Boolean.parseBoolean(sshProtocol.getReuseConnection());
        int timeout = CollectUtil.getTimeout(sshProtocol.getTimeout(), 10000);
        Closeable channel = null;
        ClientSession clientSession = null;
        try {
            clientSession = this.getConnectSession(sshProtocol, timeout, reuseConnection);
            channel = clientSession.createExecChannel(sshProtocol.getScript());
            ByteArrayOutputStream response = new ByteArrayOutputStream();
            channel.setOut(response);
            channel.setErr(new NoCloseOutputStream(System.err));
            channel.open().verify(timeout);
            ArrayList<ClientChannelEvent> list = new ArrayList<ClientChannelEvent>();
            list.add(ClientChannelEvent.CLOSED);
            Set<ClientChannelEvent> waitEvents = channel.waitFor(list, timeout);
            if (waitEvents.contains((Object)ClientChannelEvent.TIMEOUT)) {
                throw new SocketTimeoutException("Failed to retrieve command result in time: " + sshProtocol.getScript());
            }
            Long responseTime = System.currentTimeMillis() - startTime;
            String result = response.toString();
            if (!StringUtils.hasText(result)) {
                builder.setCode(CollectRep.Code.FAIL);
                builder.setMsg("ssh shell response data is null");
                return;
            }
            switch (sshProtocol.getParseType()) {
                case "log": {
                    this.parseResponseDataByLog(result, metrics.getAliasFields(), builder, responseTime);
                    return;
                }
                case "netcat": {
                    this.parseResponseDataByNetcat(result, metrics.getAliasFields(), builder, responseTime);
                    return;
                }
                case "oneRow": {
                    this.parseResponseDataByOne(result, metrics.getAliasFields(), builder, responseTime);
                    return;
                }
                case "multiRow": {
                    this.parseResponseDataByMulti(result, metrics.getAliasFields(), builder, responseTime);
                    return;
                }
                default: {
                    builder.setCode(CollectRep.Code.FAIL);
                    builder.setMsg("Ssh collect not support this parse type: " + sshProtocol.getParseType());
                    return;
                }
            }
        }
        catch (ConnectException connectException) {
            errorMsg = CommonUtil.getMessageFromThrowable(connectException);
            log.info(errorMsg);
            builder.setCode(CollectRep.Code.UN_CONNECTABLE);
            builder.setMsg("The peer refused to connect: service port does not listening or firewall: " + errorMsg);
            return;
        }
        catch (SshException sshException) {
            Throwable throwable = sshException.getCause();
            if (throwable instanceof SshChannelOpenException) {
                log.warn("Remote ssh server no more session channel, please increase sshd_config MaxSessions.");
            }
            String errorMsg2 = CommonUtil.getMessageFromThrowable(sshException);
            builder.setCode(CollectRep.Code.UN_CONNECTABLE);
            builder.setMsg("Peer ssh connection failed: " + errorMsg2);
            return;
        }
        catch (IOException ioException) {
            errorMsg = CommonUtil.getMessageFromThrowable(ioException);
            log.info(errorMsg);
            builder.setCode(CollectRep.Code.UN_CONNECTABLE);
            builder.setMsg("Peer io connection failed: " + errorMsg);
            return;
        }
        catch (Exception exception) {
            errorMsg = CommonUtil.getMessageFromThrowable(exception);
            log.warn(errorMsg, exception);
            builder.setCode(CollectRep.Code.FAIL);
            builder.setMsg(errorMsg);
            return;
        }
        finally {
            if (channel != null && channel.isOpen()) {
                try {
                    channel.close();
                }
                catch (Exception e) {
                    log.error(e.getMessage(), e);
                }
            }
            if (clientSession != null && !reuseConnection) {
                try {
                    clientSession.close();
                }
                catch (Exception e) {
                    log.error(e.getMessage(), e);
                }
            }
        }
    }

    @Override
    public String supportProtocol() {
        return "ssh";
    }

    private void parseResponseDataByLog(String result, List<String> aliasFields, CollectRep.MetricsData.Builder builder, Long responseTime) {
        String[] lines = result.split("\n");
        if (lines.length + 1 < aliasFields.size()) {
            log.error("ssh response data not enough: {}", (Object)result);
            return;
        }
        for (String line : lines) {
            CollectRep.ValueRow.Builder valueRowBuilder = CollectRep.ValueRow.newBuilder();
            for (String alias : aliasFields) {
                if ("responseTime".equalsIgnoreCase(alias)) {
                    valueRowBuilder.addColumns(responseTime.toString());
                    continue;
                }
                valueRowBuilder.addColumns(line);
            }
            builder.addValues(valueRowBuilder.build());
        }
    }

    private void parseResponseDataByNetcat(String result, List<String> aliasFields, CollectRep.MetricsData.Builder builder, Long responseTime) {
        String[] lines = result.split("\n");
        if (lines.length + 1 < aliasFields.size()) {
            log.error("ssh response data not enough: {}", (Object)result);
            return;
        }
        boolean contains = lines[0].contains("=");
        Map<String, String> mapValue = Arrays.stream(lines).map(item -> {
            if (contains) {
                return item.split("=");
            }
            return item.split("\t");
        }).filter(item -> ((String[])item).length == 2).collect(Collectors.toMap(x -> x[0], x -> x[1]));
        CollectRep.ValueRow.Builder valueRowBuilder = CollectRep.ValueRow.newBuilder();
        for (String field : aliasFields) {
            String fieldValue = mapValue.get(field);
            if (fieldValue == null) {
                valueRowBuilder.addColumns("&nbsp;");
                continue;
            }
            valueRowBuilder.addColumns(fieldValue);
        }
        builder.addValues(valueRowBuilder.build());
    }

    private void parseResponseDataByOne(String result, List<String> aliasFields, CollectRep.MetricsData.Builder builder, Long responseTime) {
        String[] lines = result.split("\n");
        if (lines.length + 1 < aliasFields.size()) {
            log.error("ssh response data not enough: {}", (Object)result);
            return;
        }
        CollectRep.ValueRow.Builder valueRowBuilder = CollectRep.ValueRow.newBuilder();
        int lineIndex = 0;
        for (int aliasIndex = 0; aliasIndex < aliasFields.size(); ++aliasIndex) {
            if ("responseTime".equalsIgnoreCase(aliasFields.get(aliasIndex))) {
                valueRowBuilder.addColumns(responseTime.toString());
                continue;
            }
            if (lineIndex < lines.length) {
                valueRowBuilder.addColumns(lines[lineIndex].trim());
            } else {
                valueRowBuilder.addColumns("&nbsp;");
            }
            ++lineIndex;
        }
        builder.addValues(valueRowBuilder.build());
    }

    private void parseResponseDataByMulti(String result, List<String> aliasFields, CollectRep.MetricsData.Builder builder, Long responseTime) {
        int i;
        String[] lines = result.split("\n");
        if (lines.length <= 1) {
            log.error("ssh response data only has header: {}", (Object)result);
            return;
        }
        String[] fields = lines[0].split(" ");
        HashMap<String, Integer> fieldMapping = new HashMap<String, Integer>(fields.length);
        for (i = 0; i < fields.length; ++i) {
            fieldMapping.put(fields[i].trim().toLowerCase(), i);
        }
        for (i = 1; i < lines.length; ++i) {
            String[] values = lines[i].split(" ");
            CollectRep.ValueRow.Builder valueRowBuilder = CollectRep.ValueRow.newBuilder();
            for (String alias : aliasFields) {
                if ("responseTime".equalsIgnoreCase(alias)) {
                    valueRowBuilder.addColumns(responseTime.toString());
                    continue;
                }
                Integer index = (Integer)fieldMapping.get(alias.toLowerCase());
                if (index != null && index < values.length) {
                    valueRowBuilder.addColumns(values[index]);
                    continue;
                }
                valueRowBuilder.addColumns("&nbsp;");
            }
            builder.addValues(valueRowBuilder.build());
        }
    }

    private void removeConnectSessionCache(SshProtocol sshProtocol) {
        CacheIdentifier identifier = CacheIdentifier.builder().ip(sshProtocol.getHost()).port(sshProtocol.getPort()).username(sshProtocol.getUsername()).password(sshProtocol.getPassword()).build();
        this.connectionCommonCache.removeCache(identifier);
    }

    private ClientSession getConnectSession(SshProtocol sshProtocol, int timeout, boolean reuseConnection) throws IOException, GeneralSecurityException {
        CacheIdentifier identifier = CacheIdentifier.builder().ip(sshProtocol.getHost()).port(sshProtocol.getPort()).username(sshProtocol.getUsername()).password(sshProtocol.getPassword()).build();
        ClientSession clientSession = null;
        if (reuseConnection) {
            Optional<SshConnect> cacheOption = this.connectionCommonCache.getCache(identifier, true);
            if (cacheOption.isPresent()) {
                clientSession = cacheOption.get().getConnection();
                try {
                    if (clientSession == null || clientSession.isClosed() || clientSession.isClosing()) {
                        clientSession = null;
                        this.connectionCommonCache.removeCache(identifier);
                    }
                }
                catch (Exception e) {
                    log.warn(e.getMessage());
                    clientSession = null;
                    this.connectionCommonCache.removeCache(identifier);
                }
            }
            if (clientSession != null) {
                return clientSession;
            }
        }
        SshClient sshClient = CommonSshClient.getSshClient();
        clientSession = (ClientSession)((ConnectFuture)sshClient.connect(sshProtocol.getUsername(), sshProtocol.getHost(), Integer.parseInt(sshProtocol.getPort())).verify(timeout, TimeUnit.MILLISECONDS)).getSession();
        if (StringUtils.hasText(sshProtocol.getPassword())) {
            clientSession.addPasswordIdentity(sshProtocol.getPassword());
        } else if (StringUtils.hasText(sshProtocol.getPrivateKey())) {
            String resourceKey = PrivateKeyUtils.writePrivateKey(sshProtocol.getHost(), sshProtocol.getPrivateKey());
            SecurityUtils.loadKeyPairIdentities(null, () -> resourceKey, new FileInputStream(resourceKey), null).forEach(clientSession::addPublicKeyIdentity);
        }
        if (!((AuthFuture)clientSession.auth().verify(timeout, TimeUnit.MILLISECONDS)).isSuccess()) {
            clientSession.close();
            throw new IllegalArgumentException("ssh auth failed.");
        }
        if (reuseConnection) {
            SshConnect sshConnect = new SshConnect(clientSession);
            this.connectionCommonCache.addCache(identifier, sshConnect);
        }
        return clientSession;
    }
}

