# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
''' query_operator_unittest.py '''
# pylint: disable=too-many-lines, missing-docstring, too-many-public-methods, undefined-variable
# over 500 bad indentation errors so disable
# pylint: disable=bad-continuation
# pylint: disable=unused-argument, unused-variable
from unittest.mock import patch, Mock
from heron.tools.tracker.src.python.metricstimeline import MetricsTimeline

from heron.tools.tracker.src.python.query_operators import *

import pytest


@pytest.mark.asyncio
async def test_TS_execute():
  ts = TS(["a", "b", "c"])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Return mocked timeline
  def getMetricTimelineSideEffect(*args):
    assert (tmanager, "a", ["c"], ["b"], 40, 360) == args
    return MetricsTimeline(
      starttime = 40,
      endtime = 360,
      component = "a",
      timeline = {
            "c": {
                "b": {
                    40: "1.0",
                    100: "1.0",
                    160: "1.0",
                    220: "1.0",
                    280: "1.0",
                    340: "1.0"
                }
            }
        },
    )

  with patch("heron.tools.tracker.src.python.query_operators.get_metrics_timeline",
             side_effect=getMetricTimelineSideEffect):
    metrics = await ts.execute(tracker, tmanager, start, end)
    assert 1 == len(metrics)
    assert "b" == metrics[0].instance
    assert "c" == metrics[0].metric_name
    assert "a" == metrics[0].component_name
    assert {
        120: 1.0,
        180: 1.0,
        240: 1.0,
        300: 1.0
    } == metrics[0].timeline

@pytest.mark.asyncio
async def test_TS_execute_when_no_timeline():
  ts = TS(["a", "b", "c"])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # If no timeline is returned
  def getMetricTimelineSideEffect(*args):
    assert (tmanager, "a", ["c"], ["b"], 40, 360) == args
    return ({
        "message": "some_exception"
    })

  # pylint: disable=unused-variable
  with pytest.raises(Exception):
    with patch("heron.tools.tracker.src.python.query_operators.get_metrics_timeline",
               side_effect=getMetricTimelineSideEffect):
      metrics = await ts.execute(tracker, tmanager, start, end)

@pytest.mark.asyncio
async def test_TS_execute_with_multiple_instances():
  ts = TS(["a", "b", "c"])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # With multiple instances
  def getMetricTimelineSideEffect(*args):
    assert (tmanager, "a", ["c"], [], 40, 360) == args
    return MetricsTimeline(
      starttime = 40,
      endtime = 360,
      component = "a",
      timeline = {
            "c": {
                "b": {
                    40: "1.0",
                    100: "1.0",
                      # 160: "1.0", # This value is missing
                    220: "1.0",
                    280: "1.0",
                    340: "1.0"
                },
                "d": {
                    40: "2.0",
                    100: "2.0",
                    160: "2.0",
                    220: "2.0",
                    280: "2.0",
                    340: "2.0"
                }
            }
        })

  # pylint: disable=unused-variable
  with patch("heron.tools.tracker.src.python.query_operators.get_metrics_timeline",
             side_effect=getMetricTimelineSideEffect):
    ts = TS(["a", "*", "c"])
    metrics = await ts.execute(tracker, tmanager, start, end)
    assert 2 == len(metrics)
    metric1 = metrics[0]
    metric2 = metrics[1]
    for metric in metrics:
      if metric.instance == "b":
        assert "c" == metric.metric_name
        assert "a" == metric.component_name
        assert {
              # 120: 1.0, # Missing value is not reported
            180: 1.0,
            240: 1.0,
            300: 1.0
        } == metric.timeline
      elif metric.instance == "d":
        assert "c" == metric.metric_name
        assert "a" == metric.component_name
        assert {
            120: 2.0,
            180: 2.0,
            240: 2.0,
            300: 2.0
        } == metric.timeline
      else:
        pytest.fail("Wrong metrics generated by TS.execute")

@pytest.mark.asyncio
async def test_DEFAULT_execute():
  ts = Mock(TS)
  default = Default([float(0), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Return mocked timeline
  def ts_side_effect(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
          120: 1.0,
          180: 1.0,
          240: 1.0,
          300: 1.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect

  metrics = await default.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert "instance" == metrics[0].instance
  assert "metric_name" == metrics[0].metric_name
  assert "component" == metrics[0].component_name
  assert {
    120: 1.0,
    180: 1.0,
    240: 1.0,
    300: 1.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_DEFAULT_execute_when_exception():
  ts = Mock(TS)
  default = Default([float(0), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # In case of exception
  def ts_side_effect2(*args):
    raise Exception("some_exception")
  ts.execute.side_effect = ts_side_effect2

  with pytest.raises(Exception):
    metrics = await default.execute(tracker, tmanager, start, end)

@pytest.mark.asyncio
async def test_DEFAULT_execute_when_missing_value():
  ts = Mock(TS)
  default = Default([float(0), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # When missing a value
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        180: 1.0,
        240: 1.0,
        300: 1.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  metrics = await default.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert "instance" == metrics[0].instance
  assert "metric_name" == metrics[0].metric_name
  assert "component" == metrics[0].component_name
  assert {
    120: 0, # Missing value filled
      180: 1.0,
    240: 1.0,
    300: 1.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_DEFAULT_execute_with_multiple_ts():
  ts = Mock(TS)
  default = Default([float(0), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Multiple timelines missing some values
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
          # 120: 1.0, # Missing
        180: 1.0,
        240: 1.0,
        300: 1.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 2.0,
          # 180: 2.0, # Missing
        240: 2.0,
          # 300: 2.0, # Missing
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  metrics = await default.execute(tracker, tmanager, start, end)
  assert 2 == len(metrics)
  for metric in metrics:
    if metric.instance == "instance":
      assert "instance" == metric.instance
      assert "metric_name" == metric.metric_name
      assert "component" == metric.component_name
      assert {
        120: 0, # Filled
          180: 1.0,
        240: 1.0,
        300: 1.0
      } == metric.timeline
    elif metric.instance == "instance2":
      assert "instance2" == metric.instance
      assert "metric_name" == metric.metric_name
      assert "component" == metric.component_name
      assert {
        120: 2.0,
        180: 0, # Filled
          240: 2.0,
        300: 0 # Filled
        } == metric.timeline
    else:
      pytest.fail("Wrong metrics generated by TS.execute")

@pytest.mark.asyncio
async def test_SUM_execute():
  ts = Mock(TS)
  operator = Sum([float(10), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Return mocked timeline
  def ts_side_effect(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 1.0,
        180: 1.0,
        240: 1.0,
        300: 1.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert {
    120: 11.0,
    180: 11.0,
    240: 11.0,
    300: 11.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_SUM_execute_when_exception():
  ts = Mock(TS)
  operator = Sum([float(10), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # In case of exception
  def ts_side_effect2(*args):
    raise Exception("some_exception")
  ts.execute.side_effect = ts_side_effect2

  with pytest.raises(Exception):
    metrics = await operator.execute(tracker, tmanager, start, end)

@pytest.mark.asyncio
async def test_SUM_execute_when_missing_value():
  ts = Mock(TS)
  operator = Sum([float(10), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # When missing a value
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        180: 1.0,
        240: 1.0,
        300: 1.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert {
    120: 10, # Missing value filled
      180: 11.0,
    240: 11.0,
    300: 11.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_SUM_execute_with_multiple_ts():
  ts = Mock(TS)
  operator = Sum([float(10), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Multiple timelines missing some values
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
          # 120: 1.0, # Missing
        180: 1.0,
        240: 1.0,
        300: 1.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert {
    120: 12.0,
    180: 13.0,
    240: 13.0,
    300: 13.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_MAX_execute():
  ts = Mock(TS)
  operator = Max([ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Return mocked timeline
  def ts_side_effect(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 1.0,
        180: 1.0,
        240: 1.0,
        300: 1.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert {
    120: 1.0,
    180: 1.0,
    240: 1.0,
    300: 1.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_MAX_execute_when_exception():
  ts = Mock(TS)
  operator = Max([ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # In case of exception
  def ts_side_effect2(*args):
    raise Exception("some_exception")
  ts.execute.side_effect = ts_side_effect2

  with pytest.raises(Exception):
    metrics = await operator.execute(tracker, tmanager, start, end)

@pytest.mark.asyncio
async def test_MAX_execute_when_missing_values():
  ts = Mock(TS)
  operator = Max([ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # When missing a value
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        180: 1.0,
        240: 1.0,
        300: 1.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert {
    180: 1.0,
    240: 1.0,
    300: 1.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_MAX_execute_with_multiple_ts():
  ts = Mock(TS)
  operator = Max([ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Multiple timelines missing some values
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
          # 120: 1.0, # Missing
        180: 1.0,
        240: 3.0,
        300: 3.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 2.0,
        180: 0.0,
        240: 2.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert {
    120: 2.0,
    180: 1.0,
    240: 3.0,
    300: 5.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_PERCENTILE_execute():
  ts = Mock(TS)
  operator = Percentile([float(90), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Return mocked timeline
  def ts_side_effect(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 1.0,
        180: 1.0,
        240: 1.0,
        300: 1.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert {
    120: 1.0,
    180: 1.0,
    240: 1.0,
    300: 1.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_PERCENTILE_execute_when_exception():
  ts = Mock(TS)
  operator = Percentile([float(90), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # In case of exception
  def ts_side_effect2(*args):
    raise Exception("some_exception")
  ts.execute.side_effect = ts_side_effect2

  with pytest.raises(Exception):
    metrics = await operator.execute(tracker, tmanager, start, end)

@pytest.mark.asyncio
async def test_PERCENTILE_execute_when_missing_values():
  ts = Mock(TS)
  operator = Percentile([float(90), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # When missing a value
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        180: 1.0,
        240: 1.0,
        300: 1.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert {
    180: 1.0,
    240: 1.0,
    300: 1.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_PERCENTILE_execute_with_multiple_ts():
  ts = Mock(TS)
  operator = Percentile([float(90), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Multiple timelines missing some values
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 1.0,
        180: 4.0,
        240: 6.0,
        300: 3.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 2.0,
        180: 5.0,
        240: 5.0,
        300: 5.0,
      }),
      Metrics("component", "metric_name", "instance3", start, end, {
        120: 4.0,
        180: 6.0,
        240: 4.0,
        300: 4.0,
      }),
      Metrics("component", "metric_name", "instance4", start, end, {
        120: 3.0,
        180: 7.0,
        240: 3.0,
        300: 6.0,
      }),
      Metrics("component", "metric_name", "instance5", start, end, {
        120: 5.0,
        180: 8.0,
        240: 2.0,
        300: 7.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert {
    120: 4.0,
    180: 7.0,
    240: 5.0,
    300: 6.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_DIVIDE_execute():
  ts = Mock(TS)
  operator = Divide([float(100), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Return mocked timeline
  def ts_side_effect(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 1.0,
        180: 2.0,
        240: 4.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert "instance" == metrics[0].instance
  assert {
    120: 100.0,
    180: 50.0,
    240: 25.0,
    300: 20.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_DIVIDE_execute_when_exception():
  ts = Mock(TS)
  operator = Divide([float(100), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # In case of exception
  def ts_side_effect2(*args):
    raise Exception("some_exception")
  ts.execute.side_effect = ts_side_effect2

  with pytest.raises(Exception):
    metrics = await operator.execute(tracker, tmanager, start, end)

@pytest.mark.asyncio
async def test_DIVIDE_execute_when_missing_values():
  ts = Mock(TS)
  operator = Divide([float(100), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # When missing a value
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        180: 2.0,
        240: 4.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert "instance" == metrics[0].instance
  assert {
    180: 50.0,
    240: 25.0,
    300: 20.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_DIVIDE_execute_with_multiple_ts():
  ts = Mock(TS)
  ts2 = Mock(TS)
  operator = Divide([ts, ts2])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Multiple timelines
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 1.0,
        180: 1.0,
        240: 1.0,
        300: 1.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      }),
      Metrics("component", "metric_name", "instance3", start, end, {
        120: 3.0,
        180: 3.0,
        240: 3.0,
        300: 3.0,
      }),
      Metrics("component", "metric_name", "instance4", start, end, {
        120: 4.0,
        180: 4.0,
        240: 4.0,
        300: 4.0,
      }),
      Metrics("component", "metric_name", "instance5", start, end, {
        120: 5.0,
        180: 5.0,
        240: 5.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  def ts_side_effect4(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 4.0,
        180: 4.0,
        240: 4.0,
        300: 4.0,
      }),
      Metrics("component", "metric_name", "instance3", start, end, {
        120: 6.0,
        180: 6.0,
        240: 6.0,
        300: 6.0,
      }),
      Metrics("component", "metric_name", "instance4", start, end, {
        120: 8.0,
        180: 8.0,
        240: 8.0,
        300: 8.0,
      }),
      Metrics("component", "metric_name", "instance5", start, end, {
        120: 10.0,
        180: 10.0,
        240: 10.0,
        300: 10.0,
      })
    ])
  ts2.execute.side_effect = ts_side_effect4

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 5 == len(metrics)
  for metric in metrics:
      # All should have same value - 0.5
    assert {
      120: 0.5,
      180: 0.5,
      240: 0.5,
      300: 0.5
    } == metric.timeline

@pytest.mark.asyncio
async def test_DIVIDE_execute_with_mulitiple_ts_when_instances_do_not_match():
  ts = Mock(TS)
  ts2 = Mock(TS)
  operator = Divide([ts, ts2])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Multiple timelines
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 1.0,
        180: 1.0,
        240: 1.0,
        300: 1.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      }),
      Metrics("component", "metric_name", "instance3", start, end, {
        120: 3.0,
        180: 3.0,
        240: 3.0,
        300: 3.0,
      }),
      Metrics("component", "metric_name", "instance4", start, end, {
        120: 4.0,
        180: 4.0,
        240: 4.0,
        300: 4.0,
      }),
      Metrics("component", "metric_name", "instance5", start, end, {
        120: 5.0,
        180: 5.0,
        240: 5.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  # When instances do not match
  def ts_side_effect4(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 4.0,
        180: 4.0,
        240: 4.0,
        300: 4.0,
      })
    ])
  ts2.execute.side_effect = ts_side_effect4

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 2 == len(metrics)
  instances = []
  for metric in metrics:
    instances.append(metric.instance)
    assert {
      120: 0.5,
      180: 0.5,
      240: 0.5,
      300: 0.5
    } == metric.timeline
  assert "instance" in instances and "instance2" in instances

@pytest.mark.asyncio
async def test_MULTIPLY_execute():
  ts = Mock(TS)
  operator = Multiply([float(100), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Return mocked timeline
  def ts_side_effect(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 1.0,
        180: 2.0,
        240: 4.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert "instance" == metrics[0].instance
  assert {
    120: 100.0,
    180: 200.0,
    240: 400.0,
    300: 500.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_MULTIPLY_execute_when_exception():
  ts = Mock(TS)
  operator = Multiply([float(100), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # In case of exception
  def ts_side_effect2(*args):
    raise Exception("some_exception")
  ts.execute.side_effect = ts_side_effect2

  with pytest.raises(Exception):
    metrics = await operator.execute(tracker, tmanager, start, end)

@pytest.mark.asyncio
async def test_MULTIPLY_execute_when_missing_values():
  ts = Mock(TS)
  operator = Multiply([float(100), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # When missing a value
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        180: 2.0,
        240: 4.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert "instance" == metrics[0].instance
  assert {
    180: 200.0,
    240: 400.0,
    300: 500.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_MULTIPLY_execute_with_multiple_ts():
  ts = Mock(TS)
  ts2 = Mock(TS)
  operator = Multiply([ts, ts2])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Multiple timelines
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 1.0,
        180: 1.0,
        240: 1.0,
        300: 1.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      }),
      Metrics("component", "metric_name", "instance3", start, end, {
        120: 3.0,
        180: 3.0,
        240: 3.0,
        300: 3.0,
      }),
      Metrics("component", "metric_name", "instance4", start, end, {
        120: 4.0,
        180: 4.0,
        240: 4.0,
        300: 4.0,
      }),
      Metrics("component", "metric_name", "instance5", start, end, {
        120: 5.0,
        180: 5.0,
        240: 5.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  def ts_side_effect4(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 4.0,
        180: 4.0,
        240: 4.0,
        300: 4.0,
      }),
      Metrics("component", "metric_name", "instance3", start, end, {
        120: 6.0,
        180: 6.0,
        240: 6.0,
        300: 6.0,
      }),
      Metrics("component", "metric_name", "instance4", start, end, {
        120: 8.0,
        180: 8.0,
        240: 8.0,
        300: 8.0,
      }),
      Metrics("component", "metric_name", "instance5", start, end, {
        120: 10.0,
        180: 10.0,
        240: 10.0,
        300: 10.0,
      })
    ])
  ts2.execute.side_effect = ts_side_effect4

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 5 == len(metrics)
  for metric in metrics:
    if metric.instance == "instance":
      assert {
        120: 2,
        180: 2,
        240: 2,
        300: 2
      } == metric.timeline
    elif metric.instance == "instance2":
      assert {
        120: 8,
        180: 8,
        240: 8,
        300: 8
      } == metric.timeline
    elif metric.instance == "instance3":
      assert {
        120: 18,
        180: 18,
        240: 18,
        300: 18
      } == metric.timeline
    elif metric.instance == "instance4":
      assert {
        120: 32,
        180: 32,
        240: 32,
        300: 32
      } == metric.timeline
    elif metric.instance == "instance5":
      assert {
        120: 50,
        180: 50,
        240: 50,
        300: 50
      } == metric.timeline

@pytest.mark.asyncio
async def test_MULTIPLY_execute_with_multiple_ts_when_instances_do_not_match():
  ts = Mock(TS)
  ts2 = Mock(TS)
  operator = Multiply([ts, ts2])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Multiple timelines
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 1.0,
        180: 1.0,
        240: 1.0,
        300: 1.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      }),
      Metrics("component", "metric_name", "instance3", start, end, {
        120: 3.0,
        180: 3.0,
        240: 3.0,
        300: 3.0,
      }),
      Metrics("component", "metric_name", "instance4", start, end, {
        120: 4.0,
        180: 4.0,
        240: 4.0,
        300: 4.0,
      }),
      Metrics("component", "metric_name", "instance5", start, end, {
        120: 5.0,
        180: 5.0,
        240: 5.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  # When instances do not match
  def ts_side_effect4(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 4.0,
        180: 4.0,
        240: 4.0,
        300: 4.0,
      })
    ])
  ts2.execute.side_effect = ts_side_effect4

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 2 == len(metrics)
  instances = []
  for metric in metrics:
    instances.append(metric.instance)
    if metric.instance == "instance":
      assert {
        120: 2,
        180: 2,
        240: 2,
        300: 2
      } == metric.timeline
    elif metric.instance == "instance2":
      assert {
        120: 8,
        180: 8,
        240: 8,
        300: 8
      } == metric.timeline
  assert "instance" in instances and "instance2" in instances

@pytest.mark.asyncio
async def test_SUBTRACT_execute():
  ts = Mock(TS)
  operator = Subtract([float(100), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Return mocked timeline
  def ts_side_effect(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 1.0,
        180: 2.0,
        240: 4.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert "instance" == metrics[0].instance
  assert {
    120: 99.0,
    180: 98.0,
    240: 96.0,
    300: 95.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_SUBTRACT_execute_when_exception():
  ts = Mock(TS)
  operator = Subtract([float(100), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # In case of exception
  def ts_side_effect2(*args):
    raise Exception("some_exception")
  ts.execute.side_effect = ts_side_effect2

  with pytest.raises(Exception):
    metrics = await operator.execute(tracker, tmanager, start, end)

@pytest.mark.asyncio
async def test_SUBTRACT_execute_when_missing_values():
  ts = Mock(TS)
  operator = Subtract([float(100), ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # When missing a value
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        180: 2.0,
        240: 4.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert "instance" == metrics[0].instance
  assert {
    180: 98.0,
    240: 96.0,
    300: 95.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_SUBTRACT_execute_with_multiple_ts():
  ts = Mock(TS)
  ts2 = Mock(TS)
  operator = Subtract([ts, ts2])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Multiple timelines
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 1.0,
        180: 1.0,
        240: 1.0,
        300: 1.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      }),
      Metrics("component", "metric_name", "instance3", start, end, {
        120: 3.0,
        180: 3.0,
        240: 3.0,
        300: 3.0,
      }),
      Metrics("component", "metric_name", "instance4", start, end, {
        120: 4.0,
        180: 4.0,
        240: 4.0,
        300: 4.0,
      }),
      Metrics("component", "metric_name", "instance5", start, end, {
        120: 5.0,
        180: 5.0,
        240: 5.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  def ts_side_effect4(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 4.0,
        180: 4.0,
        240: 4.0,
        300: 4.0,
      }),
      Metrics("component", "metric_name", "instance3", start, end, {
        120: 6.0,
        180: 6.0,
        240: 6.0,
        300: 6.0,
      }),
      Metrics("component", "metric_name", "instance4", start, end, {
        120: 8.0,
        180: 8.0,
        240: 8.0,
        300: 8.0,
      }),
      Metrics("component", "metric_name", "instance5", start, end, {
        120: 10.0,
        180: 10.0,
        240: 10.0,
        300: 10.0,
      })
    ])
  ts2.execute.side_effect = ts_side_effect4

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 5 == len(metrics)
  for metric in metrics:
    if metric.instance == "instance":
      assert {
        120: -1,
        180: -1,
        240: -1,
        300: -1
      } == metric.timeline
    elif metric.instance == "instance2":
      assert {
        120: -2,
        180: -2,
        240: -2,
        300: -2
      } == metric.timeline
    elif metric.instance == "instance3":
      assert {
        120: -3,
        180: -3,
        240: -3,
        300: -3
      } == metric.timeline
    elif metric.instance == "instance4":
      assert {
        120: -4,
        180: -4,
        240: -4,
        300: -4
      } == metric.timeline
    elif metric.instance == "instance5":
      assert {
        120: -5,
        180: -5,
        240: -5,
        300: -5
      } == metric.timeline

@pytest.mark.asyncio
async def test_SUBTRACT_execute_with_multiple_ts_when_instances_do_not_match():
  ts = Mock(TS)
  ts2 = Mock(TS)
  operator = Subtract([ts, ts2])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Multiple timelines
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 1.0,
        180: 1.0,
        240: 1.0,
        300: 1.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      }),
      Metrics("component", "metric_name", "instance3", start, end, {
        120: 3.0,
        180: 3.0,
        240: 3.0,
        300: 3.0,
      }),
      Metrics("component", "metric_name", "instance4", start, end, {
        120: 4.0,
        180: 4.0,
        240: 4.0,
        300: 4.0,
      }),
      Metrics("component", "metric_name", "instance5", start, end, {
        120: 5.0,
        180: 5.0,
        240: 5.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  # When instances do not match
  def ts_side_effect4(*args):
    assert (tracker, tmanager, 100, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start, end, {
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      }),
      Metrics("component", "metric_name", "instance2", start, end, {
        120: 4.0,
        180: 4.0,
        240: 4.0,
        300: 4.0,
      })
    ])
  ts2.execute.side_effect = ts_side_effect4

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 2 == len(metrics)
  instances = []
  for metric in metrics:
    instances.append(metric.instance)
    if metric.instance == "instance":
      assert {
        120: -1,
        180: -1,
        240: -1,
        300: -1
      } == metric.timeline
    elif metric.instance == "instance2":
      assert {
        120: -2,
        180: -2,
        240: -2,
        300: -2
      } == metric.timeline
  assert "instance" in instances and "instance2" in instances

@pytest.mark.asyncio
async def test_RATE_execute():
  ts = Mock(TS)
  operator = Rate([ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Return mocked timeline
  def ts_side_effect(*args):
    assert (tracker, tmanager, 40, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start-60, end, {
        60: 0.0,
        120: 1.0,
        180: 2.0,
        240: 4.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert "instance" == metrics[0].instance
  assert {
    120: 1.0,
    180: 1.0,
    240: 2.0,
    300: 1.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_RATE_execute_when_exception():
  ts = Mock(TS)
  operator = Rate([ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # In case of exception
  def ts_side_effect2(*args):
    raise Exception("some_exception")
  ts.execute.side_effect = ts_side_effect2

  with pytest.raises(Exception):
    metrics = await operator.execute(tracker, tmanager, start, end)

@pytest.mark.asyncio
async def test_RATE_execute_when_missing_values():
  ts = Mock(TS)
  operator = Rate([ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # When missing a value
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 40, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start-60, end, {
        60: 0.0,
          # 120: 1.0, # Missing
        180: 2.0,
        240: 4.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 1 == len(metrics)
  assert "instance" == metrics[0].instance
  assert {
      # 180: 2.0, # Won't be there since 120 is missing
    240: 2.0,
    300: 1.0
  } == metrics[0].timeline

@pytest.mark.asyncio
async def test_RATE_execute_with_multiple_ts():
  ts = Mock(TS)
  operator = Rate([ts])
  tmanager = Mock()
  tracker = Mock()
  start = 100
  end = 300

  # Multiple timelines
  def ts_side_effect3(*args):
    assert (tracker, tmanager, 40, 300) == args
    return ([
      Metrics("component", "metric_name", "instance", start-60, end, {
        60: 0.0,
        120: 1.0,
        180: 1.0,
        240: 1.0,
        300: 1.0,
      }),
      Metrics("component", "metric_name", "instance2", start-60, end, {
        60: 0.0,
        120: 2.0,
        180: 2.0,
        240: 2.0,
        300: 2.0,
      }),
      Metrics("component", "metric_name", "instance3", start-60, end, {
        60: 0.0,
        120: 3.0,
        180: 3.0,
        240: 3.0,
        300: 3.0,
      }),
      Metrics("component", "metric_name", "instance4", start-60, end, {
        60: 0.0,
        120: 4.0,
        180: 4.0,
        240: 4.0,
        300: 4.0,
      }),
      Metrics("component", "metric_name", "instance5", start-60, end, {
        60: 0.0,
        120: 5.0,
        180: 5.0,
        240: 5.0,
        300: 5.0,
      })
    ])
  ts.execute.side_effect = ts_side_effect3

  metrics = await operator.execute(tracker, tmanager, start, end)
  assert 5 == len(metrics)
  for metric in metrics:
    if metric.instance == "instance":
      assert {
        120: 1,
        180: 0,
        240: 0,
        300: 0
      } == metric.timeline
    elif metric.instance == "instance2":
      assert {
        120: 2,
        180: 0,
        240: 0,
        300: 0
      } == metric.timeline
    elif metric.instance == "instance3":
      assert {
        120: 3,
        180: 0,
        240: 0,
        300: 0
      } == metric.timeline
    elif metric.instance == "instance4":
      assert {
        120: 4,
        180: 0,
        240: 0,
        300: 0
      } == metric.timeline
    elif metric.instance == "instance5":
      assert {
        120: 5,
        180: 0,
        240: 0,
        300: 0
      } == metric.timeline
