/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.fury.serializer;

import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertFalse;
import static org.testng.Assert.assertNotEquals;
import static org.testng.Assert.assertNotSame;
import static org.testng.Assert.assertTrue;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apache.fury.Fury;
import org.apache.fury.codegen.CompileUnit;
import org.apache.fury.codegen.JaninoUtils;
import org.apache.fury.test.bean.BeanA;
import org.testng.annotations.Test;

public class ForwardSerializerTest {

  private ForwardSerializer createSerializer(String serializerType) {
    switch (serializerType) {
      case "Fury":
        return new ForwardSerializer(
            new ForwardSerializer.DefaultFuryProxy() {
              @Override
              protected Fury newFurySerializer(ClassLoader loader) {
                Fury fury = super.newFurySerializer(loader);
                // We can register custom serializers here.
                System.out.printf("Created serializer %s, start to do init staff.\n", fury);
                return fury;
              }
            });
      case "Kryo":
        return new ForwardSerializer(
            new ForwardSerializer.SerializerProxy<Kryo>() {
              private final ThreadLocal<Output> outputLocal =
                  ThreadLocal.withInitial(() -> new Output(32, Integer.MAX_VALUE));

              @Override
              protected Kryo newSerializer() {
                // We can register custom serializers here.
                Kryo kryo = new Kryo();
                kryo.setRegistrationRequired(false);
                return kryo;
              }

              @Override
              protected void register(Kryo serializer, Class clz) {
                serializer.register(clz);
              }

              @Override
              protected byte[] serialize(Kryo serializer, Object obj) {
                Output output = outputLocal.get();
                output.reset();
                serializer.writeClassAndObject(output, obj);
                return output.toBytes();
              }

              @Override
              protected Object deserialize(Kryo serializer, byte[] bytes) {
                Input input = new Input(bytes);
                return serializer.readClassAndObject(input);
              }
            });
      default:
        throw new UnsupportedOperationException("Unsupported serializer type " + serializerType);
    }
  }

  @Test
  public void testSerialize() {
    BeanA beanA = BeanA.createBeanA(3);
    ForwardSerializer kryo = createSerializer("Kryo");
    assertEquals(kryo.deserialize(kryo.serialize(beanA)), beanA);
    ForwardSerializer fury = createSerializer("Fury");
    assertEquals(fury.deserialize(fury.serialize(beanA)), beanA);
  }

  private volatile boolean hasException;

  @Test
  public void testConcurrent() throws InterruptedException {
    BeanA beanA = BeanA.createBeanA(3);
    for (String type : new String[] {"Kryo", "Fury"}) {
      ForwardSerializer serializer = createSerializer(type);
      serializer.register(BeanA.class);
      assertEquals(serializer.deserialize(serializer.serialize(beanA)), beanA);
      ExecutorService executorService = Executors.newFixedThreadPool(12);
      for (int i = 0; i < 1000; i++) {
        executorService.execute(
            () -> {
              for (int j = 0; j < 10; j++) {
                try {
                  assertEquals(serializer.deserialize(serializer.serialize(beanA)), beanA);
                } catch (Exception e) {
                  hasException = true;
                  e.printStackTrace();
                  throw e;
                }
              }
            });
      }
      executorService.shutdown();
      assertTrue(executorService.awaitTermination(30, TimeUnit.SECONDS));
      assertFalse(hasException);
    }
  }

  @Test
  public void testClassLoader() throws Exception {
    ForwardSerializer serializer = createSerializer("Fury");
    CompileUnit unit =
        new CompileUnit(
            "demo.pkg1",
            "A",
            (""
                + "package demo.pkg1;\n"
                + "public class A {\n"
                + "  public String f1 = \"str1\";\n"
                + "  public String f2 = \"str2\";\n"
                + "}"));
    ClassLoader loader = null;
    for (int i = 0; i < 5; i++) {
      ClassLoader newLoader = JaninoUtils.compile(getClass().getClassLoader(), unit);
      assertNotSame(loader, newLoader);
      assertNotEquals(loader, newLoader);
      loader = newLoader;
      Class<?> clz = loader.loadClass("demo.pkg1.A");
      Object a = clz.newInstance();
      serializer.setClassLoader(loader);
      serializer.deserialize(serializer.serialize(a));
    }
  }
}
