// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

use crate::buffer::{Reader, Writer};
use crate::error::Error;
use crate::fory::Fory;
use crate::meta::TypeMeta;
use std::any::TypeId;
use std::collections::HashMap;
use std::rc::Rc;

#[derive(Default)]
pub struct MetaReaderResolver {
    reading_type_defs: Vec<Rc<TypeMeta>>,
}

impl MetaReaderResolver {
    pub fn get(&self, index: usize) -> &Rc<TypeMeta> {
        unsafe { self.reading_type_defs.get_unchecked(index) }
    }

    pub fn load(&mut self, reader: &mut Reader) {
        let meta_size = reader.var_int32();
        self.reading_type_defs.reserve(meta_size as usize);
        for _ in 0..meta_size {
            self.reading_type_defs
                .push(Rc::new(TypeMeta::from_bytes(reader)));
        }
    }
}

#[derive(Default)]
pub struct MetaWriterResolver<'a> {
    type_defs: Vec<&'a Vec<u8>>,
    type_id_index_map: HashMap<TypeId, usize>,
}

#[allow(dead_code)]
impl<'a> MetaWriterResolver<'a> {
    pub fn push<'b: 'a>(&mut self, type_id: TypeId, fory: &'a Fory) -> usize {
        match self.type_id_index_map.get(&type_id) {
            None => {
                let index = self.type_defs.len();
                self.type_defs.push(
                    fory.get_class_resolver()
                        .get_class_info(type_id)
                        .get_type_def(),
                );
                self.type_id_index_map.insert(type_id, index);
                index
            }
            Some(index) => *index,
        }
    }

    pub fn to_bytes(&self, writer: &mut Writer) -> Result<(), Error> {
        writer.var_int32(self.type_defs.len() as i32);
        for item in &self.type_defs {
            writer.bytes(item)
        }
        Ok(())
    }

    pub fn reset(&mut self) {
        self.type_defs.clear();
    }
}
