/*
 * Decompiled with CFR 0.152.
 */
package org.redisson.connection;

import io.netty.resolver.AddressResolver;
import io.netty.resolver.AddressResolverGroup;
import io.netty.util.concurrent.EventExecutor;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.FutureListener;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.ScheduledFuture;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import org.redisson.client.RedisClient;
import org.redisson.connection.ClientConnectionsEntry;
import org.redisson.connection.ConnectionManager;
import org.redisson.connection.MasterSlaveEntry;
import org.redisson.misc.AsyncCountDownLatch;
import org.redisson.misc.RedisURI;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DNSMonitor {
    private static final Logger log = LoggerFactory.getLogger(DNSMonitor.class);
    private final AddressResolver<InetSocketAddress> resolver;
    private final ConnectionManager connectionManager;
    private final Map<RedisURI, InetSocketAddress> masters = new HashMap<RedisURI, InetSocketAddress>();
    private final Map<RedisURI, InetSocketAddress> slaves = new HashMap<RedisURI, InetSocketAddress>();
    private ScheduledFuture<?> dnsMonitorFuture;
    private long dnsMonitoringInterval;

    public DNSMonitor(ConnectionManager connectionManager, RedisClient masterHost, Collection<RedisURI> slaveHosts, long dnsMonitoringInterval, AddressResolverGroup<InetSocketAddress> resolverGroup) {
        this.resolver = resolverGroup.getResolver((EventExecutor)connectionManager.getGroup().next());
        masterHost.resolveAddr().join();
        this.masters.put(masterHost.getConfig().getAddress(), masterHost.getAddr());
        for (RedisURI host : slaveHosts) {
            Future resolveFuture = this.resolver.resolve((SocketAddress)InetSocketAddress.createUnresolved(host.getHost(), host.getPort()));
            resolveFuture.syncUninterruptibly();
            this.slaves.put(host, (InetSocketAddress)resolveFuture.getNow());
        }
        this.connectionManager = connectionManager;
        this.dnsMonitoringInterval = dnsMonitoringInterval;
    }

    public void start() {
        this.monitorDnsChange();
        log.debug("DNS monitoring enabled; Current masters: {}, slaves: {}", this.masters, this.slaves);
    }

    public void stop() {
        if (this.dnsMonitorFuture != null) {
            this.dnsMonitorFuture.cancel(true);
        }
    }

    private void monitorDnsChange() {
        this.dnsMonitorFuture = this.connectionManager.getGroup().schedule(new Runnable(){

            @Override
            public void run() {
                if (DNSMonitor.this.connectionManager.isShuttingDown()) {
                    return;
                }
                AsyncCountDownLatch latch = new AsyncCountDownLatch();
                latch.latch(() -> DNSMonitor.this.monitorDnsChange(), DNSMonitor.this.masters.size() + DNSMonitor.this.slaves.size());
                DNSMonitor.this.monitorMasters(latch);
                DNSMonitor.this.monitorSlaves(latch);
            }
        }, this.dnsMonitoringInterval, TimeUnit.MILLISECONDS);
    }

    private void monitorMasters(AsyncCountDownLatch latch) {
        for (Map.Entry<RedisURI, InetSocketAddress> entry : this.masters.entrySet()) {
            log.debug("Request sent to resolve ip address for master host: {}", (Object)entry.getKey().getHost());
            Future resolveFuture = this.resolver.resolve((SocketAddress)InetSocketAddress.createUnresolved(entry.getKey().getHost(), entry.getKey().getPort()));
            resolveFuture.addListener((GenericFutureListener)((FutureListener)future -> {
                if (!future.isSuccess()) {
                    log.error("Unable to resolve " + ((RedisURI)entry.getKey()).getHost(), future.cause());
                    latch.countDown();
                    return;
                }
                log.debug("Resolved ip: {} for master host: {}", (Object)((InetSocketAddress)future.getNow()).getAddress(), (Object)((RedisURI)entry.getKey()).getHost());
                InetSocketAddress currentMasterAddr = (InetSocketAddress)entry.getValue();
                InetSocketAddress newMasterAddr = (InetSocketAddress)future.getNow();
                if (!newMasterAddr.getAddress().equals(currentMasterAddr.getAddress())) {
                    log.info("Detected DNS change. Master {} has changed ip from {} to {}", new Object[]{entry.getKey(), currentMasterAddr.getAddress().getHostAddress(), newMasterAddr.getAddress().getHostAddress()});
                    MasterSlaveEntry masterSlaveEntry = this.connectionManager.getEntry(currentMasterAddr);
                    if (masterSlaveEntry == null) {
                        log.error("Unable to find entry for current master {}", (Object)currentMasterAddr);
                        latch.countDown();
                        return;
                    }
                    CompletableFuture<RedisClient> changeFuture = masterSlaveEntry.changeMaster(newMasterAddr, (RedisURI)entry.getKey());
                    changeFuture.whenComplete((r, e) -> {
                        latch.countDown();
                        if (e == null) {
                            this.masters.put((RedisURI)entry.getKey(), newMasterAddr);
                        }
                    });
                } else {
                    latch.countDown();
                }
            }));
        }
    }

    private void monitorSlaves(AsyncCountDownLatch latch) {
        for (Map.Entry<RedisURI, InetSocketAddress> entry : this.slaves.entrySet()) {
            log.debug("Request sent to resolve ip address for slave host: {}", (Object)entry.getKey().getHost());
            Future resolveFuture = this.resolver.resolve((SocketAddress)InetSocketAddress.createUnresolved(entry.getKey().getHost(), entry.getKey().getPort()));
            resolveFuture.addListener((GenericFutureListener)((FutureListener)future -> {
                if (!future.isSuccess()) {
                    log.error("Unable to resolve " + ((RedisURI)entry.getKey()).getHost(), future.cause());
                    latch.countDown();
                    return;
                }
                log.debug("Resolved ip: {} for slave host: {}", (Object)((InetSocketAddress)future.getNow()).getAddress(), (Object)((RedisURI)entry.getKey()).getHost());
                InetSocketAddress currentSlaveAddr = (InetSocketAddress)entry.getValue();
                InetSocketAddress newSlaveAddr = (InetSocketAddress)future.getNow();
                if (!newSlaveAddr.getAddress().equals(currentSlaveAddr.getAddress())) {
                    log.info("Detected DNS change. Slave {} has changed ip from {} to {}", new Object[]{((RedisURI)entry.getKey()).getHost(), currentSlaveAddr.getAddress().getHostAddress(), newSlaveAddr.getAddress().getHostAddress()});
                    boolean slaveFound = false;
                    for (MasterSlaveEntry masterSlaveEntry : this.connectionManager.getEntrySet()) {
                        if (!masterSlaveEntry.hasSlave(currentSlaveAddr)) continue;
                        slaveFound = true;
                        if (masterSlaveEntry.hasSlave(newSlaveAddr)) {
                            masterSlaveEntry.slaveUp(newSlaveAddr, ClientConnectionsEntry.FreezeReason.MANAGER);
                            masterSlaveEntry.slaveDown(currentSlaveAddr, ClientConnectionsEntry.FreezeReason.MANAGER);
                            this.slaves.put((RedisURI)entry.getKey(), newSlaveAddr);
                            latch.countDown();
                            break;
                        }
                        CompletableFuture<Void> addFuture = masterSlaveEntry.addSlave(newSlaveAddr, (RedisURI)entry.getKey());
                        addFuture.whenComplete((res, e) -> {
                            latch.countDown();
                            if (e != null) {
                                log.error("Can't add slave: " + newSlaveAddr, e);
                                return;
                            }
                            masterSlaveEntry.slaveDown(currentSlaveAddr, ClientConnectionsEntry.FreezeReason.MANAGER);
                            this.slaves.put((RedisURI)entry.getKey(), newSlaveAddr);
                        });
                        break;
                    }
                    if (!slaveFound) {
                        latch.countDown();
                    }
                } else {
                    latch.countDown();
                }
            }));
        }
    }
}

