/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.eventmesh.runtime.admin.handler;

import org.apache.eventmesh.common.Constants;
import org.apache.eventmesh.common.protocol.tcp.UserAgent;
import org.apache.eventmesh.runtime.boot.EventMeshTCPServer;
import org.apache.eventmesh.runtime.constants.EventMeshConstants;
import org.apache.eventmesh.runtime.core.protocol.tcp.client.group.ClientGroupWrapper;
import org.apache.eventmesh.runtime.core.protocol.tcp.client.group.ClientSessionGroupMapping;
import org.apache.eventmesh.runtime.core.protocol.tcp.client.session.Session;
import org.apache.eventmesh.runtime.util.NetUtils;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;

/**
 * query client subscription by topic
 */
public class ShowListenClientByTopicHandler implements HttpHandler {

    private Logger logger = LoggerFactory.getLogger(ShowListenClientByTopicHandler.class);

    private final EventMeshTCPServer eventMeshTCPServer;

    public ShowListenClientByTopicHandler(EventMeshTCPServer eventMeshTCPServer) {
        this.eventMeshTCPServer = eventMeshTCPServer;
    }

    @Override
    public void handle(HttpExchange httpExchange) throws IOException {
        StringBuilder result = new StringBuilder();
        try (OutputStream out = httpExchange.getResponseBody()) {
            String queryString = httpExchange.getRequestURI().getQuery();
            Map<String, String> queryStringInfo = NetUtils.formData2Dic(queryString);
            String topic = queryStringInfo.get(EventMeshConstants.MANAGE_TOPIC);

            String newLine = System.getProperty("line.separator");
            logger.info("showListeningClientByTopic,topic:{}=================", topic);
            ClientSessionGroupMapping clientSessionGroupMapping = eventMeshTCPServer.getClientSessionGroupMapping();
            ConcurrentHashMap<String, ClientGroupWrapper> clientGroupMap = clientSessionGroupMapping.getClientGroupMap();
            if (!clientGroupMap.isEmpty()) {
                for (ClientGroupWrapper cgw : clientGroupMap.values()) {
                    Set<Session> listenSessionSet = cgw.getTopic2sessionInGroupMapping().get(topic);
                    if (listenSessionSet != null && listenSessionSet.size() > 0) {
                        result.append(String.format("group:%s", cgw.getGroup())).append(newLine);
                        for (Session session : listenSessionSet) {
                            UserAgent userAgent = session.getClient();
                            result.append(String.format("pid=%s | ip=%s | port=%s | path=%s | version=%s", userAgent.getPid(), userAgent
                                    .getHost(), userAgent.getPort(), userAgent.getPath(), userAgent.getVersion()))
                                    .append(newLine);
                        }
                    }
                }
            }
            httpExchange.sendResponseHeaders(200, 0);
            out.write(result.toString().getBytes(Constants.DEFAULT_CHARSET));
        } catch (Exception e) {
            logger.error("ShowListenClientByTopicHandler fail...", e);
        }
    }
}
