/*
Licensed to the Apache Software Foundation (ASF) under one or more
contributor license agreements.  See the NOTICE file distributed with
this work for additional information regarding copyright ownership.
The ASF licenses this file to You under the Apache License, Version 2.0
(the "License"); you may not use this file except in compliance with
the License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package api

import (
	"database/sql"
	"github.com/apache/incubator-devlake/core/dal"
	"gorm.io/gorm/migrator"
	"testing"

	coreModels "github.com/apache/incubator-devlake/core/models"
	mockdal "github.com/apache/incubator-devlake/mocks/core/dal"
	mockplugin "github.com/apache/incubator-devlake/mocks/core/plugin"

	"github.com/apache/incubator-devlake/core/errors"
	"github.com/apache/incubator-devlake/core/models/common"
	"github.com/apache/incubator-devlake/core/models/domainlayer/devops"
	"github.com/apache/incubator-devlake/core/plugin"
	helper "github.com/apache/incubator-devlake/helpers/pluginhelper/api"
	"github.com/apache/incubator-devlake/helpers/unithelper"
	"github.com/apache/incubator-devlake/plugins/bamboo/models"
	"github.com/apache/incubator-devlake/plugins/bamboo/tasks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func TestMakeDataSourcePipelinePlanV200(t *testing.T) {
	const testConnectionID uint64 = 1
	const testScopeConfigId uint64 = 2
	const testKey string = "TEST"
	const testBambooEndPoint string = "http://mail.nddtf.com:8085/rest/api/latest/"
	const testLink string = "http://mail.nddtf.com:8085/rest/api/latest/plan/TEST"
	const testUser string = "username"
	const testPass string = "password"
	const testName string = "bamboo-test"
	const testScopeConfigName string = "bamboo scope config"
	const testProxy string = ""

	bpScopes := []*coreModels.BlueprintScope{
		{
			ScopeId: testKey,
		},
	}

	var testBambooPlan = &models.BambooPlan{
		Scope: common.Scope{
			ConnectionId:  testConnectionID,
			ScopeConfigId: testScopeConfigId,
		},
		PlanKey: testKey,
		Name:    testName,
		Href:    testLink,
	}

	var testScopeConfig = &models.BambooScopeConfig{
		ScopeConfig: common.ScopeConfig{
			Model: common.Model{
				ID: testScopeConfigId,
			},
			Entities: []string{plugin.DOMAIN_TYPE_CICD},
			Name:     testScopeConfigName,
		},
	}

	var testColumTypes = []dal.ColumnMeta{
		migrator.ColumnType{
			NameValue: sql.NullString{
				String: "abc",
				Valid:  true,
			},
		},
	}

	var testBambooConnection = &models.BambooConnection{
		BaseConnection: helper.BaseConnection{
			Name: testName,
			Model: common.Model{
				ID: testConnectionID,
			},
		},
		BambooConn: models.BambooConn{
			RestConnection: helper.RestConnection{
				Endpoint:         testBambooEndPoint,
				Proxy:            testProxy,
				RateLimitPerHour: 0,
			},
			BasicAuth: helper.BasicAuth{
				Username: testUser,
				Password: testPass,
			},
		},
	}

	var expectRepoId = "bamboo:BambooPlan:1:TEST"

	var testSubTaskMeta = []plugin.SubTaskMeta{
		tasks.ConvertPlansMeta,
	}

	var expectPlans = coreModels.PipelinePlan{
		{
			{
				Plugin: "bamboo",
				Subtasks: []string{
					tasks.ConvertPlansMeta.Name,
				},
				Options: map[string]interface{}{
					"connectionId":  uint64(1),
					"planKey":       testKey,
					"scopeConfigId": testScopeConfigId,
				},
			},
		},
	}

	expectCicdScope := devops.NewCicdScope(expectRepoId, testName)
	expectCicdScope.Description = ""
	expectCicdScope.Url = ""

	var err errors.Error

	// register bamboo plugin for NewDomainIdGenerator
	mockMeta := mockplugin.NewPluginMeta(t)
	mockMeta.On("Name").Return("bamboo").Maybe()
	mockMeta.On("RootPkgPath").Return("github.com/apache/incubator-devlake/plugins/bamboo")
	err = plugin.RegisterPlugin("bamboo", mockMeta)
	assert.Equal(t, err, nil)

	// Refresh Global Variables and set the sql mock
	basicRes = unithelper.DummyBasicRes(func(mockDal *mockdal.Dal) {
		mockDal.On("First", mock.AnythingOfType("*models.BambooConnection"), mock.Anything).Run(func(args mock.Arguments) {
			dst := args.Get(0).(*models.BambooConnection)
			*dst = *testBambooConnection
		}).Return(nil)

		mockDal.On("First", mock.AnythingOfType("*models.BambooPlan"), mock.Anything).Run(func(args mock.Arguments) {
			dst := args.Get(0).(*models.BambooPlan)
			*dst = *testBambooPlan
		}).Return(nil)

		mockDal.On("First", mock.AnythingOfType("*models.BambooScopeConfig"), mock.Anything).Run(func(args mock.Arguments) {
			dst := args.Get(0).(*models.BambooScopeConfig)
			*dst = *testScopeConfig
		}).Return(nil)
		mockDal.On("GetPrimarykeyColumns", mock.AnythingOfType("*models.BambooConnection"), mock.Anything).Run(nil).Return(
			testColumTypes, nil)
		mockDal.On("GetColumns", mock.AnythingOfType("models.BambooConnection"), mock.Anything).Run(nil).Return(
			testColumTypes, nil)
		mockDal.On("GetColumns", mock.AnythingOfType("models.BambooPlan"), mock.Anything).Run(nil).Return(
			testColumTypes, nil)
		mockDal.On("GetColumns", mock.AnythingOfType("models.BambooScopeConfig"), mock.Anything).Run(nil).Return(
			testColumTypes, nil)
	})
	Init(basicRes, mockMeta)

	plans, scopes, err := MakePipelinePlanV200(testSubTaskMeta, testConnectionID, bpScopes)
	assert.Equal(t, err, nil)

	assert.Equal(t, expectPlans, plans)

	// ignore CreatedDate UpdatedDate  CreatedAt UpdatedAt checking
	expectCicdScope.CreatedDate = scopes[0].(*devops.CicdScope).CreatedDate
	expectCicdScope.UpdatedDate = scopes[0].(*devops.CicdScope).UpdatedDate
	expectCicdScope.CreatedAt = scopes[0].(*devops.CicdScope).CreatedAt
	expectCicdScope.UpdatedAt = scopes[0].(*devops.CicdScope).UpdatedAt

	var expectScopes = []plugin.Scope{
		expectCicdScope,
	}

	assert.Equal(t, expectScopes, scopes)
}
