/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
import React, {
  Fragment,
  useEffect,
  useCallback,
  useState,
  useRef
} from 'react'
import { CSSTransition } from 'react-transition-group'
import { useHistory } from 'react-router-dom'
import { ToastNotification } from '@/components/Toast'
import { DEVLAKE_ENDPOINT } from '@/utils/config'
import request from '@/utils/request'
import {
  H2,
  Button,
  Icon,
  Intent,
  Card,
  Elevation,
  Tag,
  Tooltip,
  Position,
  Spinner,
  Colors
} from '@blueprintjs/core'
import { ReactComponent as Logo } from '@/images/devlake-logo.svg'
import { ReactComponent as LogoText } from '@/images/devlake-textmark.svg'
import '@/styles/offline.scss'

const Offline = (props) => {
  const { pollTimer = 7000, retryLimit = 100, endpointUrl = null } = props
  const history = useHistory()
  const testInterval = useRef()
  const retryCount = useRef()

  const [apiEndpointUrl, setApiEndpointUrl] = useState(
    `${DEVLAKE_ENDPOINT}/ping`
  )
  const [isOffline, setIsOffline] = useState(true)
  const [testResponse, setTestResponse] = useState({})
  const [testStatus, setTestStatus] = useState(504)
  const [cardReady, setCardReady] = useState(false)
  // eslint-disable-next-line no-unused-vars
  const [aborted, setAborted] = useState(false)

  const testEndpoint = useCallback(async () => {
    console.log('>> OFFLINE MODE -  TESTING ENDPOINT!', apiEndpointUrl)
    retryCount.current++
    try {
      setCardReady(false)
      const t = await request.get(apiEndpointUrl)
      setTestResponse(t.data)
      setTestStatus(() => t.status ?? 504)
    } catch (e) {
      setTestResponse((tR) => {
        return {
          ...tR,
          data: [],
          error: e.response.data,
          statusCode: e.response.status
        }
      })
      setTestStatus(504)
    }
  }, [apiEndpointUrl])

  useEffect(() => {
    console.log('>> OFFLINE MODE -  ENDPOINT TEST RESPONSE = ', testResponse)
    if (testResponse?.error) {
      ToastNotification.clear()
      ToastNotification.show({
        message: testResponse.error ?? 'Network Error 504 (Gateway Timeout)',
        intent: 'danger',
        icon: 'error'
      })
    }
    setTimeout(() => {
      setCardReady(true)
    }, 0)
  }, [testResponse])

  useEffect(() => {
    console.log('>> OFFLINE MODE - CURRENT API HTTP/STATUS: ', testStatus)
    setIsOffline(testStatus !== 200)
    if (testStatus === 200) {
      clearInterval(testInterval.current)
      ToastNotification.clear()
      ToastNotification.show({
        message: 'DevLake API Online',
        intent: 'danger',
        icon: 'tick-circle'
      })
      console.log('>> OFFLINE MODE - API RESTORED, TERMINATING POLLING!!!')
    }
  }, [testStatus])

  useEffect(() => {
    console.log('>> OFFLINE MODE -  OFFLINE CHECK?', isOffline)
  }, [isOffline])

  useEffect(() => {
    testEndpoint()
  }, [testEndpoint])

  useEffect(() => {
    testInterval.current = setInterval(() => {
      setTestStatus(504)
      testEndpoint()
      // retryCount.current++
    }, pollTimer)

    return () => {
      clearInterval(testInterval.current)
      setCardReady(false)
      setAborted(false)
    }
  }, [pollTimer, testEndpoint])

  useEffect(() => {
    console.log('>> OFFLINE MODE (MOUNTED) - ENDPOINT URL ...', endpointUrl)
    retryCount.current = 1
    setAborted(false)
    setApiEndpointUrl((url) =>
      endpointUrl !== null && endpointUrl !== url
        ? endpointUrl
        : `${DEVLAKE_ENDPOINT}/ping`
    )
  }, [endpointUrl])

  useEffect(() => {
    if (retryCount.current >= retryLimit) {
      console.log(
        '>> OFFLINE MODE - RETRY LIMIT EXCEEDED, TERMINATING POLLING!!!'
      )
      clearInterval(testInterval.current)
      setAborted(true)
      ToastNotification.clear()
      ToastNotification.show({
        message: 'Retry Limit Exceeded, please refresh your browser.',
        intent: 'danger',
        icon: 'error'
      })
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [retryCount.current, retryLimit])

  return (
    <>
      {
        <div
          className={`offline-wrapper ${
            isOffline ? 'is-offline' : 'is-online'
          }`}
        >
          <div
            style={{
              maxWidth: '500px',
              margin: '0 auto',
              display: 'flex',
              height: '100%',
              flexDirection: 'column'
            }}
          >
            <div style={{ margin: 'auto auto', padding: '16px' }}>
              <div className='devlake-logo' style={{ margin: 0 }}>
                <Logo width={48} height={48} className='logo' />
                <LogoText width={100} height={13} className='logo-textmark' />
              </div>
              <CSSTransition
                in={cardReady}
                timeout={900}
                classNames='offline-card'
              >
                <Card elevation={Elevation.THREE}>
                  <span
                    style={{
                      display: 'block',
                      position: 'absolute',
                      right: 0,
                      top: 0,
                      margin: '10px'
                    }}
                  >
                    {isOffline ? (
                      <Spinner size={16} />
                    ) : (
                      <Icon icon='tick-circle' size={16} color={Colors.GRAY4} />
                    )}
                  </span>
                  <div style={{ textAlign: 'center' }}>
                    {isOffline ? (
                      <img
                        style={{
                          margin: '0px auto 10px auto',
                          width: '54px',
                          height: '54px',
                          maxWidth: '5 4px'
                        }}
                        width={54}
                        height={54}
                        // eslint-disable-next-line max-len
                        src='data:image/png;base64,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'
                      />
                    ) : (
                      <img
                        style={{
                          margin: '0px auto 10px auto',
                          width: '64px',
                          height: '64px',
                          maxWidth: '64px'
                        }}
                        width={64}
                        height={64}
                        // eslint-disable-next-line max-len
                        src='data:image/png;base64,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'
                      />
                    )}
                  </div>
                  <H2>
                    DevLake API&nbsp;
                    <CSSTransition
                      in={cardReady}
                      timeout={300}
                      classNames='offline-card-os'
                    >
                      {isOffline ? (
                        <strong
                          style={{
                            padding: '0 3px',
                            fontWeight: '900',
                            backgroundColor: 'rgba(220,50,50,0.2)',
                            color: 'rgb(212, 69, 36)'
                          }}
                        >
                          Offline
                        </strong>
                      ) : (
                        <strong
                          style={{
                            padding: '0 3px',
                            fontWeight: '900',
                            backgroundColor: 'rgba(50,221,80,0.2)',
                            color: 'rgb(36, 212, 72)'
                          }}
                        >
                          Online
                        </strong>
                      )}
                    </CSSTransition>
                  </H2>
                  <Tag>
                    <code>DEVLAKE_ENDPOINT: {DEVLAKE_ENDPOINT}</code>
                  </Tag>
                  {isOffline ? (
                    <>
                      <p style={{ fontSize: '16px', color: '#222222' }}>
                        Please wait for the&nbsp;
                        <strong>Lake API</strong> to start before accessing the{' '}
                        <strong>Configuration Interface</strong>.
                      </p>
                      <Tooltip
                        intent={Intent.PRIMARY}
                        className='offline-continue-tooltip'
                        targetClassName='offline-continue-tooltip-target'
                        content={
                          <>
                            <Icon size={14} icon='warning-sign' /> Proceed with
                            caution.
                          </>
                        }
                        position={Position.RIGHT}
                        style={{ cursor: 'pointer' }}
                      >
                        <Button
                          intent={Intent.PRIMARY}
                          icon='warning-sign'
                          style={{ fontWeight: 700 }}
                          onClick={() => history.replace('/')}
                        >
                          Continue
                        </Button>
                      </Tooltip>
                      <Button
                        icon='refresh'
                        text='Refresh'
                        minimal
                        style={{ marginLeft: '5px' }}
                        disabled={!cardReady}
                        onClick={() => testEndpoint()}
                      />
                    </>
                  ) : (
                    <>
                      <p style={{ fontSize: '16px', color: '#222222' }}>
                        Connectivity to the Lake API service was successful.
                      </p>
                      <Button
                        icon='cog'
                        intent={Intent.PRIMARY}
                        onClick={() => history.replace('/')}
                      >
                        Open Dashboard
                      </Button>
                      <a
                        href='https://github.com/apache/incubator-devlake/blob/main/README.md'
                        target='_blank'
                        rel='noreferrer'
                        style={{ marginLeft: '5px' }}
                        className='bp3-button bp3-minimal'
                      >
                        <Icon
                          icon='help'
                          size={16}
                          style={{ marginRight: '5px' }}
                        />
                        Read Documentation
                      </a>
                    </>
                  )}
                </Card>
              </CSSTransition>
              <div
                style={{
                  margin: '10px 5px 10px 5px',
                  display: 'flex',
                  justifyContnt: 'flex-start'
                }}
              >
                <div style={{ color: 'rgba(33, 33, 33, 0.6)' }}>
                  {isOffline ? (
                    <>
                      <strong style={{ fontWeight: 900, letterSpacing: '2px' }}>
                        OFFLINE &middot;&nbsp;
                      </strong>
                      <a
                        href='https://github.com/apache/incubator-devlake/blob/main/README.md'
                        target='_blank'
                        rel='noreferrer'
                        style={{ fontWeight: 400, letterSpacing: 0 }}
                      >
                        Documentation on{' '}
                        <span style={{ fontWeight: 800 }}>GitHub</span>
                      </a>
                      <br /> Please verify network connectivity if the problem
                      persists.
                      <br />
                      <strong style={{ fontWeight: 600, letterSpacing: '0px' }}>
                        {retryCount.current} failed attempt(s).
                      </strong>
                    </>
                  ) : (
                    <>
                      <strong style={{ fontWeight: 900, letterSpacing: '2px' }}>
                        ONLINE &middot;&nbsp;
                      </strong>
                      <a
                        href='/integrations'
                        style={{
                          color: 'rgba(50,50,50,0.6)',
                          textDecoration: 'underline',
                          fontWeight: 400,
                          letterSpacing: 0
                        }}
                      >
                        Proceed with{' '}
                        <span style={{ fontWeight: 800 }}>
                          Data Connections
                        </span>
                      </a>
                      <br /> Lake API connection restored.
                    </>
                  )}
                </div>
              </div>
            </div>
          </div>
        </div>
      }
    </>
  )
}

export default Offline
