/*
 * Decompiled with CFR 0.152.
 */
package org.apache.celeborn.common.network.client;

import java.io.Closeable;
import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.celeborn.common.exception.CelebornIOException;
import org.apache.celeborn.common.network.buffer.NioManagedBuffer;
import org.apache.celeborn.common.network.client.ChunkReceivedCallback;
import org.apache.celeborn.common.network.client.RpcResponseCallback;
import org.apache.celeborn.common.network.client.TransportResponseHandler;
import org.apache.celeborn.common.network.protocol.ChunkFetchRequest;
import org.apache.celeborn.common.network.protocol.OneWayMessage;
import org.apache.celeborn.common.network.protocol.PushData;
import org.apache.celeborn.common.network.protocol.PushMergedData;
import org.apache.celeborn.common.network.protocol.RpcRequest;
import org.apache.celeborn.common.network.protocol.StreamChunkSlice;
import org.apache.celeborn.common.network.util.NettyUtils;
import org.apache.celeborn.common.read.FetchRequestInfo;
import org.apache.celeborn.common.write.PushRequestInfo;
import org.apache.celeborn.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.celeborn.shaded.com.google.common.base.Objects;
import org.apache.celeborn.shaded.com.google.common.base.Preconditions;
import org.apache.celeborn.shaded.com.google.common.util.concurrent.SettableFuture;
import org.apache.celeborn.shaded.io.netty.channel.Channel;
import org.apache.celeborn.shaded.io.netty.channel.ChannelFuture;
import org.apache.celeborn.shaded.io.netty.util.concurrent.Future;
import org.apache.celeborn.shaded.io.netty.util.concurrent.GenericFutureListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TransportClient
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(TransportClient.class);
    private final Channel channel;
    private final TransportResponseHandler handler;
    private volatile boolean timedOut;
    private static final AtomicLong counter = new AtomicLong();

    public TransportClient(Channel channel, TransportResponseHandler handler) {
        this.channel = Preconditions.checkNotNull(channel);
        this.handler = Preconditions.checkNotNull(handler);
        this.timedOut = false;
    }

    public Channel getChannel() {
        return this.channel;
    }

    public boolean isActive() {
        return !this.timedOut && (this.channel.isOpen() || this.channel.isActive());
    }

    public SocketAddress getSocketAddress() {
        return this.channel.remoteAddress();
    }

    public void fetchChunk(long streamId, int chunkIndex, long fetchDataTimeout, ChunkReceivedCallback callback) {
        this.fetchChunk(streamId, chunkIndex, 0, Integer.MAX_VALUE, fetchDataTimeout, callback);
    }

    public void fetchChunk(long streamId, final int chunkIndex, int offset, int len, long fetchDataTimeout, final ChunkReceivedCallback callback) {
        final StreamChunkSlice streamChunkSlice = new StreamChunkSlice(streamId, chunkIndex, offset, len);
        if (logger.isDebugEnabled()) {
            logger.debug("Sending fetch chunk request {} to {}.", (Object)streamChunkSlice, (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        StdChannelListener listener = new StdChannelListener(streamChunkSlice){

            @Override
            protected void handleFailure(String errorMsg, Throwable cause) {
                TransportClient.this.handler.removeFetchRequest(streamChunkSlice);
                callback.onFailure(chunkIndex, new IOException(errorMsg, cause));
            }
        };
        long dueTime = System.currentTimeMillis() + fetchDataTimeout;
        FetchRequestInfo info = new FetchRequestInfo(dueTime, callback);
        this.handler.addFetchRequest(streamChunkSlice, info);
        ChannelFuture channelFuture = this.channel.writeAndFlush(new ChunkFetchRequest(streamChunkSlice)).addListener(listener);
        info.setChannelFuture(channelFuture);
    }

    public long sendRpc(ByteBuffer message, RpcResponseCallback callback) {
        if (logger.isTraceEnabled()) {
            logger.trace("Sending RPC to {}", (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        long requestId = TransportClient.requestId();
        this.handler.addRpcRequest(requestId, callback);
        RpcChannelListener listener = new RpcChannelListener(requestId, callback);
        this.channel.writeAndFlush(new RpcRequest(requestId, new NioManagedBuffer(message))).addListener(listener);
        return requestId;
    }

    public long sendRpc(ByteBuffer message) {
        if (logger.isTraceEnabled()) {
            logger.trace("Sending RPC to {}", (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        long requestId = TransportClient.requestId();
        this.channel.writeAndFlush(new RpcRequest(requestId, new NioManagedBuffer(message)));
        return requestId;
    }

    public ChannelFuture pushData(PushData pushData, long pushDataTimeout, RpcResponseCallback callback) {
        return this.pushData(pushData, pushDataTimeout, callback, null);
    }

    public ChannelFuture pushData(PushData pushData, long pushDataTimeout, RpcResponseCallback callback, Runnable rpcSendoutCallback) {
        if (logger.isTraceEnabled()) {
            logger.trace("Pushing data to {}", (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        long requestId = TransportClient.requestId();
        long dueTime = System.currentTimeMillis() + pushDataTimeout;
        PushRequestInfo info = new PushRequestInfo(dueTime, callback);
        this.handler.addPushRequest(requestId, info);
        pushData.requestId = requestId;
        PushChannelListener listener = new PushChannelListener(requestId, callback, rpcSendoutCallback);
        ChannelFuture channelFuture = this.channel.writeAndFlush(pushData).addListener(listener);
        info.setChannelFuture(channelFuture);
        return channelFuture;
    }

    public ChannelFuture pushMergedData(PushMergedData pushMergedData, long pushDataTimeout, RpcResponseCallback callback) {
        if (logger.isTraceEnabled()) {
            logger.trace("Pushing merged data to {}", (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        long requestId = TransportClient.requestId();
        long dueTime = System.currentTimeMillis() + pushDataTimeout;
        PushRequestInfo info = new PushRequestInfo(dueTime, callback);
        this.handler.addPushRequest(requestId, info);
        pushMergedData.requestId = requestId;
        PushChannelListener listener = new PushChannelListener(requestId, callback);
        ChannelFuture channelFuture = this.channel.writeAndFlush(pushMergedData).addListener(listener);
        info.setChannelFuture(channelFuture);
        return channelFuture;
    }

    public ByteBuffer sendRpcSync(ByteBuffer message, long timeoutMs) throws IOException {
        final SettableFuture result = SettableFuture.create();
        this.sendRpc(message, new RpcResponseCallback(){

            @Override
            public void onSuccess(ByteBuffer response) {
                ByteBuffer copy = ByteBuffer.allocate(response.remaining());
                copy.put(response);
                copy.flip();
                result.set(copy);
            }

            @Override
            public void onFailure(Throwable e) {
                result.setException(e);
            }
        });
        try {
            return (ByteBuffer)result.get(timeoutMs, TimeUnit.MILLISECONDS);
        }
        catch (Exception e) {
            throw new IOException("Exception in sendRpcSync to: " + this.getSocketAddress(), e);
        }
    }

    public void send(ByteBuffer message) {
        this.channel.writeAndFlush(new OneWayMessage(new NioManagedBuffer(message)));
    }

    public void removeRpcRequest(long requestId) {
        this.handler.removeRpcRequest(requestId);
    }

    public void timeOut() {
        this.timedOut = true;
    }

    @VisibleForTesting
    public TransportResponseHandler getHandler() {
        return this.handler;
    }

    @Override
    public void close() {
        this.channel.close().awaitUninterruptibly(10L, TimeUnit.SECONDS);
    }

    public String toString() {
        return Objects.toStringHelper(this).add("remoteAdress", this.channel.remoteAddress()).add("isActive", this.isActive()).toString();
    }

    public static long requestId() {
        return counter.getAndIncrement();
    }

    private class PushChannelListener
    extends StdChannelListener {
        final long pushRequestId;
        final RpcResponseCallback callback;
        Runnable rpcSendOutCallback;

        PushChannelListener(long pushRequestId, RpcResponseCallback callback) {
            this(pushRequestId, callback, null);
        }

        PushChannelListener(long pushRequestId, RpcResponseCallback callback, Runnable rpcSendOutCallback) {
            super("PUSH " + pushRequestId);
            this.pushRequestId = pushRequestId;
            this.callback = callback;
            this.rpcSendOutCallback = rpcSendOutCallback;
        }

        @Override
        public void operationComplete(Future<? super Void> future) throws Exception {
            super.operationComplete(future);
            if (this.rpcSendOutCallback != null) {
                this.rpcSendOutCallback.run();
            }
        }

        @Override
        protected void handleFailure(String errorMsg, Throwable cause) {
            TransportClient.this.handler.removePushRequest(this.pushRequestId);
            this.callback.onFailure(new CelebornIOException(errorMsg, cause));
        }
    }

    private class RpcChannelListener
    extends StdChannelListener {
        final long rpcRequestId;
        final RpcResponseCallback callback;

        RpcChannelListener(long rpcRequestId, RpcResponseCallback callback) {
            super("RPC " + rpcRequestId);
            this.rpcRequestId = rpcRequestId;
            this.callback = callback;
        }

        @Override
        protected void handleFailure(String errorMsg, Throwable cause) {
            TransportClient.this.handler.removeRpcRequest(this.rpcRequestId);
            this.callback.onFailure(new IOException(errorMsg, cause));
        }
    }

    public class StdChannelListener
    implements GenericFutureListener<Future<? super Void>> {
        final long startTime = System.currentTimeMillis();
        final Object requestId;

        public StdChannelListener(Object requestId) {
            this.requestId = requestId;
        }

        @Override
        public void operationComplete(Future<? super Void> future) throws Exception {
            if (future.isSuccess()) {
                if (logger.isDebugEnabled()) {
                    long timeTaken = System.currentTimeMillis() - this.startTime;
                    logger.debug("Sending request {} to {} took {} ms", new Object[]{this.requestId, NettyUtils.getRemoteAddress(TransportClient.this.channel), timeTaken});
                }
            } else {
                String errorMsg = String.format("Failed to send request %s to %s: %s, channel will be closed", this.requestId, NettyUtils.getRemoteAddress(TransportClient.this.channel), future.cause());
                logger.warn(errorMsg);
                TransportClient.this.channel.close();
                try {
                    this.handleFailure(errorMsg, future.cause());
                }
                catch (Exception e) {
                    logger.error("Uncaught exception in RPC response callback handler!", (Throwable)e);
                }
            }
        }

        protected void handleFailure(String errorMsg, Throwable cause) {
            logger.error("Error encountered " + errorMsg, cause);
        }
    }
}

