/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.client.impl;

import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.util.EnumSet;
import java.util.UUID;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.fs.ReadOption;
import org.apache.hadoop.hdfs.BlockReader;
import org.apache.hadoop.hdfs.PeerCache;
import org.apache.hadoop.hdfs.client.impl.BlockReaderUtil;
import org.apache.hadoop.hdfs.net.Peer;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.datatransfer.DataTransferProtoUtil;
import org.apache.hadoop.hdfs.protocol.datatransfer.PacketHeader;
import org.apache.hadoop.hdfs.protocol.datatransfer.PacketReceiver;
import org.apache.hadoop.hdfs.protocol.datatransfer.Sender;
import org.apache.hadoop.hdfs.protocol.proto.DataTransferProtos;
import org.apache.hadoop.hdfs.protocolPB.PBHelperClient;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenIdentifier;
import org.apache.hadoop.hdfs.server.datanode.CachingStrategy;
import org.apache.hadoop.hdfs.shortcircuit.ClientMmap;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.util.DataChecksum;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class BlockReaderRemote
implements BlockReader {
    static final Logger LOG = LoggerFactory.getLogger(BlockReaderRemote.class);
    static final int TCP_WINDOW_SIZE = 131072;
    private final Peer peer;
    private final DatanodeID datanodeID;
    private final PeerCache peerCache;
    private final long blockId;
    private final ReadableByteChannel in;
    private DataChecksum checksum;
    private final PacketReceiver packetReceiver = new PacketReceiver(true);
    private ByteBuffer curDataSlice = null;
    private long lastSeqNo = -1L;
    private long startOffset;
    private final String filename;
    private final int bytesPerChecksum;
    private final int checksumSize;
    private long bytesNeededToFinish;
    private final boolean verifyChecksum;
    private boolean sentStatusCode = false;
    private final int networkDistance;

    @VisibleForTesting
    public Peer getPeer() {
        return this.peer;
    }

    @Override
    public synchronized int read(byte[] buf, int off, int len) throws IOException {
        boolean logTraceEnabled = LOG.isTraceEnabled();
        UUID randomId = null;
        if (logTraceEnabled) {
            randomId = UUID.randomUUID();
            LOG.trace("Starting read #{} file {} from datanode {}", new Object[]{randomId, this.filename, this.datanodeID.getHostName()});
        }
        if (this.curDataSlice == null || this.curDataSlice.remaining() == 0 && this.bytesNeededToFinish > 0L) {
            this.readNextPacket();
        }
        if (logTraceEnabled) {
            LOG.trace("Finishing read #{}", (Object)randomId);
        }
        if (this.curDataSlice.remaining() == 0) {
            return -1;
        }
        int nRead = Math.min(this.curDataSlice.remaining(), len);
        this.curDataSlice.get(buf, off, nRead);
        return nRead;
    }

    @Override
    public synchronized int read(ByteBuffer buf) throws IOException {
        if (this.curDataSlice == null || this.curDataSlice.remaining() == 0 && this.bytesNeededToFinish > 0L) {
            this.readNextPacket();
        }
        if (this.curDataSlice.remaining() == 0) {
            return -1;
        }
        int nRead = Math.min(this.curDataSlice.remaining(), buf.remaining());
        ByteBuffer writeSlice = this.curDataSlice.duplicate();
        writeSlice.limit(writeSlice.position() + nRead);
        buf.put(writeSlice);
        this.curDataSlice.position(writeSlice.position());
        return nRead;
    }

    private void readNextPacket() throws IOException {
        this.packetReceiver.receiveNextPacket(this.in);
        PacketHeader curHeader = this.packetReceiver.getHeader();
        this.curDataSlice = this.packetReceiver.getDataSlice();
        assert (this.curDataSlice.capacity() == curHeader.getDataLen());
        LOG.trace("DFSClient readNextPacket got header {}", (Object)curHeader);
        if (!curHeader.sanityCheck(this.lastSeqNo)) {
            throw new IOException("BlockReader: error in packet header " + curHeader);
        }
        if (curHeader.getDataLen() > 0) {
            int chunks = 1 + (curHeader.getDataLen() - 1) / this.bytesPerChecksum;
            int checksumsLen = chunks * this.checksumSize;
            assert (this.packetReceiver.getChecksumSlice().capacity() == checksumsLen) : "checksum slice capacity=" + this.packetReceiver.getChecksumSlice().capacity() + " checksumsLen=" + checksumsLen;
            this.lastSeqNo = curHeader.getSeqno();
            if (this.verifyChecksum && this.curDataSlice.remaining() > 0) {
                this.checksum.verifyChunkedSums(this.curDataSlice, this.packetReceiver.getChecksumSlice(), this.filename, curHeader.getOffsetInBlock());
            }
            this.bytesNeededToFinish -= (long)curHeader.getDataLen();
        }
        if (curHeader.getOffsetInBlock() < this.startOffset) {
            int newPos = (int)(this.startOffset - curHeader.getOffsetInBlock());
            this.curDataSlice.position(newPos);
        }
        if (this.bytesNeededToFinish <= 0L) {
            this.readTrailingEmptyPacket();
            if (this.verifyChecksum) {
                this.sendReadResult(DataTransferProtos.Status.CHECKSUM_OK);
            } else {
                this.sendReadResult(DataTransferProtos.Status.SUCCESS);
            }
        }
    }

    @Override
    public synchronized long skip(long n) throws IOException {
        long skipped;
        int skip;
        for (skipped = 0L; skipped < n; skipped += (long)skip) {
            long needToSkip = n - skipped;
            if (this.curDataSlice == null || this.curDataSlice.remaining() == 0 && this.bytesNeededToFinish > 0L) {
                this.readNextPacket();
            }
            if (this.curDataSlice.remaining() == 0) break;
            skip = (int)Math.min((long)this.curDataSlice.remaining(), needToSkip);
            this.curDataSlice.position(this.curDataSlice.position() + skip);
        }
        return skipped;
    }

    private void readTrailingEmptyPacket() throws IOException {
        LOG.trace("Reading empty packet at end of read");
        this.packetReceiver.receiveNextPacket(this.in);
        PacketHeader trailer = this.packetReceiver.getHeader();
        if (!trailer.isLastPacketInBlock() || trailer.getDataLen() != 0) {
            throw new IOException("Expected empty end-of-read packet! Header: " + trailer);
        }
    }

    protected BlockReaderRemote(String file, long blockId, DataChecksum checksum, boolean verifyChecksum, long startOffset, long firstChunkOffset, long bytesToRead, Peer peer, DatanodeID datanodeID, PeerCache peerCache, int networkDistance) {
        this.peer = peer;
        this.datanodeID = datanodeID;
        this.in = peer.getInputStreamChannel();
        this.checksum = checksum;
        this.verifyChecksum = verifyChecksum;
        this.startOffset = Math.max(startOffset, 0L);
        this.filename = file;
        this.peerCache = peerCache;
        this.blockId = blockId;
        this.bytesNeededToFinish = bytesToRead + (startOffset - firstChunkOffset);
        this.bytesPerChecksum = this.checksum.getBytesPerChecksum();
        this.checksumSize = this.checksum.getChecksumSize();
        this.networkDistance = networkDistance;
    }

    @Override
    public synchronized void close() throws IOException {
        this.packetReceiver.close();
        this.startOffset = -1L;
        this.checksum = null;
        if (this.peerCache != null && this.sentStatusCode) {
            this.peerCache.put(this.datanodeID, this.peer);
        } else {
            this.peer.close();
        }
    }

    void sendReadResult(DataTransferProtos.Status statusCode) {
        assert (!this.sentStatusCode) : "already sent status code to " + this.peer;
        try {
            BlockReaderRemote.writeReadResult(this.peer.getOutputStream(), statusCode);
            this.sentStatusCode = true;
        }
        catch (IOException e) {
            LOG.info("Could not send read status (" + (Object)((Object)statusCode) + ") to datanode " + this.peer.getRemoteAddressString() + ": " + e.getMessage());
        }
    }

    static void writeReadResult(OutputStream out, DataTransferProtos.Status statusCode) throws IOException {
        DataTransferProtos.ClientReadStatusProto.newBuilder().setStatus(statusCode).build().writeDelimitedTo(out);
        out.flush();
    }

    public static String getFileName(InetSocketAddress s, String poolId, long blockId) {
        return s.toString() + ":" + poolId + ":" + blockId;
    }

    @Override
    public int readAll(byte[] buf, int offset, int len) throws IOException {
        return BlockReaderUtil.readAll(this, buf, offset, len);
    }

    @Override
    public void readFully(byte[] buf, int off, int len) throws IOException {
        BlockReaderUtil.readFully(this, buf, off, len);
    }

    public static BlockReader newBlockReader(String file, ExtendedBlock block, Token<BlockTokenIdentifier> blockToken, long startOffset, long len, boolean verifyChecksum, String clientName, Peer peer, DatanodeID datanodeID, PeerCache peerCache, CachingStrategy cachingStrategy, int networkDistance) throws IOException {
        DataOutputStream out = new DataOutputStream(new BufferedOutputStream(peer.getOutputStream()));
        new Sender(out).readBlock(block, blockToken, clientName, startOffset, len, verifyChecksum, cachingStrategy);
        DataInputStream in = new DataInputStream(peer.getInputStream());
        DataTransferProtos.BlockOpResponseProto status = DataTransferProtos.BlockOpResponseProto.parseFrom(PBHelperClient.vintPrefixed(in));
        BlockReaderRemote.checkSuccess(status, peer, block, file);
        DataTransferProtos.ReadOpChecksumInfoProto checksumInfo = status.getReadOpChecksumInfo();
        DataChecksum checksum = DataTransferProtoUtil.fromProto(checksumInfo.getChecksum());
        long firstChunkOffset = checksumInfo.getChunkOffset();
        if (firstChunkOffset < 0L || firstChunkOffset > startOffset || firstChunkOffset <= startOffset - (long)checksum.getBytesPerChecksum()) {
            throw new IOException("BlockReader: error in first chunk offset (" + firstChunkOffset + ") startOffset is " + startOffset + " for file " + file);
        }
        return new BlockReaderRemote(file, block.getBlockId(), checksum, verifyChecksum, startOffset, firstChunkOffset, len, peer, datanodeID, peerCache, networkDistance);
    }

    static void checkSuccess(DataTransferProtos.BlockOpResponseProto status, Peer peer, ExtendedBlock block, String file) throws IOException {
        String logInfo = "for OP_READ_BLOCK, self=" + peer.getLocalAddressString() + ", remote=" + peer.getRemoteAddressString() + ", for file " + file + ", for pool " + block.getBlockPoolId() + " block " + block.getBlockId() + "_" + block.getGenerationStamp();
        DataTransferProtoUtil.checkBlockOpStatus(status, logInfo);
    }

    @Override
    public int available() {
        return 131072;
    }

    @Override
    public boolean isShortCircuit() {
        return false;
    }

    @Override
    public ClientMmap getClientMmap(EnumSet<ReadOption> opts) {
        return null;
    }

    @Override
    public DataChecksum getDataChecksum() {
        return this.checksum;
    }

    @Override
    public int getNetworkDistance() {
        return this.networkDistance;
    }
}

