/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sis.referencing.operation.matrix;

import java.util.Arrays;
import java.util.Objects;
import org.apache.sis.math.DecimalFunctions;
import org.apache.sis.math.MathFunctions;
import org.apache.sis.referencing.internal.Resources;
import org.apache.sis.referencing.operation.matrix.GeneralMatrix;
import org.apache.sis.referencing.operation.matrix.Matrix1;
import org.apache.sis.referencing.operation.matrix.Matrix2;
import org.apache.sis.referencing.operation.matrix.Matrix3;
import org.apache.sis.referencing.operation.matrix.Matrix4;
import org.apache.sis.referencing.operation.matrix.MatrixSIS;
import org.apache.sis.referencing.operation.matrix.MismatchedMatrixSizeException;
import org.apache.sis.referencing.operation.matrix.NonSquareMatrix;
import org.apache.sis.referencing.operation.matrix.NoninvertibleMatrixException;
import org.apache.sis.referencing.operation.matrix.Solver;
import org.apache.sis.referencing.operation.matrix.UnmodifiableMatrix;
import org.apache.sis.referencing.util.AxisDirections;
import org.apache.sis.referencing.util.ExtendedPrecisionMatrix;
import org.apache.sis.util.ArgumentChecks;
import org.apache.sis.util.CharSequences;
import org.apache.sis.util.ComparisonMode;
import org.apache.sis.util.Static;
import org.apache.sis.util.StringBuilders;
import org.apache.sis.util.internal.DoubleDouble;
import org.apache.sis.util.internal.Numerics;
import org.apache.sis.util.resources.Errors;
import org.opengis.geometry.DirectPosition;
import org.opengis.geometry.Envelope;
import org.opengis.referencing.cs.AxisDirection;
import org.opengis.referencing.operation.Matrix;

public final class Matrices
extends Static {
    private static final int SPACING = 2;

    private Matrices() {
    }

    public static MatrixSIS createIdentity(int size) {
        switch (size) {
            case 1: {
                return new Matrix1();
            }
            case 2: {
                return new Matrix2();
            }
            case 3: {
                return new Matrix3();
            }
            case 4: {
                return new Matrix4();
            }
        }
        return new GeneralMatrix(size, size, true);
    }

    public static MatrixSIS createDiagonal(int numRow, int numCol) {
        if (numRow == numCol) {
            return Matrices.createIdentity(numRow);
        }
        return new NonSquareMatrix(numRow, numCol, true);
    }

    public static MatrixSIS createZero(int numRow, int numCol) {
        if (numRow == numCol) {
            switch (numRow) {
                case 1: {
                    return new Matrix1(false);
                }
                case 2: {
                    return new Matrix2(false);
                }
                case 3: {
                    return new Matrix3(false);
                }
                case 4: {
                    return new Matrix4(false);
                }
            }
            return new GeneralMatrix(numRow, numCol, false);
        }
        return new NonSquareMatrix(numRow, numCol, false);
    }

    static MatrixSIS createZero(int numRow, int numCol, Matrix source) {
        return Matrices.isExtendedPrecision(source) ? GeneralMatrix.create(numRow, numCol, false) : Matrices.createZero(numRow, numCol);
    }

    public static MatrixSIS create(int numRow, int numCol, double[] elements) {
        GeneralMatrix matrix;
        if (numRow == numCol) {
            switch (numRow) {
                case 1: {
                    return new Matrix1(elements);
                }
                case 2: {
                    return new Matrix2(elements);
                }
                case 3: {
                    return new Matrix3(elements);
                }
                case 4: {
                    return new Matrix4(elements);
                }
            }
            matrix = new GeneralMatrix(numRow, numCol, false);
        } else {
            matrix = new NonSquareMatrix(numRow, numCol, false);
        }
        matrix.setElements(elements);
        return matrix;
    }

    public static MatrixSIS create(int numRow, int numCol, Number[] elements) {
        boolean setToIdentity;
        boolean bl = setToIdentity = elements == ExtendedPrecisionMatrix.CREATE_IDENTITY;
        if (setToIdentity || elements == ExtendedPrecisionMatrix.CREATE_ZERO) {
            return GeneralMatrix.create(numRow, numCol, setToIdentity);
        }
        int expected = numRow * numCol;
        if (elements.length != expected) {
            throw new IllegalArgumentException(Errors.format((short)133, (Object)expected, (Object)elements.length));
        }
        if (numRow == numCol) {
            return new GeneralMatrix(numRow, numCol, elements);
        }
        return new NonSquareMatrix(numRow, numCol, elements);
    }

    private static MatrixSIS createTransform(Envelope srcEnvelope, AxisDirection[] srcAxes, Envelope dstEnvelope, AxisDirection[] dstAxes, boolean useEnvelopes) {
        DirectPosition srcOppositeCorner;
        DirectPosition srcCorner;
        DirectPosition dstCorner;
        if (useEnvelopes) {
            dstCorner = dstEnvelope.getLowerCorner();
            srcCorner = srcEnvelope.getLowerCorner();
            srcOppositeCorner = srcEnvelope.getUpperCorner();
        } else {
            srcOppositeCorner = null;
            srcCorner = null;
            dstCorner = null;
        }
        GeneralMatrix matrix = GeneralMatrix.create(dstAxes.length + 1, srcAxes.length + 1, false);
        for (int dstIndex = 0; dstIndex < dstAxes.length; ++dstIndex) {
            boolean hasFound = false;
            AxisDirection dstDir = dstAxes[dstIndex];
            AxisDirection search = AxisDirections.absolute(dstDir);
            for (int srcIndex = 0; srcIndex < srcAxes.length; ++srcIndex) {
                AxisDirection srcDir = srcAxes[srcIndex];
                if (!search.equals((Object)AxisDirections.absolute(srcDir))) continue;
                if (hasFound) {
                    throw new IllegalArgumentException(Resources.format((short)10, srcDir, dstDir));
                }
                hasFound = true;
                boolean same = srcDir.equals((Object)dstDir);
                if (useEnvelopes) {
                    boolean decimal = true;
                    DoubleDouble scale = DoubleDouble.of((double)dstEnvelope.getSpan(dstIndex), (boolean)true).divide(srcEnvelope.getSpan(srcIndex), true);
                    if (!same) {
                        scale = scale.negate();
                    }
                    DoubleDouble translate = scale.multiply((same ? srcCorner : srcOppositeCorner).getOrdinate(srcIndex), true);
                    translate = DoubleDouble.of((double)dstCorner.getOrdinate(dstIndex), (boolean)true).subtract(translate);
                    ((MatrixSIS)matrix).setNumber(dstIndex, srcIndex, (Number)scale);
                    ((MatrixSIS)matrix).setNumber(dstIndex, srcAxes.length, (Number)translate);
                    continue;
                }
                matrix.setElement(dstIndex, srcIndex, same ? 1.0 : -1.0);
            }
            if (hasFound) continue;
            throw new IllegalArgumentException(Resources.format((short)6, dstAxes[dstIndex]));
        }
        matrix.setElement(dstAxes.length, srcAxes.length, 1.0);
        return matrix;
    }

    public static MatrixSIS createTransform(Envelope srcEnvelope, Envelope dstEnvelope) {
        ArgumentChecks.ensureNonNull((String)"srcEnvelope", (Object)srcEnvelope);
        ArgumentChecks.ensureNonNull((String)"dstEnvelope", (Object)dstEnvelope);
        int srcDim = srcEnvelope.getDimension();
        int dstDim = dstEnvelope.getDimension();
        DirectPosition srcCorner = srcEnvelope.getLowerCorner();
        DirectPosition dstCorner = dstEnvelope.getLowerCorner();
        MatrixSIS matrix = Matrices.createZero(dstDim + 1, srcDim + 1);
        int i = Math.min(srcDim, dstDim);
        while (--i >= 0) {
            double scale = dstEnvelope.getSpan(i) / srcEnvelope.getSpan(i);
            double translate = dstCorner.getOrdinate(i) - srcCorner.getOrdinate(i) * scale;
            matrix.setElement(i, i, scale);
            matrix.setElement(i, srcDim, translate);
        }
        matrix.setElement(dstDim, srcDim, 1.0);
        return matrix;
    }

    public static MatrixSIS createTransform(AxisDirection[] srcAxes, AxisDirection[] dstAxes) {
        ArgumentChecks.ensureNonNull((String)"srcAxes", (Object)srcAxes);
        ArgumentChecks.ensureNonNull((String)"dstAxes", (Object)dstAxes);
        if (Arrays.equals(srcAxes, dstAxes)) {
            int n = srcAxes.length + 1;
            return new GeneralMatrix(n, n, true);
        }
        return Matrices.createTransform(null, srcAxes, null, dstAxes, false);
    }

    public static MatrixSIS createTransform(Envelope srcEnvelope, AxisDirection[] srcAxes, Envelope dstEnvelope, AxisDirection[] dstAxes) {
        ArgumentChecks.ensureNonNull((String)"srcEnvelope", (Object)srcEnvelope);
        ArgumentChecks.ensureNonNull((String)"dstEnvelope", (Object)dstEnvelope);
        ArgumentChecks.ensureDimensionMatches((String)"srcEnvelope", (int)srcAxes.length, (Envelope)srcEnvelope);
        ArgumentChecks.ensureDimensionMatches((String)"dstEnvelope", (int)dstAxes.length, (Envelope)dstEnvelope);
        return Matrices.createTransform(srcEnvelope, srcAxes, dstEnvelope, dstAxes, true);
    }

    public static MatrixSIS createDimensionSelect(int sourceDimensions, int[] selectedDimensions) {
        int numTargetDim = selectedDimensions.length;
        MatrixSIS matrix = Matrices.createZero(numTargetDim + 1, sourceDimensions + 1);
        for (int j = 0; j < numTargetDim; ++j) {
            int i = selectedDimensions[j];
            ArgumentChecks.ensureValidIndex((int)sourceDimensions, (int)i);
            matrix.setElement(j, i, 1.0);
        }
        matrix.setElement(numTargetDim, sourceDimensions, 1.0);
        return matrix;
    }

    public static MatrixSIS createPassThrough(int firstAffectedCoordinate, Matrix subMatrix, int numTrailingCoordinates) {
        ArgumentChecks.ensureNonNull((String)"subMatrix", (Object)subMatrix);
        ArgumentChecks.ensurePositive((String)"firstAffectedCoordinate", (int)firstAffectedCoordinate);
        ArgumentChecks.ensurePositive((String)"numTrailingCoordinates", (int)numTrailingCoordinates);
        int expansion = firstAffectedCoordinate + numTrailingCoordinates;
        int sourceDimensions = subMatrix.getNumCol();
        int targetDimensions = subMatrix.getNumRow();
        MatrixSIS matrix = Matrices.createZero(targetDimensions-- + expansion, sourceDimensions-- + expansion, subMatrix);
        for (int j = 0; j < firstAffectedCoordinate; ++j) {
            matrix.setElement(j, j, 1.0);
        }
        int lastColumn = sourceDimensions + expansion;
        matrix.setElements(subMatrix, 0, 0, firstAffectedCoordinate, firstAffectedCoordinate, targetDimensions, sourceDimensions);
        matrix.setElements(subMatrix, 0, sourceDimensions, firstAffectedCoordinate, lastColumn, targetDimensions, 1);
        int diff = targetDimensions - sourceDimensions;
        for (int i = lastColumn - numTrailingCoordinates; i < lastColumn; ++i) {
            matrix.setElement(diff + i, i, 1.0);
        }
        int lastRow = targetDimensions + expansion;
        matrix.setElements(subMatrix, targetDimensions, 0, lastRow, firstAffectedCoordinate, 1, sourceDimensions);
        matrix.setElements(subMatrix, targetDimensions, sourceDimensions, lastRow, lastColumn, 1, 1);
        return matrix;
    }

    public static MatrixSIS createAffine(Matrix derivative, DirectPosition translation) {
        int j;
        MatrixSIS matrix;
        int numCol;
        int numRow;
        if (derivative != null) {
            numRow = derivative.getNumRow();
            numCol = derivative.getNumCol();
            if (translation != null) {
                MatrixSIS.ensureNumRowMatch(translation.getDimension(), numRow, numCol);
            }
            matrix = Matrices.createZero(numRow + 1, numCol + 1);
            matrix.setElement(numRow, numCol, 1.0);
            for (j = 0; j < numRow; ++j) {
                for (int i = 0; i < numCol; ++i) {
                    matrix.setElement(j, i, derivative.getElement(j, i));
                }
            }
        } else {
            ArgumentChecks.ensureNonNull((String)"derivative", (Object)translation);
            numRow = numCol = translation.getDimension();
            matrix = Matrices.createIdentity(numRow + 1);
        }
        if (translation != null) {
            for (j = 0; j < numRow; ++j) {
                matrix.setElement(j, numCol, translation.getOrdinate(j));
            }
        }
        return matrix;
    }

    public static Matrix resizeAffine(Matrix matrix, int numRow, int numCol) {
        ArgumentChecks.ensureNonNull((String)"matrix", (Object)matrix);
        ArgumentChecks.ensureStrictlyPositive((String)"numRow", (int)numRow);
        ArgumentChecks.ensureStrictlyPositive((String)"numCol", (int)numCol);
        int srcRow = matrix.getNumRow();
        int srcCol = matrix.getNumCol();
        if (numRow == srcRow && numCol == srcCol) {
            return matrix;
        }
        MatrixSIS resized = Matrices.createZero(numRow, numCol, matrix);
        int copyRow = Math.min(--numRow, --srcRow);
        int copyCol = Math.min(--numCol, --srcCol);
        for (int j = copyRow; j < numRow; ++j) {
            resized.setElement(j, j, 1.0);
        }
        resized.setElements(matrix, 0, 0, 0, 0, copyRow, copyCol);
        resized.setElements(matrix, 0, srcCol, 0, numCol, copyRow, 1);
        resized.setElements(matrix, srcRow, 0, numRow, 0, 1, copyCol);
        resized.setElements(matrix, srcRow, srcCol, numRow, numCol, 1, 1);
        return resized;
    }

    public static boolean forceUniformScale(Matrix matrix, double selector, double[] anchor) {
        ArgumentChecks.ensureNonNull((String)"matrix", (Object)matrix);
        ArgumentChecks.ensureFinite((String)"selector", (double)selector);
        int srcDim = matrix.getNumCol() - 1;
        int tgtDim = matrix.getNumRow() - 1;
        ArgumentChecks.ensureDimensionMatches((String)"anchor", (int)tgtDim, (double[])anchor);
        double[] row = new double[srcDim];
        double[] mgn = new double[tgtDim];
        double min = Double.POSITIVE_INFINITY;
        double max = Double.NEGATIVE_INFINITY;
        for (int j = 0; j < tgtDim; ++j) {
            for (int i = 0; i < srcDim; ++i) {
                row[i] = matrix.getElement(j, i);
            }
            double m = MathFunctions.magnitude((double[])row);
            if (m < min) {
                min = m;
            }
            if (m > max) {
                max = m;
            }
            mgn[j] = m;
        }
        boolean changed = false;
        if (min < max) {
            double scale = (1.0 - selector) * min + selector * max;
            for (int j = 0; j < tgtDim; ++j) {
                double rescale = scale / mgn[j];
                for (int i = 0; i < srcDim; ++i) {
                    double e = matrix.getElement(j, i);
                    changed |= e != (e *= rescale);
                    matrix.setElement(j, i, e);
                }
                if (anchor == null) continue;
                double p = anchor[j];
                double e = matrix.getElement(j, srcDim);
                changed |= e != (e = Math.fma(rescale, e - p, p));
                matrix.setElement(j, srcDim, e);
            }
        }
        return changed;
    }

    private static boolean isExtendedPrecision(Matrix matrix) {
        if (matrix instanceof UnmodifiableMatrix) {
            matrix = ((UnmodifiableMatrix)matrix).matrix;
        }
        return matrix instanceof ExtendedPrecisionMatrix;
    }

    public static MatrixSIS copy(Matrix matrix) {
        GeneralMatrix copy;
        int numCol;
        if (matrix == null) {
            return null;
        }
        int numRow = matrix.getNumRow();
        if (numRow == (numCol = matrix.getNumCol())) {
            if (!Matrices.isExtendedPrecision(matrix)) {
                switch (numRow) {
                    case 1: {
                        return new Matrix1(matrix);
                    }
                    case 2: {
                        return new Matrix2(matrix);
                    }
                    case 3: {
                        return new Matrix3(matrix);
                    }
                    case 4: {
                        return new Matrix4(matrix);
                    }
                }
            }
            copy = new GeneralMatrix(numRow, numCol, false);
        } else {
            copy = new NonSquareMatrix(numRow, numCol, false);
        }
        copy.setMatrix(matrix);
        return copy;
    }

    public static MatrixSIS unmodifiable(Matrix matrix) {
        if (matrix == null || matrix instanceof UnmodifiableMatrix) {
            return (MatrixSIS)matrix;
        }
        return new UnmodifiableMatrix(matrix);
    }

    public static MatrixSIS multiply(Matrix m1, Matrix m2) throws MismatchedMatrixSizeException {
        if (m1 instanceof MatrixSIS) {
            return ((MatrixSIS)m1).multiply(m2);
        }
        int nc = m2.getNumCol();
        MatrixSIS.ensureNumRowMatch(m1.getNumCol(), m2.getNumRow(), nc);
        GeneralMatrix result = GeneralMatrix.create(m1.getNumRow(), nc, false);
        result.setToProduct(m1, m2);
        return result;
    }

    public static MatrixSIS inverse(Matrix matrix) throws NoninvertibleMatrixException {
        int numCol;
        if (matrix == null) {
            return null;
        }
        if (matrix instanceof MatrixSIS) {
            return ((MatrixSIS)matrix).inverse();
        }
        int numRow = matrix.getNumRow();
        if (numRow == (numCol = matrix.getNumCol())) {
            return Solver.inverse(matrix);
        }
        NonSquareMatrix result = new NonSquareMatrix(numRow, numCol, false);
        result.setMatrix(matrix);
        return result.inverse();
    }

    public static boolean isAffine(Matrix matrix) {
        if (matrix instanceof MatrixSIS) {
            return ((MatrixSIS)matrix).isAffine();
        }
        return MatrixSIS.isAffine(matrix);
    }

    public static boolean isTranslation(Matrix matrix) {
        if (!Matrices.isAffine(matrix)) {
            return false;
        }
        int numRow = matrix.getNumRow() - 1;
        int numCol = matrix.getNumCol() - 1;
        for (int j = 0; j < numRow; ++j) {
            for (int i = 0; i < numCol; ++i) {
                if (matrix.getElement(j, i) == (double)(i == j ? 1 : 0)) continue;
                return false;
            }
        }
        return true;
    }

    public static boolean isIdentity(Matrix matrix, double tolerance) {
        int numCol;
        int numRow = matrix.getNumRow();
        if (numRow != (numCol = matrix.getNumCol())) {
            return false;
        }
        for (int j = 0; j < numRow; ++j) {
            for (int i = 0; i < numCol; ++i) {
                double e = matrix.getElement(j, i);
                if (i == j) {
                    e -= 1.0;
                }
                if (Math.abs(e) <= tolerance) continue;
                return false;
            }
        }
        return true;
    }

    public static boolean equals(Matrix m1, Matrix m2, double epsilon, boolean relative) {
        if (m1 != m2) {
            if (m1 == null || m2 == null) {
                return false;
            }
            int numRow = m1.getNumRow();
            if (numRow != m2.getNumRow()) {
                return false;
            }
            int numCol = m1.getNumCol();
            if (numCol != m2.getNumCol()) {
                return false;
            }
            for (int j = 0; j < numRow; ++j) {
                for (int i = 0; i < numCol; ++i) {
                    double f;
                    double v1 = m1.getElement(j, i);
                    double v2 = m2.getElement(j, i);
                    double tolerance = epsilon;
                    if (relative && (f = Math.max(Math.abs(v1), Math.abs(v2))) <= Double.MAX_VALUE) {
                        tolerance *= f;
                    }
                    if (Math.abs(v1 - v2) <= tolerance || Numerics.equals((double)v1, (double)v2)) continue;
                    return false;
                }
            }
        }
        return true;
    }

    public static boolean equals(Matrix m1, Matrix m2, ComparisonMode mode) {
        switch (mode) {
            case STRICT: {
                return Objects.equals(m1, m2);
            }
            case BY_CONTRACT: 
            case IGNORE_METADATA: {
                return Matrices.equals(m1, m2, 0.0, false);
            }
            case DEBUG: 
            case ALLOW_VARIANT: 
            case APPROXIMATE: {
                return Matrices.equals(m1, m2, 1.0E-13, true);
            }
        }
        throw new IllegalArgumentException(Errors.format((short)146, ComparisonMode.class, (Object)mode));
    }

    public static String toString(Matrix matrix) {
        int numRow = matrix.getNumRow();
        int numCol = matrix.getNumCol();
        String[] elements = new String[numCol * numRow];
        boolean[] noFractionDigits = new boolean[numCol * numRow];
        boolean[] hasDecimalSeparator = new boolean[numCol];
        byte[] maximumFractionDigits = new byte[numCol];
        byte[] maximumPaddingZeros = new byte[numCol * numRow];
        byte[] widthBeforeFraction = new byte[numCol];
        byte[] columnWidth = new byte[numCol];
        int totalWidth = 1;
        int spacing = 1;
        for (int i = 0; i < numCol; ++i) {
            for (int j = 0; j < numRow; ++j) {
                int width;
                int flatIndex = j * numCol + i;
                double value = matrix.getElement(j, i);
                String element = Double.toString(value);
                if (value == -1.0 || value == 0.0 || value == 1.0) {
                    noFractionDigits[flatIndex] = true;
                    width = spacing + element.length() - 2;
                    widthBeforeFraction[i] = (byte)Math.max(widthBeforeFraction[i], width);
                } else {
                    int s = element.lastIndexOf(46);
                    if (s < 0) {
                        element = CharSequences.replace((CharSequence)element, (CharSequence)"Infinity", (CharSequence)"\u221e").toString();
                        width = spacing + element.length();
                        widthBeforeFraction[i] = (byte)Math.max(widthBeforeFraction[i], width);
                    } else {
                        hasDecimalSeparator[i] = true;
                        int numFractionDigits = element.length() - ++s;
                        widthBeforeFraction[i] = (byte)Math.max(widthBeforeFraction[i], spacing + s);
                        maximumFractionDigits[i] = (byte)Math.max(maximumFractionDigits[i], numFractionDigits);
                        width = widthBeforeFraction[i] + maximumFractionDigits[i];
                        if (element.indexOf(69) < 0) {
                            int accuracy = -DecimalFunctions.floorLog10((double)Math.ulp(value));
                            maximumPaddingZeros[flatIndex] = (byte)(accuracy - numFractionDigits);
                        }
                    }
                }
                columnWidth[i] = (byte)Math.max(columnWidth[i], width);
                elements[flatIndex] = element;
            }
            totalWidth += columnWidth[i];
            spacing = 2;
        }
        String lineSeparator = System.lineSeparator();
        CharSequence whiteLine = CharSequences.spaces((int)totalWidth);
        StringBuilder buffer = new StringBuilder((totalWidth + 2 + lineSeparator.length()) * (numRow + 2));
        buffer.append('\u250c').append(whiteLine).append('\u2510').append(lineSeparator);
        int flatIndex = 0;
        for (int j = 0; j < numRow; ++j) {
            buffer.append('\u2502');
            for (int i = 0; i < numCol; ++i) {
                int spaces;
                String element = elements[flatIndex];
                int width = element.length();
                int s = element.lastIndexOf(46);
                if (s >= 0) {
                    if (hasDecimalSeparator[i]) {
                        ++s;
                    }
                    spaces = widthBeforeFraction[i] - s;
                } else {
                    spaces = columnWidth[i] - width;
                }
                buffer.append(CharSequences.spaces((int)spaces)).append(element);
                if (s >= 0) {
                    s += maximumFractionDigits[i] - width;
                    if (noFractionDigits[flatIndex]) {
                        buffer.setLength(buffer.length() - 2);
                        s += 2;
                    } else {
                        int n = Math.min(s, maximumPaddingZeros[flatIndex]);
                        StringBuilders.repeat((StringBuilder)buffer, (char)'0', (int)n);
                        s -= n;
                    }
                    buffer.append(CharSequences.spaces((int)s));
                }
                ++flatIndex;
            }
            buffer.append(" \u2502").append(lineSeparator);
        }
        return buffer.append('\u2514').append(whiteLine).append('\u2518').append(lineSeparator).toString();
    }
}

