/**
 * @license
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

const path = require('path');

const babel = require('@babel/core');

// Use the root babel.config.js for module resolution.
// Relevant issue: tleunen/eslint-import-resolver-babel-module#89
const babelOptions = babel.loadOptions({ cwd: __dirname });
const babelModuleResolver = babelOptions.plugins.find(
  ({ key }) => key === 'module-resolver',
);

module.exports = {
  root: true,
  env: {
    es6: true,
    'shared-node-browser': true,
  },
  extends: ['eslint:recommended', 'plugin:import/recommended', 'prettier'],
  globals: {
    globalThis: true,
  },
  parser: 'babel-eslint',
  parserOptions: {
    ecmaVersion: 2020,
    sourceType: 'module',
  },
  plugins: ['import', 'prettier'],
  rules: {
    'import/extensions': ['error', 'always', { ignorePackages: true }],
    'import/first': 'error',
    'import/newline-after-import': 'error',
    'import/no-default-export': 'error',
    'import/no-internal-modules': 'error',
    'import/no-relative-parent-imports': 'error',
    'import/order': ['error', { 'newlines-between': 'always' }],
    'import/unambiguous': 'error',
    'no-restricted-syntax': [
      'error',
      'BindExpression',
      'ClassProperty',
      'Decorator',
      'DoExpression',
      'ExportDefaultSpecifier',
      'ExportNamespaceSpecifier',
      'TypeAnnotation',
      'JSXElement',
    ],
    'prettier/prettier': [
      'error',
      {
        singleQuote: true,
        trailingComma: 'all',
      },
    ],
  },
  settings: {
    'import/resolver': {
      'babel-module': babelModuleResolver.options,
    },
  },
  overrides: [
    {
      files: [
        '.eslintrc.js',
        '.mocharc.js',
        'babel.config.js',
        'husky.config.js',
        'nyc.config.js',
      ],
      env: {
        node: true,
      },
      parser: 'espree',
      parserOptions: {
        sourceType: 'script',
      },
      plugins: ['node'],
      rules: {
        'no-console': 'off',
        'node/no-unsupported-features': 'error',
      },
    },
    {
      files: ['web/demo/**/*.js'],
      env: {
        browser: true,
      },
    },
    {
      files: ['packages/*/test/**/*.js', 'test/**/*.js'],
      env: {
        mocha: true,
      },
      globals: {
        assert: true,
      },
      rules: {
        'import/no-internal-modules': [
          'error',
          {
            allow: [
              'ajv/lib/refs/json-schema-draft-04.json',
              path.resolve(__dirname, './packages/*/src/**'),
            ],
          },
        ],
        'import/no-relative-parent-imports': 'off',
      },
    },
    {
      files: ['packages/dom/{src,test}/**/*.js'],
      env: {
        browser: true,
      },
    },
  ],
};
