/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.sql.engine.sql;

import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.HashSet;
import java.util.Set;
import java.util.TreeSet;
import org.apache.calcite.sql.parser.SqlAbstractParserImpl;
import org.apache.ignite.internal.generated.query.calcite.sql.IgniteSqlParserImpl;
import org.apache.ignite.internal.lang.IgniteStringBuilder;
import org.junit.jupiter.api.Test;

/**
 * Test reserved keywords.
 */
public class SqlReservedWordsTest extends AbstractParserTest {
    /** List of keywords reserved in Ignite SQL. */
    private static final Set<String> RESERVED_WORDS = Set.of(
            "ABS",
            "ALL", // UNION ALL
            "ALTER",
            "AND",
            "ANY",
            "ARRAY",
            "ARRAY_MAX_CARDINALITY",
            "AS",
            "ASOF",
            "ASYMMETRIC", // BETWEEN ASYMMETRIC .. AND ..
            "AVG",
            "BETWEEN",
            "BOTH", // TRIM(BOTH .. FROM ..)
            "BY", // GROUP BY
            "CALL",
            "CARDINALITY",
            "CASE",
            "CAST",
            "CEILING",
            "CHAR",
            "CHARACTER",
            "CHARACTER_LENGTH",
            "CHAR_LENGTH",
            "COALESCE",
            "COLLECT",
            "COLUMN",
            "CONSTRAINT",
            "CONVERT",
            "COUNT",
            "COVAR_POP",
            "COVAR_SAMP",
            "CREATE",
            "CROSS", // CROSS JOIN
            "CUBE",
            "CUME_DIST",
            "CURRENT",
            "CURRENT_CATALOG",
            "CURRENT_DATE",
            "CURRENT_DEFAULT_TRANSFORM_GROUP",
            "CURRENT_PATH",
            "CURRENT_ROLE",
            "CURRENT_ROW",
            "CURRENT_SCHEMA",
            "CURRENT_TIME",
            "CURRENT_TIMESTAMP",
            "CURRENT_TRANSFORM_GROUP_FOR_TYPE",
            "CURRENT_USER",
            "DATE",
            "DATETIME",
            "DECIMAL",
            "DEFAULT",
            "DELETE",
            "DENSE_RANK",
            "DESCRIBE",
            "DISTINCT",
            "DROP",
            "ELEMENT",
            "ELSE",
            "EVERY",
            "EXCEPT",
            "EXISTS",
            "EXP",
            "EXPLAIN",
            "EXTEND",
            "EXTRACT",
            "FALSE",
            "FETCH",
            "FILTER",
            "FIRST_VALUE",
            "FLOOR",
            "FOR", // SUBSTRING(.. FROM .. FOR ..)
            "FRIDAY",
            "FROM",
            "FULL", // FULL JOIN
            "FUSION",
            "GROUP",
            "GROUPING",
            "HAVING",
            "HOUR",
            "IF",
            "IN",
            "INDEX",
            "INNER",
            "INSERT",
            "INTERSECT",
            "INTERSECTION",
            "INTERVAL",
            "INTO",
            "IS",
            "JOIN",
            "JSON_SCOPE",
            "LAG",
            "LAST_VALUE",
            "LEAD",
            "LEADING", // TRIM(LEADING .. FROM ..)
            "LEFT", // LEFT JOIN
            "LIKE",
            "LIMIT",
            "LN",
            "LOCALTIME",
            "LOCALTIMESTAMP",
            "LOWER",
            "MATCH_CONDITION",
            "MATCH_RECOGNIZE",
            "MAX",
            "MEASURE",
            "MERGE",
            "MIN",
            "MINUS",
            "MINUTE",
            "MOD",
            "MONDAY",
            "MONTH",
            "MULTISET",
            "NATURAL", // NATURAL JOIN
            "NEW",
            "NEXT",
            "NOT",
            "NTH_VALUE",
            "NTILE",
            "NULL",
            "NULLIF",
            "OCTET_LENGTH",
            "OFFSET",
            "ON",
            "OR",
            "ORDER",
            "OUTER", // OUTER JOIN
            "OVER",
            "PARTITION",
            "PERCENTILE_CONT",
            "PERCENTILE_DISC",
            "PERCENT_RANK",
            "PERIOD",
            "PERMUTE",
            "POWER",
            "PRECISION",
            "PRIMARY",
            "QUALIFY",
            "RANK",
            "REGR_COUNT",
            "REGR_SXX",
            "REGR_SYY",
            "RENAME",
            "RESET",
            "RIGHT",
            "ROLLUP",
            "ROW",
            "ROW_NUMBER",
            "SATURDAY",
            "SECOND",
            "SELECT",
            "SESSION_USER",
            "SET",
            "SOME",
            "SPECIFIC",
            "SQRT",
            "STDDEV_POP",
            "STDDEV_SAMP",
            "STREAM",
            "SUBSTRING",
            "SUM",
            "SUNDAY",
            "SYMMETRIC", // BETWEEN SYMMETRIC .. AND ..
            "SYSTEM_TIME",
            "SYSTEM_USER",
            "TABLE",
            "TABLESAMPLE",
            "THEN",
            "THURSDAY",
            "TIME",
            "TIMESTAMP",
            "TO",
            "TRAILING", // TRIM(TRAILING .. FROM ..)
            "TRUE",
            "TRUNCATE",
            "TUESDAY",
            "UESCAPE",
            "UNION",
            "UNKNOWN",
            "UPDATE",
            "UPPER",
            "UPSERT",
            "USER",
            "USING",
            "UUID",
            "VALUE",
            "VALUES",
            "VARIANT",
            "VAR_POP",
            "VAR_SAMP",
            "WEDNESDAY",
            "WHEN",
            "WHERE",
            "WINDOW",
            "WITH",
            "WITHIN",
            "YEAR"
    );

    private static final Set<String> NON_RESERVED_WORDS = Set.of(
            "A",
            "ABSENT",
            "ABSOLUTE",
            "ACTION",
            "ADA",
            "ADD",
            "ADMIN",
            "AFTER",
            "ALGORITHM",
            "ALLOCATE",
            "ALLOW",
            "ALWAYS",
            "APPLY",
            "ARE",
            "ARRAY_AGG",
            "ARRAY_CONCAT_AGG",
            "ASC",
            "ASENSITIVE",
            "ASSERTION",
            "ASSIGNMENT",
            "AT",
            "ATOMIC",
            "ATTRIBUTE",
            "ATTRIBUTES",
            "AUTHORIZATION",
            "AUTO",
            "BEFORE",
            "BEGIN",
            "BEGIN_FRAME",
            "BEGIN_PARTITION",
            "BERNOULLI",
            "BIGINT",
            "BINARY",
            "BIT",
            "BLOB",
            "BOOLEAN",
            "BREADTH",
            "C",
            "CALLED",
            "CASCADE",
            "CASCADED",
            "CATALOG",
            "CATALOG_NAME",
            "CEIL",
            "CENTURY",
            "CHAIN",
            "CHARACTERISTICS",
            "CHARACTERS",
            "CHARACTER_SET_CATALOG",
            "CHARACTER_SET_NAME",
            "CHARACTER_SET_SCHEMA",
            "CHECK",
            "CLASSIFIER",
            "CLASS_ORIGIN",
            "CLOB",
            "CLOSE",
            "COBOL",
            "COLLATE",
            "COLLATION",
            "COLLATION_CATALOG",
            "COLLATION_NAME",
            "COLLATION_SCHEMA",
            "COLOCATE",
            "COLUMN_NAME",
            "COMMAND_FUNCTION",
            "COMMAND_FUNCTION_CODE",
            "COMMIT",
            "COMMITTED",
            "COMPUTE",
            "CONDITION",
            "CONDITIONAL",
            "CONDITION_NUMBER",
            "CONNECT",
            "CONNECTION",
            "CONNECTION_NAME",
            "CONSISTENCY",
            "CONSTRAINTS",
            "CONSTRAINT_CATALOG",
            "CONSTRAINT_NAME",
            "CONSTRAINT_SCHEMA",
            "CONSTRUCTOR",
            "CONTAINS",
            "CONTAINS_SUBSTR",
            "CONTINUE",
            "CORR",
            "CORRESPONDING",
            "CURSOR",
            "CURSOR_NAME",
            "CYCLE",
            "DATA",
            "DATABASE",
            "DATETIME_DIFF",
            "DATETIME_INTERVAL_CODE",
            "DATETIME_INTERVAL_PRECISION",
            "DATETIME_TRUNC",
            "DATE_DIFF",
            "DATE_TRUNC",
            "DAY",
            "DAYOFWEEK",
            "DAYOFYEAR",
            "DAYS",
            "DEALLOCATE",
            "DEC",
            "DECADE",
            "DECLARE",
            "DEFAULTS",
            "DEFERRABLE",
            "DEFERRED",
            "DEFINE",
            "DEFINED",
            "DEFINER",
            "DEGREE",
            "DEPTH",
            "DEREF",
            "DERIVED",
            "DESC",
            "DESCRIPTION",
            "DESCRIPTOR",
            "DETERMINISTIC",
            "DIAGNOSTICS",
            "DISALLOW",
            "DISCONNECT",
            "DISPATCH",
            "DISTRIBUTION",
            "DOMAIN",
            "DOT",
            "DOUBLE",
            "DOW",
            "DOWN",
            "DOY",
            "DYNAMIC",
            "DYNAMIC_FUNCTION",
            "DYNAMIC_FUNCTION_CODE",
            "EACH",
            "EMPTY",
            "ENCODING",
            "END",
            "END-EXEC",
            "END_FRAME",
            "END_PARTITION",
            "ENGINE",
            "EPOCH",
            "EQUALS",
            "ERROR",
            "ESCAPE",
            "EXCEPTION",
            "EXCLUDE",
            "EXCLUDING",
            "EXEC",
            "EXECUTE",
            "EXTERNAL",
            "FINAL",
            "FIRST",
            "FLOAT",
            "FOLLOWING",
            "FOREIGN",
            "FORMAT",
            "FORTRAN",
            "FOUND",
            "FRAC_SECOND",
            "FRAME_ROW",
            "FREE",
            "FUNCTION",
            "G",
            "GENERAL",
            "GENERATED",
            "GEOMETRY",
            "GET",
            "GLOBAL",
            "GO",
            "GOTO",
            "GRANT",
            "GRANTED",
            "GROUPS",
            "GROUP_CONCAT",
            "HASH",
            "HIERARCHY",
            "HOLD",
            "HOP",
            "HOURS",
            "IDENTITY",
            "IGNORE",
            "ILIKE",
            "IMMEDIATE",
            "IMMEDIATELY",
            "IMPLEMENTATION",
            "IMPORT",
            "INCLUDE",
            "INCLUDING",
            "INCREMENT",
            "INDICATOR",
            "INITIAL",
            "INITIALLY",
            "INOUT",
            "INPUT",
            "INSENSITIVE",
            "INSTANCE",
            "INSTANTIABLE",
            "INT",
            "INTEGER",
            "INVOKER",
            "ISODOW",
            "ISOLATION",
            "ISOYEAR",
            "JAVA",
            "JSON",
            "JSON_ARRAY",
            "JSON_ARRAYAGG",
            "JSON_EXISTS",
            "JSON_OBJECT",
            "JSON_OBJECTAGG",
            "JSON_QUERY",
            "JSON_VALUE",
            "K",
            "KEY",
            "KEY_MEMBER",
            "KEY_TYPE",
            "KILL",
            "LABEL",
            "LANGUAGE",
            "LARGE",
            "LAST",
            "LATERAL",
            "LENGTH",
            "LEVEL",
            "LIBRARY",
            "LIKE_REGEX",
            "LOCAL",
            "LOCATOR",
            "M",
            "MAP",
            "MAPPING",
            "MATCH",
            "MATCHED",
            "MATCHES",
            "MATCH_NUMBER",
            "MAXVALUE",
            "MEASURES",
            "MEMBER",
            "MESSAGE_LENGTH",
            "MESSAGE_OCTET_LENGTH",
            "MESSAGE_TEXT",
            "METHOD",
            "MICROSECOND",
            "MILLENNIUM",
            "MILLISECOND",
            "MINUTES",
            "MINVALUE",
            "MODE",
            "MODIFIES",
            "MODULE",
            "MONTHS",
            "MORE",
            "MUMPS",
            "NAME",
            "NAMES",
            "NANOSECOND",
            "NATIONAL",
            "NCHAR",
            "NCLOB",
            "NESTING",
            "NO",
            "NODES",
            "NONE",
            "NORMALIZE",
            "NORMALIZED",
            "NULLABLE",
            "NULLS",
            "NUMBER",
            "NUMERIC",
            "OBJECT",
            "OCCURRENCES_REGEX",
            "OCTETS",
            "OF",
            "OFF",
            "OLD",
            "OMIT",
            "ONE",
            "ONLY",
            "OPEN",
            "OPTION",
            "OPTIONS",
            "ORDERING",
            "ORDINAL",
            "ORDINALITY",
            "OTHERS",
            "OUT",
            "OUTPUT",
            "OVERLAPS",
            "OVERLAY",
            "OVERRIDING",
            "PAD",
            "PARAMETER",
            "PARAMETER_MODE",
            "PARAMETER_NAME",
            "PARAMETER_ORDINAL_POSITION",
            "PARAMETER_SPECIFIC_CATALOG",
            "PARAMETER_SPECIFIC_NAME",
            "PARAMETER_SPECIFIC_SCHEMA",
            "PARTIAL",
            "PARTITIONS",
            "PASCAL",
            "PASSING",
            "PASSTHROUGH",
            "PAST",
            "PATH",
            "PATTERN",
            "PER",
            "PERCENT",
            "PIVOT",
            "PLACING",
            "PLAN",
            "PLI",
            "PORTION",
            "POSITION",
            "POSITION_REGEX",
            "PRECEDES",
            "PRECEDING",
            "PREPARE",
            "PRESERVE",
            "PREV",
            "PRIOR",
            "PRIVILEGES",
            "PROCEDURE",
            "PROFILE",
            "PROFILES",
            "PUBLIC",
            "QUARTER",
            "QUARTERS",
            "QUERY",
            "QUORUM",
            "RANGE",
            "READ",
            "READS",
            "REAL",
            "RECURSIVE",
            "REF",
            "REFERENCES",
            "REFERENCING",
            "REGR_AVGX",
            "REGR_AVGY",
            "REGR_INTERCEPT",
            "REGR_R2",
            "REGR_SLOPE",
            "REGR_SXY",
            "RELATIVE",
            "RELEASE",
            "REPEATABLE",
            "REPLACE",
            "REPLICAS",
            "RESPECT",
            "RESTART",
            "RESTRICT",
            "RESULT",
            "RETURN",
            "RETURNED_CARDINALITY",
            "RETURNED_LENGTH",
            "RETURNED_OCTET_LENGTH",
            "RETURNED_SQLSTATE",
            "RETURNING",
            "RETURNS",
            "REVOKE",
            "RLIKE",
            "ROLE",
            "ROLLBACK",
            "ROUTINE",
            "ROUTINE_CATALOG",
            "ROUTINE_NAME",
            "ROUTINE_SCHEMA",
            "ROWS",
            "ROW_COUNT",
            "RUNNING",
            "SAFE_CAST",
            "SAFE_OFFSET",
            "SAFE_ORDINAL",
            "SAVEPOINT",
            "SCALAR",
            "SCALE",
            "SCHEMA",
            "SCHEMA_NAME",
            "SCOPE",
            "SCOPE_CATALOGS",
            "SCOPE_NAME",
            "SCOPE_SCHEMA",
            "SCROLL",
            "SEARCH",
            "SECONDS",
            "SECTION",
            "SECURITY",
            "SEEK",
            "SELF",
            "SENSITIVE",
            "SEPARATOR",
            "SEQUENCE",
            "SERIALIZABLE",
            "SERVER",
            "SERVER_NAME",
            "SESSION",
            "SETS",
            "SHOW",
            "SIMILAR",
            "SIMPLE",
            "SIZE",
            "SKIP",
            "SMALLINT",
            "SORTED",
            "SOURCE",
            "SPACE",
            "SPECIFICTYPE",
            "SPECIFIC_NAME",
            "SQL",
            "SQLEXCEPTION",
            "SQLSTATE",
            "SQLWARNING",
            "SQL_BIGINT",
            "SQL_BINARY",
            "SQL_BIT",
            "SQL_BLOB",
            "SQL_BOOLEAN",
            "SQL_CHAR",
            "SQL_CLOB",
            "SQL_DATE",
            "SQL_DECIMAL",
            "SQL_DOUBLE",
            "SQL_FLOAT",
            "SQL_INTEGER",
            "SQL_INTERVAL_DAY",
            "SQL_INTERVAL_DAY_TO_HOUR",
            "SQL_INTERVAL_DAY_TO_MINUTE",
            "SQL_INTERVAL_DAY_TO_SECOND",
            "SQL_INTERVAL_HOUR",
            "SQL_INTERVAL_HOUR_TO_MINUTE",
            "SQL_INTERVAL_HOUR_TO_SECOND",
            "SQL_INTERVAL_MINUTE",
            "SQL_INTERVAL_MINUTE_TO_SECOND",
            "SQL_INTERVAL_MONTH",
            "SQL_INTERVAL_SECOND",
            "SQL_INTERVAL_YEAR",
            "SQL_INTERVAL_YEAR_TO_MONTH",
            "SQL_LONGVARBINARY",
            "SQL_LONGVARCHAR",
            "SQL_LONGVARNCHAR",
            "SQL_NCHAR",
            "SQL_NCLOB",
            "SQL_NUMERIC",
            "SQL_NVARCHAR",
            "SQL_REAL",
            "SQL_SMALLINT",
            "SQL_TIME",
            "SQL_TIMESTAMP",
            "SQL_TINYINT",
            "SQL_TSI_DAY",
            "SQL_TSI_FRAC_SECOND",
            "SQL_TSI_HOUR",
            "SQL_TSI_MICROSECOND",
            "SQL_TSI_MINUTE",
            "SQL_TSI_MONTH",
            "SQL_TSI_QUARTER",
            "SQL_TSI_SECOND",
            "SQL_TSI_WEEK",
            "SQL_TSI_YEAR",
            "SQL_VARBINARY",
            "SQL_VARCHAR",
            "START",
            "STATE",
            "STATEMENT",
            "STATIC",
            "STORAGE",
            "STRING_AGG",
            "STRUCTURE",
            "STYLE",
            "SUBCLASS_ORIGIN",
            "SUBMULTISET",
            "SUBSET",
            "SUBSTITUTE",
            "SUBSTRING_REGEX",
            "SUCCEEDS",
            "SYSTEM",
            "TABLE_NAME",
            "TEMPORARY",
            "TIES",
            "TIMESTAMPADD",
            "TIMESTAMPDIFF",
            "TIMESTAMP_DIFF",
            "TIMESTAMP_TRUNC",
            "TIMEZONE_HOUR",
            "TIMEZONE_MINUTE",
            "TIME_DIFF",
            "TIME_TRUNC",
            "TINYINT",
            "TOP_LEVEL_COUNT",
            "TRANSACTION",
            "TRANSACTIONS_ACTIVE",
            "TRANSACTIONS_COMMITTED",
            "TRANSACTIONS_ROLLED_BACK",
            "TRANSFORM",
            "TRANSFORMS",
            "TRANSLATE",
            "TRANSLATE_REGEX",
            "TRANSLATION",
            "TREAT",
            "TRIGGER",
            "TRIGGER_CATALOG",
            "TRIGGER_NAME",
            "TRIGGER_SCHEMA",
            "TRIM",
            "TRIM_ARRAY",
            "TRY_CAST",
            "TUMBLE",
            "TYPE",
            "UNBOUNDED",
            "UNCOMMITTED",
            "UNCONDITIONAL",
            "UNDER",
            "UNIQUE",
            "UNNAMED",
            "UNNEST",
            "UNPIVOT",
            "UP",
            "USAGE",
            "USER_DEFINED_TYPE_CATALOG",
            "USER_DEFINED_TYPE_CODE",
            "USER_DEFINED_TYPE_NAME",
            "USER_DEFINED_TYPE_SCHEMA",
            "UTF16",
            "UTF32",
            "UTF8",
            "VALUE_OF",
            "VARBINARY",
            "VARCHAR",
            "VARYING",
            "VERSION",
            "VERSIONING",
            "VIEW",
            "WAIT",
            "WEEK",
            "WEEKS",
            "WHENEVER",
            "WIDTH_BUCKET",
            "WITHOUT",
            "WORK",
            "WRAPPER",
            "WRITE",
            "XML",
            "YEARS",
            "ZONE"
    );

    @Test
    public void testReservedWords() {
        @SuppressWarnings("DataFlowIssue")
        SqlAbstractParserImpl.Metadata md = IgniteSqlParserImpl.FACTORY.getParser(null).getMetadata();

        Set<String> unexpectedReserved = new TreeSet<>();
        Set<String> shouldBeReserved = new TreeSet<>();

        for (String s : md.getTokens()) {
            if (md.isReservedWord(s) && !RESERVED_WORDS.contains(s)) {
                unexpectedReserved.add(s);
            } else if (!md.isReservedWord(s) && RESERVED_WORDS.contains(s)) {
                shouldBeReserved.add(s);
            }
        }

        assertTrue(
                unexpectedReserved.isEmpty(),
                "Unexpected reserved keywords: \n" + formatKeywords(unexpectedReserved)
        );

        assertTrue(
                shouldBeReserved.isEmpty(),
                "Keywords should be reserved: \n" + formatKeywords(shouldBeReserved)
        );
    }

    @Test
    public void testKeywordsWords() {
        @SuppressWarnings("DataFlowIssue")
        SqlAbstractParserImpl.Metadata md = IgniteSqlParserImpl.FACTORY.getParser(null).getMetadata();

        Set<String> unexpectedKeywords = new TreeSet<>();
        Set<String> shouldBeKeyword = new TreeSet<>();

        Set<String> allKeywords = new HashSet<>(RESERVED_WORDS);
        allKeywords.addAll(NON_RESERVED_WORDS);

        for (String s : md.getTokens()) {
            if (md.isKeyword(s) && !allKeywords.contains(s)) {
                unexpectedKeywords.add(s);
            } else if (!md.isKeyword(s) && allKeywords.contains(s)) {
                shouldBeKeyword.add(s);
            }
        }

        assertTrue(
                unexpectedKeywords.isEmpty(),
                "Unexpected keywords: \n" + formatKeywords(unexpectedKeywords)
        );

        assertTrue(
                shouldBeKeyword.isEmpty(),
                "Token should be keyword: \n" + formatKeywords(shouldBeKeyword)
        );
    }

    /**
     * Formatted as config.fmpp:nonReservedKeywordsToAdd keywords.
     */
    private static String formatKeywords(Set<String> keywords) {
        IgniteStringBuilder sb = new IgniteStringBuilder();

        keywords.forEach(s -> sb.app("      \"").app(s).app("\"").nl());

        return sb.toString();
    }
}
