/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.network.processor.messages;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.Modifier;
import javax.lang.model.type.TypeKind;
import javax.tools.Diagnostic;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import org.apache.ignite.internal.network.processor.MessageClass;
import org.apache.ignite.internal.network.processor.MessageGroupWrapper;
import org.apache.ignite.network.NetworkMessage;

/**
 * Class for generating implementations of the {@link NetworkMessage} interfaces and their builders, generated by a
 * {@link MessageBuilderGenerator}.
 */
public class MessageImplGenerator {
    /** */
    private final ProcessingEnvironment processingEnv;

    /** Message group. */
    private final MessageGroupWrapper messageGroup;

    /**
     * @param processingEnv processing environment
     * @param messageGroup message group
     */
    public MessageImplGenerator(ProcessingEnvironment processingEnv, MessageGroupWrapper messageGroup) {
        this.processingEnv = processingEnv;
        this.messageGroup = messageGroup;
    }

    /**
     * Generates the implementation of a given Network Message interface and its Builder (as a nested class).
     *
     * @param message network message
     * @param builderInterface generated builder interface
     * @return {@code TypeSpec} of the generated message implementation
     */
    public TypeSpec generateMessageImpl(MessageClass message, TypeSpec builderInterface) {
        ClassName messageImplClassName = message.implClassName();

        processingEnv.getMessager()
            .printMessage(Diagnostic.Kind.NOTE, "Generating " + messageImplClassName, message.element());

        List<ExecutableElement> getters = message.getters();

        var fields = new ArrayList<FieldSpec>(getters.size());
        var getterImpls = new ArrayList<MethodSpec>(getters.size());

        // create a field and a getter implementation for every getter in the message interface
        for (ExecutableElement getter : getters) {
            var getterReturnType = TypeName.get(getter.getReturnType());

            String getterName = getter.getSimpleName().toString();

            FieldSpec field = FieldSpec.builder(getterReturnType, getterName)
                .addModifiers(Modifier.PRIVATE, Modifier.FINAL)
                .build();

            fields.add(field);

            MethodSpec getterImpl = MethodSpec.overriding(getter)
                .addStatement("return $N", field)
                .build();

            getterImpls.add(getterImpl);
        }

        TypeSpec.Builder messageImpl = TypeSpec.classBuilder(messageImplClassName)
            .addModifiers(Modifier.PUBLIC)
            .addSuperinterface(message.className())
            .addFields(fields)
            .addMethods(getterImpls)
            .addMethod(constructor(fields));

        // group type constant and getter
        FieldSpec groupTypeField = FieldSpec.builder(short.class, "GROUP_TYPE")
            .addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL)
            .initializer("$L", messageGroup.groupType())
            .build();

        messageImpl.addField(groupTypeField);

        MethodSpec groupTypeMethod = MethodSpec.methodBuilder("groupType")
            .addAnnotation(Override.class)
            .addModifiers(Modifier.PUBLIC)
            .returns(short.class)
            .addStatement("return $N", groupTypeField)
            .build();

        messageImpl.addMethod(groupTypeMethod);

        // message type constant and getter
        FieldSpec messageTypeField = FieldSpec.builder(short.class, "TYPE")
            .addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL)
            .initializer("$L", message.messageType())
            .build();

        messageImpl.addField(messageTypeField);

        MethodSpec messageTypeMethod = MethodSpec.methodBuilder("messageType")
            .addAnnotation(Override.class)
            .addModifiers(Modifier.PUBLIC)
            .returns(short.class)
            .addStatement("return $N", messageTypeField)
            .build();

        messageImpl.addMethod(messageTypeMethod);

        // equals and hashCode
        generateEqualsAndHashCode(messageImpl, message);

        var builderName = ClassName.get(message.packageName(), builderInterface.name);

        // nested builder interface and static factory method
        TypeSpec builder = generateBuilderImpl(message, messageImplClassName, builderName);

        messageImpl.addType(builder);

        MethodSpec builderMethod = MethodSpec.methodBuilder("builder")
            .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
            .returns(builderName)
            .addStatement("return new $N()", builder)
            .build();

        messageImpl.addMethod(builderMethod);

        messageImpl
            .addOriginatingElement(message.element())
            .addOriginatingElement(messageGroup.element());

        return messageImpl.build();
    }

    /**
     * Generates implementations of {@link #hashCode} and {@link #equals} for the provided {@code message}
     * and adds them to the provided builder.
     */
    private static void generateEqualsAndHashCode(TypeSpec.Builder messageImplBuilder, MessageClass message) {
        MethodSpec.Builder equals = MethodSpec.methodBuilder("equals")
            .addAnnotation(Override.class)
            .addModifiers(Modifier.PUBLIC)
            .returns(boolean.class)
            .addParameter(Object.class, "other")
            .addCode(CodeBlock.builder()
                .beginControlFlow("if (this == other)")
                .addStatement("return true")
                .endControlFlow()
                .build())
            .addCode(CodeBlock.builder()
                .beginControlFlow("if (other == null || getClass() != other.getClass())")
                .addStatement("return false")
                .endControlFlow()
                .build());

        MethodSpec.Builder hashCode = MethodSpec.methodBuilder("hashCode")
            .addAnnotation(Override.class)
            .addModifiers(Modifier.PUBLIC)
            .returns(int.class);

        if (message.getters().isEmpty()) {
            equals.addStatement("return true");

            hashCode.addStatement("return $T.class.hashCode()", message.implClassName());

            messageImplBuilder
                .addMethod(equals.build())
                .addMethod(hashCode.build());

            return;
        }

        var arrays = new ArrayList<ExecutableElement>();
        var primitives = new ArrayList<ExecutableElement>();
        var others = new ArrayList<ExecutableElement>();

        for (ExecutableElement element : message.getters()) {
            TypeKind typeKind = element.getReturnType().getKind();

            if (typeKind.isPrimitive())
                primitives.add(element);
            else if (typeKind == TypeKind.ARRAY)
                arrays.add(element);
            else
                others.add(element);
        }

        CodeBlock.Builder comparisonStatement = CodeBlock.builder().add("return ");

        boolean first = true;

        // objects are compared using "Objects.equals"
        for (ExecutableElement other : others) {
            if (first)
                first = false;
            else
                comparisonStatement.add(" && ");

            String fieldName = other.getSimpleName().toString();

            comparisonStatement.add("$T.equals(this.$L, otherMessage.$L)", Objects.class, fieldName, fieldName);
        }

        // arrays are compared using "Arrays.equals"
        for (ExecutableElement array : arrays) {
            if (first)
                first = false;
            else
                comparisonStatement.add(" && ");

            String fieldName = array.getSimpleName().toString();

            comparisonStatement.add("$T.equals(this.$L, otherMessage.$L)", Arrays.class, fieldName, fieldName);
        }

        // primitives are compared using "==", except for floating point values (because of NaNs and stuff)
        for (ExecutableElement primitive : primitives) {
            if (first)
                first = false;
            else
                comparisonStatement.add(" && ");

            String fieldName = primitive.getSimpleName().toString();

            switch (primitive.getReturnType().getKind()) {
                case FLOAT:
                    comparisonStatement.add("$T.compare(this.$L, otherMessage.$L) == 0", Float.class, fieldName, fieldName);
                    break;
                case DOUBLE:
                    comparisonStatement.add("$T.compare(this.$L, otherMessage.$L) == 0", Double.class, fieldName, fieldName);
                    break;
                default:
                    comparisonStatement.add("this.$L == otherMessage.$L", fieldName, fieldName);
                    break;
            }
        }

        equals
            .addStatement("var otherMessage = ($T)other", message.implClassName())
            .addStatement(comparisonStatement.build());

        hashCode
            .addStatement("int result = 0");

        // primitives can be boxed and used in "Objects.hash"
        String objectHashCode = Stream.concat(primitives.stream(), others.stream())
            .map(element -> "this." + element.getSimpleName())
            .collect(Collectors.joining(", ", "result = $T.hash(", ")"));

        if (!objectHashCode.isEmpty())
            hashCode.addStatement(objectHashCode, Objects.class);

        for (ExecutableElement array : arrays)
            hashCode.addStatement("result = 31 * result + $T.hashCode(this.$L)", Arrays.class, array.getSimpleName());

        hashCode.addStatement("return result");

        messageImplBuilder
            .addMethod(equals.build())
            .addMethod(hashCode.build());
    }

    /**
     * Creates a constructor for the current Network Message implementation.
     */
    private static MethodSpec constructor(List<FieldSpec> fields) {
        MethodSpec.Builder constructor = MethodSpec.constructorBuilder()
            .addModifiers(Modifier.PRIVATE);

        fields.forEach(field ->
            constructor
                .addParameter(field.type, field.name)
                .addStatement("this.$N = $N", field, field)
        );

        return constructor.build();
    }

    /**
     * Generates a nested static class that implements the Builder interface, generated during previous steps.
     */
    private static TypeSpec generateBuilderImpl(
        MessageClass message, ClassName messageImplClass, ClassName builderName
    ) {
        List<ExecutableElement> messageGetters = message.getters();

        var fields = new ArrayList<FieldSpec>(messageGetters.size());
        var setters = new ArrayList<MethodSpec>(messageGetters.size());
        var getters = new ArrayList<MethodSpec>(messageGetters.size());

        for (ExecutableElement messageGetter : messageGetters) {
            var getterReturnType = TypeName.get(messageGetter.getReturnType());

            String getterName = messageGetter.getSimpleName().toString();

            FieldSpec field = FieldSpec.builder(getterReturnType, getterName)
                .addModifiers(Modifier.PRIVATE)
                .build();

            fields.add(field);

            MethodSpec setter = MethodSpec.methodBuilder(getterName)
                .addAnnotation(Override.class)
                .addModifiers(Modifier.PUBLIC)
                .returns(builderName)
                .addParameter(getterReturnType, getterName)
                .addStatement("this.$N = $L", field, getterName)
                .addStatement("return this")
                .build();

            setters.add(setter);

            MethodSpec getter = MethodSpec.methodBuilder(getterName)
                .addAnnotation(Override.class)
                .addModifiers(Modifier.PUBLIC)
                .returns(getterReturnType)
                .addStatement("return $N", field)
                .build();

            getters.add(getter);
        }

        return TypeSpec.classBuilder("Builder")
            .addModifiers(Modifier.PRIVATE, Modifier.STATIC)
            .addSuperinterface(builderName)
            .addFields(fields)
            .addMethods(setters)
            .addMethods(getters)
            .addMethod(buildMethod(message, messageImplClass, fields))
            .build();
    }

    /**
     * Generates the {@code build()} method for the Builder interface implementation.
     */
    private static MethodSpec buildMethod(MessageClass message, ClassName messageImplClass, List<FieldSpec> fields) {
        String constructorParameters = fields.stream()
            .map(field -> field.name)
            .collect(Collectors.joining(", ", "(", ")"));

        return MethodSpec.methodBuilder("build")
            .addAnnotation(Override.class)
            .addModifiers(Modifier.PUBLIC)
            .returns(message.className())
            .addStatement("return new $T$L", messageImplClass, constructorParameters)
            .build();
    }
}
