/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.processors;

import java.io.Serializable;
import javax.cache.Cache;
import javax.cache.configuration.Factory;
import javax.cache.integration.CacheLoaderException;
import javax.cache.integration.CacheWriterException;
import org.apache.ignite.IgniteCache;
import org.apache.ignite.cache.CacheAtomicityMode;
import org.apache.ignite.cache.CacheMode;
import org.apache.ignite.cache.CacheRebalanceMode;
import org.apache.ignite.cache.store.CacheStore;
import org.apache.ignite.cache.store.CacheStoreAdapter;
import org.apache.ignite.configuration.CacheConfiguration;
import org.apache.ignite.configuration.IgniteConfiguration;
import org.apache.ignite.testframework.MvccFeatureChecker;
import org.apache.ignite.testframework.junits.common.GridCommonAbstractTest;
import org.apache.ignite.transactions.Transaction;
import org.apache.ignite.transactions.TransactionConcurrency;
import org.apache.ignite.transactions.TransactionIsolation;
import org.junit.Before;
import org.junit.Test;

/**
 *
 */
public class GridCacheTxLoadFromStoreOnLockSelfTest extends GridCommonAbstractTest {
    /** */
    @Before
    public void beforeGridCacheTxLoadFromStoreOnLockSelfTest() {
        MvccFeatureChecker.skipIfNotSupported(MvccFeatureChecker.Feature.CACHE_STORE);
    }

    /** {@inheritDoc} */
    @Override protected IgniteConfiguration getConfiguration(String igniteInstanceName) throws Exception {
        IgniteConfiguration cfg = super.getConfiguration(igniteInstanceName);

        cfg.getTransactionConfiguration().setTxSerializableEnabled(true);

        return cfg;
    }

    /** {@inheritDoc} */
    @Override protected void beforeTestsStarted() throws Exception {
        startGridsMultiThreaded(4);
    }

    /**
     * @throws Exception If failed.
     */
    @Test
    public void testLoadedValueOneBackup() throws Exception {
        checkLoadedValue(1);
    }

    /**
     * @throws Exception If failed.
     */
    @Test
    public void testLoadedValueNoBackups() throws Exception {
        checkLoadedValue(0);
    }

    /**
     * @throws Exception If failed.
     */
    private void checkLoadedValue(int backups) throws Exception {
        CacheConfiguration<Integer, Integer> cacheCfg = new CacheConfiguration<>(DEFAULT_CACHE_NAME);

        cacheCfg.setCacheMode(CacheMode.PARTITIONED);
        cacheCfg.setAtomicityMode(CacheAtomicityMode.TRANSACTIONAL);
        cacheCfg.setRebalanceMode(CacheRebalanceMode.SYNC);
        cacheCfg.setCacheStoreFactory(new StoreFactory());
        cacheCfg.setReadThrough(true);
        cacheCfg.setBackups(backups);
        cacheCfg.setLoadPreviousValue(true);

        IgniteCache<Integer, Integer> cache = ignite(0).createCache(cacheCfg);

        for (int i = 0; i < 10; i++)
            assertEquals((Integer)i, cache.get(i));

        cache.removeAll();

        assertEquals(0, cache.size());

        for (TransactionConcurrency conc : TransactionConcurrency.values()) {
            for (TransactionIsolation iso : TransactionIsolation.values()) {
                info("Checking transaction [conc=" + conc + ", iso=" + iso + ']');

                try (Transaction tx = ignite(0).transactions().txStart(conc, iso)) {
                    for (int i = 0; i < 10; i++)
                        assertEquals("Invalid value for transaction [conc=" + conc + ", iso=" + iso + ']',
                            (Integer)i, cache.get(i));

                    tx.commit();
                }

                cache.removeAll();
                assertEquals(0, cache.size());
            }
        }

        cache.destroy();
    }

    /**
     *
     */
    private static class StoreFactory implements Factory<CacheStore<? super Integer, ? super Integer>> {
        /** {@inheritDoc} */
        @Override public CacheStore<? super Integer, ? super Integer> create() {
            return new Store();
        }
    }

    /**
     *
     */
    private static class Store extends CacheStoreAdapter<Integer, Integer> implements Serializable {
        /** {@inheritDoc} */
        @Override public Integer load(Integer key) throws CacheLoaderException {
            return key;
        }

        /** {@inheritDoc} */
        @Override public void write(Cache.Entry<? extends Integer, ? extends Integer> e)
            throws CacheWriterException {
            // No-op.
        }

        /** {@inheritDoc} */
        @Override public void delete(Object key) throws CacheWriterException {
            // No-op.
        }
    }
}
