/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hudi.common.model;

import org.apache.hudi.common.util.HoodieAvroUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.exception.HoodieIOException;

import org.apache.avro.Schema;
import org.apache.avro.generic.GenericRecord;
import org.apache.avro.generic.IndexedRecord;

import java.io.IOException;

/**
 * The implementation of {@link HoodieRecordPayload} base on avro for testing.
 */
public class AvroBinaryTestPayload implements HoodieRecordPayload {

  private final byte[] recordBytes;

  public AvroBinaryTestPayload(Option<GenericRecord> record) {

    try {
      if (record.isPresent()) {
        recordBytes = HoodieAvroUtils.avroToBytes(record.get());
      } else {
        recordBytes = new byte[0];
      }
    } catch (IOException io) {
      throw new HoodieIOException("unable to convert payload to bytes");
    }
  }

  @Override
  public HoodieRecordPayload preCombine(HoodieRecordPayload another) {
    return this;
  }

  @Override
  public Option<IndexedRecord> combineAndGetUpdateValue(IndexedRecord currentValue, Schema schema) throws IOException {
    return getInsertValue(schema);
  }

  @Override
  public Option<IndexedRecord> getInsertValue(Schema schema) throws IOException {
    return Option.of(HoodieAvroUtils.bytesToAvro(recordBytes, schema));
  }
}
