/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.pipeline.transforms.tableoutput;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.hop.core.HopEnvironment;
import org.apache.hop.core.database.DatabaseMeta;
import org.apache.hop.core.database.IDatabase;
import org.apache.hop.core.exception.HopException;
import org.apache.hop.core.plugins.PluginRegistry;
import org.apache.hop.core.row.RowMeta;
import org.apache.hop.core.row.value.ValueMetaFactory;
import org.apache.hop.core.variables.IVariables;
import org.apache.hop.metadata.api.IHopMetadataProvider;
import org.apache.hop.metadata.serializer.memory.MemoryMetadataProvider;
import org.apache.hop.pipeline.transform.ITransformMeta;
import org.apache.hop.pipeline.transforms.loadsave.LoadSaveTester;
import org.apache.hop.pipeline.transforms.loadsave.initializer.IInitializer;
import org.apache.hop.pipeline.transforms.loadsave.validator.IFieldLoadSaveValidator;
import org.apache.hop.pipeline.transforms.loadsave.validator.ListLoadSaveValidator;
import org.junit.Before;
import org.junit.Test;

import java.util.*;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

public class TableOutputMetaTest implements IInitializer<ITransformMeta> {

  private IVariables variables;
  private List<DatabaseMeta> databases;
  private IHopMetadataProvider metadataProvider;
  LoadSaveTester loadSaveTester;
  Class<TableOutputMeta> testMetaClass = TableOutputMeta.class;

  @SuppressWarnings("unchecked")
  @Before
  public void setUp() {
    variables = mock(IVariables.class);
    databases = mock(List.class);
    metadataProvider = new MemoryMetadataProvider();
  }

  @Test
  public void testIsReturningGeneratedKeys() throws Exception {
    TableOutputMeta tableOutputMeta = new TableOutputMeta(),
        tableOutputMetaSpy = spy(tableOutputMeta);

    DatabaseMeta databaseMeta = mock(DatabaseMeta.class);
    doReturn(true).when(databaseMeta).supportsAutoGeneratedKeys();
    doReturn(databaseMeta).when(tableOutputMetaSpy).getDatabaseMeta();

    tableOutputMetaSpy.setReturningGeneratedKeys(true);
    assertTrue(tableOutputMetaSpy.isReturningGeneratedKeys());

    doReturn(false).when(databaseMeta).supportsAutoGeneratedKeys();
    assertFalse(tableOutputMetaSpy.isReturningGeneratedKeys());

    tableOutputMetaSpy.setReturningGeneratedKeys(true);
    assertFalse(tableOutputMetaSpy.isReturningGeneratedKeys());

    tableOutputMetaSpy.setReturningGeneratedKeys(false);
    assertFalse(tableOutputMetaSpy.isReturningGeneratedKeys());
  }

  @Test
  public void testProvidesModeler() throws Exception {
    TableOutputMeta tableOutputMeta = new TableOutputMeta();
    tableOutputMeta.getFields().add(new TableOutputField("f1", "s4"));
    tableOutputMeta.getFields().add(new TableOutputField("f2", "s5"));
    tableOutputMeta.getFields().add(new TableOutputField("f3", "s6"));

    TableOutputData tableOutputData = new TableOutputData();
    tableOutputData.insertRowMeta = mock(RowMeta.class);
    assertEquals(
        tableOutputData.insertRowMeta, tableOutputMeta.getRowMeta(variables, tableOutputData));

    tableOutputMeta.setSpecifyFields(false);
    assertEquals(0, tableOutputMeta.getDatabaseFields().size());
    assertEquals(0, tableOutputMeta.getStreamFields().size());

    tableOutputMeta.setSpecifyFields(true);
    assertEquals(3, tableOutputMeta.getDatabaseFields().size());
    assertEquals("f1", tableOutputMeta.getDatabaseFields().get(0));
    assertEquals("f2", tableOutputMeta.getDatabaseFields().get(1));
    assertEquals("f3", tableOutputMeta.getDatabaseFields().get(2));
    assertEquals(3, tableOutputMeta.getStreamFields().size());
    assertEquals("s4", tableOutputMeta.getStreamFields().get(0));
    assertEquals("s5", tableOutputMeta.getStreamFields().get(1));
    assertEquals("s6", tableOutputMeta.getStreamFields().get(2));
  }

  @Test
  public void testSetupDefault() throws Exception {
    TableOutputMeta tableOutputMeta = new TableOutputMeta();
    tableOutputMeta.setDefault();
    assertEquals("", tableOutputMeta.getTableName());
    assertEquals("1000", tableOutputMeta.getCommitSize());
    assertFalse(tableOutputMeta.isPartitioningEnabled());
    assertTrue(tableOutputMeta.isPartitioningMonthly());
    assertEquals("", tableOutputMeta.getPartitioningField());
    assertTrue(tableOutputMeta.isTableNameInTable());
    assertEquals("", tableOutputMeta.getTableNameField());
    assertFalse(tableOutputMeta.isSpecifyFields());
  }

  @Test
  public void testClone() throws Exception {
    TableOutputMeta tableOutputMeta = new TableOutputMeta();
    tableOutputMeta.setDefault();
    tableOutputMeta.getFields().add(new TableOutputField("d1", "1"));
    tableOutputMeta.getFields().add(new TableOutputField("d2", "2"));
    tableOutputMeta.getFields().add(new TableOutputField("d3", "3"));
    TableOutputMeta clone = (TableOutputMeta) tableOutputMeta.clone();
    assertNotSame(clone, tableOutputMeta);
    assertEquals(clone.getXml(), tableOutputMeta.getXml());
  }

  @Test
  public void testSupportsErrorHandling() throws Exception {
    TableOutputMeta tableOutputMeta = new TableOutputMeta();
    DatabaseMeta dbMeta = mock(DatabaseMeta.class);
    tableOutputMeta.setDatabaseMeta(dbMeta);
    IDatabase iDatabase = mock(IDatabase.class);
    when(dbMeta.getIDatabase()).thenReturn(iDatabase);
    when(iDatabase.supportsErrorHandling()).thenReturn(true, false);
    assertTrue(tableOutputMeta.supportsErrorHandling());
    assertFalse(tableOutputMeta.supportsErrorHandling());
    tableOutputMeta.setDatabaseMeta(null);
    assertTrue(tableOutputMeta.supportsErrorHandling());
  }

  @Before
  public void setUpLoadSave() throws Exception {
    HopEnvironment.init();
    PluginRegistry.init(false);

    List<String> attributesList = new ArrayList<>();
    Map<String, String> getterMap = new HashMap<>();

    Map<String, String> setterMap = new HashMap<>();

    Map<String, IFieldLoadSaveValidator<?>> attrValidatorMap = new HashMap<>();
    attrValidatorMap.put(
        "fields",
        new ListLoadSaveValidator<>(new TableOutputFieldInputFieldLoadSaveValidator(), 5));

    Map<String, IFieldLoadSaveValidator<?>> typeValidatorMap = new HashMap<>();

    loadSaveTester =
        new LoadSaveTester(
            testMetaClass,
            attributesList,
            new ArrayList<>(),
            getterMap,
            setterMap,
            attrValidatorMap,
            typeValidatorMap,
            this);
  }

  // Call the allocate method on the LoadSaveTester meta class
  @Override
  public void modify(ITransformMeta someMeta) {
    if (someMeta instanceof TableOutputMeta) {
      ((TableOutputMeta) someMeta).getFields().clear();
      ((TableOutputMeta) someMeta)
          .getFields()
          .addAll(
              Arrays.asList(
                  new TableOutputField("DatabaseField1", "StreamField1"),
                  new TableOutputField("DatabaseField2", "StreamField2"),
                  new TableOutputField("DatabaseField3", "StreamField3"),
                  new TableOutputField("DatabaseField4", "StreamField4"),
                  new TableOutputField("DatabaseField5", "StreamField5")));
    }
  }

  @Test
  public void testSerialization() throws HopException {
    loadSaveTester.testSerialization();
  }

  public class TableOutputFieldInputFieldLoadSaveValidator
      implements IFieldLoadSaveValidator<TableOutputField> {
    final Random rand = new Random();

    @Override
    public TableOutputField getTestObject() {
      String[] types = ValueMetaFactory.getAllValueMetaNames();

      TableOutputField field =
          new TableOutputField(UUID.randomUUID().toString(), UUID.randomUUID().toString());

      return field;
    }

    @Override
    public boolean validateTestObject(TableOutputField testObject, Object actual) {
      if (!(actual instanceof TableOutputField)) {
        return false;
      }
      TableOutputField another = (TableOutputField) actual;
      return new EqualsBuilder()
          .append(testObject.getFieldStream(), another.getFieldStream())
          .append(testObject.getFieldDatabase(), another.getFieldDatabase())
          .isEquals();
    }
  }
}
