/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.databases.mssqlnative;

import org.apache.hop.core.database.IDatabase;
import org.apache.hop.core.util.Utils;
import org.apache.hop.junit.rules.RestoreHopEnvironment;
import org.junit.ClassRule;
import org.junit.Test;

import static org.junit.Assert.*;

public class MsSqlServerNativeSequenceAndReleaseSavePointTest {
  @ClassRule public static RestoreHopEnvironment env = new RestoreHopEnvironment();

  final String sequenceName = "sequence_name";

  // Set these parameters for the test
  IDatabase db = new MsSqlServerNativeDatabaseMeta();
  Boolean sequenceSupport = true;
  Boolean savepointSupport = true;

  @Test
  public void testSequenceSupport() {
    assertSupports(db, sequenceSupport);
    assertEquals("SELECT NEXT VALUE FOR sequence_name", db.getSqlNextSequenceValue(sequenceName));
    assertEquals(
        "SELECT current_value FROM sys.sequences WHERE name = 'sequence_name'",
        db.getSqlCurrentSequenceValue(sequenceName));
    assertEquals("SELECT name FROM sys.sequences", db.getSqlListOfSequences());
    assertEquals(
        "SELECT 1 FROM sys.sequences WHERE name = 'sequence_name'",
        db.getSqlSequenceExists(sequenceName));
  }

  @Test
  public void testSavepointSuport() {
    if (savepointSupport) {
      assertTrue(db.releaseSavepoint());
    } else {
      assertFalse(db.releaseSavepoint());
    }
  }

  public static void assertSupports(IDatabase db, boolean expected) {
    String dbType = db.getClass().getSimpleName();
    if (expected) {
      assertTrue(dbType, db.supportsSequences());
      assertFalse(dbType + ": List of Sequences", Utils.isEmpty(db.getSqlListOfSequences()));
      assertFalse(dbType + ": Sequence Exists", Utils.isEmpty(db.getSqlSequenceExists("testSeq")));
      assertFalse(
          dbType + ": Current Value", Utils.isEmpty(db.getSqlCurrentSequenceValue("testSeq")));
      assertFalse(dbType + ": Next Value", Utils.isEmpty(db.getSqlNextSequenceValue("testSeq")));
    } else {
      assertFalse(db.getClass().getSimpleName(), db.supportsSequences());
      assertTrue(dbType + ": List of Sequences", Utils.isEmpty(db.getSqlListOfSequences()));
      assertTrue(dbType + ": Sequence Exists", Utils.isEmpty(db.getSqlSequenceExists("testSeq")));
      assertTrue(
          dbType + ": Current Value", Utils.isEmpty(db.getSqlCurrentSequenceValue("testSeq")));
      assertTrue(dbType + ": Next Value", Utils.isEmpty(db.getSqlNextSequenceValue("testSeq")));
    }
  }
}
