/*
 * Decompiled with CFR 0.152.
 */
package co.cask.tephra;

import co.cask.tephra.ChangeId;
import co.cask.tephra.InvalidTruncateTimeException;
import co.cask.tephra.Transaction;
import co.cask.tephra.TransactionNotInProgressException;
import co.cask.tephra.TransactionType;
import co.cask.tephra.TxConstants;
import co.cask.tephra.metrics.DefaultMetricsCollector;
import co.cask.tephra.metrics.MetricsCollector;
import co.cask.tephra.persist.NoOpTransactionStateStorage;
import co.cask.tephra.persist.TransactionEdit;
import co.cask.tephra.persist.TransactionLog;
import co.cask.tephra.persist.TransactionLogReader;
import co.cask.tephra.persist.TransactionSnapshot;
import co.cask.tephra.persist.TransactionStateStorage;
import co.cask.tephra.snapshot.SnapshotCodecProvider;
import co.cask.tephra.util.TxUtils;
import com.google.inject.Inject;
import it.unimi.dsi.fastutil.longs.LongArrayList;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hive.com.google.common.base.Objects;
import org.apache.hive.com.google.common.base.Preconditions;
import org.apache.hive.com.google.common.base.Stopwatch;
import org.apache.hive.com.google.common.base.Throwables;
import org.apache.hive.com.google.common.collect.Lists;
import org.apache.hive.com.google.common.collect.Maps;
import org.apache.hive.com.google.common.collect.Sets;
import org.apache.hive.com.google.common.util.concurrent.AbstractService;
import org.apache.hive.com.google.common.util.concurrent.Service;
import org.apache.hive.org.slf4j.Logger;
import org.apache.hive.org.slf4j.LoggerFactory;

public class TransactionManager
extends AbstractService {
    private static final Logger LOG = LoggerFactory.getLogger(TransactionManager.class);
    private static final long SNAPSHOT_POLL_INTERVAL = 1000L;
    private static final long METRICS_POLL_INTERVAL = 10000L;
    private static final long[] NO_INVALID_TX = new long[0];
    private final NavigableMap<Long, InProgressTx> inProgress = new ConcurrentSkipListMap<Long, InProgressTx>();
    private final LongArrayList invalid = new LongArrayList();
    private long[] invalidArray = NO_INVALID_TX;
    private final NavigableMap<Long, Set<ChangeId>> committedChangeSets = new ConcurrentSkipListMap<Long, Set<ChangeId>>();
    private final Map<Long, Set<ChangeId>> committingChangeSets = Maps.newConcurrentMap();
    private long readPointer;
    private long lastWritePointer;
    private MetricsCollector txMetricsCollector;
    private final TransactionStateStorage persistor;
    private final int cleanupInterval;
    private final int defaultTimeout;
    private final int defaultLongTimeout;
    private DaemonThreadExecutor cleanupThread = null;
    private volatile TransactionLog currentLog;
    private long lastSnapshotTime;
    private final long snapshotFrequencyInSeconds;
    private final int snapshotRetainCount;
    private DaemonThreadExecutor snapshotThread;
    private DaemonThreadExecutor metricsThread;
    private final ReentrantReadWriteLock logLock = new ReentrantReadWriteLock();
    private final Lock logReadLock = this.logLock.readLock();
    private final Lock logWriteLock = this.logLock.writeLock();
    private final long longTimeoutTolerance;

    public TransactionManager(Configuration config) {
        this(config, new NoOpTransactionStateStorage(new SnapshotCodecProvider(config)), new DefaultMetricsCollector());
    }

    @Inject
    public TransactionManager(Configuration conf, @Nonnull TransactionStateStorage persistor, MetricsCollector txMetricsCollector) {
        this.persistor = persistor;
        this.cleanupInterval = conf.getInt("data.tx.cleanup.interval", 10);
        this.defaultTimeout = conf.getInt("data.tx.timeout", 30);
        this.defaultLongTimeout = conf.getInt("data.tx.long.timeout", TxConstants.Manager.DEFAULT_TX_LONG_TIMEOUT);
        this.snapshotFrequencyInSeconds = conf.getLong("data.tx.snapshot.interval", 300L);
        this.snapshotRetainCount = Math.max(conf.getInt("data.tx.snapshot.retain", 10), 1);
        this.longTimeoutTolerance = conf.getLong("data.tx.long.timeout.tolerance", 10000L);
        this.txMetricsCollector = txMetricsCollector;
        this.txMetricsCollector.configure(conf);
        this.clear();
    }

    private void clear() {
        this.invalid.clear();
        this.invalidArray = NO_INVALID_TX;
        this.inProgress.clear();
        this.committedChangeSets.clear();
        this.committingChangeSets.clear();
        this.lastWritePointer = 0L;
        this.readPointer = 0L;
        this.lastSnapshotTime = 0L;
    }

    private boolean isStopping() {
        return Service.State.STOPPING.equals((Object)this.state());
    }

    @Override
    public synchronized void doStart() {
        LOG.info("Starting transaction manager.");
        this.txMetricsCollector.start();
        this.persistor.startAndWait();
        this.clear();
        this.recoverState();
        this.startCleanupThread();
        this.startSnapshotThread();
        this.startMetricsThread();
        this.initLog();
        if (this.lastWritePointer == 0L) {
            this.readPointer = this.lastWritePointer = this.getNextWritePointer();
        }
        this.notifyStarted();
    }

    private void initLog() {
        if (this.currentLog == null) {
            try {
                this.currentLog = this.persistor.createLog(System.currentTimeMillis());
            }
            catch (IOException ioe) {
                throw Throwables.propagate(ioe);
            }
        }
    }

    private void startCleanupThread() {
        if (this.cleanupInterval <= 0 || this.defaultTimeout <= 0) {
            return;
        }
        LOG.info("Starting periodic timed-out transaction cleanup every " + this.cleanupInterval + " seconds with default timeout of " + this.defaultTimeout + " seconds.");
        this.cleanupThread = new DaemonThreadExecutor("tx-clean-timeout"){

            @Override
            public void doRun() {
                TransactionManager.this.cleanupTimedOutTransactions();
            }

            @Override
            public long getSleepMillis() {
                return TransactionManager.this.cleanupInterval * 1000;
            }
        };
        this.cleanupThread.start();
    }

    private void startSnapshotThread() {
        if (this.snapshotFrequencyInSeconds > 0L) {
            LOG.info("Starting periodic snapshot thread, frequency = " + this.snapshotFrequencyInSeconds + " seconds, location = " + this.persistor.getLocation());
            this.snapshotThread = new DaemonThreadExecutor("tx-snapshot"){

                @Override
                public void doRun() {
                    long currentTime = System.currentTimeMillis();
                    if (TransactionManager.this.lastSnapshotTime < currentTime - TransactionManager.this.snapshotFrequencyInSeconds * 1000L) {
                        try {
                            TransactionManager.this.doSnapshot(false);
                        }
                        catch (IOException ioe) {
                            LOG.error("Periodic snapshot failed!", ioe);
                        }
                    }
                }

                @Override
                protected void onShutdown() {
                    try {
                        LOG.info("Writing final snapshot prior to shutdown");
                        TransactionManager.this.doSnapshot(true);
                    }
                    catch (IOException ioe) {
                        LOG.error("Failed performing final snapshot on shutdown", ioe);
                    }
                }

                @Override
                public long getSleepMillis() {
                    return 1000L;
                }
            };
            this.snapshotThread.start();
        }
    }

    private void startMetricsThread() {
        LOG.info("Starting periodic Metrics Emitter thread, frequency = 10000");
        this.metricsThread = new DaemonThreadExecutor("tx-metrics"){

            @Override
            public void doRun() {
                TransactionManager.this.txMetricsCollector.gauge("committing.size", TransactionManager.this.committingChangeSets.size(), new String[0]);
                TransactionManager.this.txMetricsCollector.gauge("committed.size", TransactionManager.this.committedChangeSets.size(), new String[0]);
                TransactionManager.this.txMetricsCollector.gauge("inprogress.size", TransactionManager.this.inProgress.size(), new String[0]);
                TransactionManager.this.txMetricsCollector.gauge("invalid.size", TransactionManager.this.invalidArray.length, new String[0]);
            }

            @Override
            protected void onShutdown() {
                TransactionManager.this.txMetricsCollector.gauge("committing.size", TransactionManager.this.committingChangeSets.size(), new String[0]);
                TransactionManager.this.txMetricsCollector.gauge("committed.size", TransactionManager.this.committedChangeSets.size(), new String[0]);
                TransactionManager.this.txMetricsCollector.gauge("inprogress.size", TransactionManager.this.inProgress.size(), new String[0]);
                TransactionManager.this.txMetricsCollector.gauge("invalid.size", TransactionManager.this.invalidArray.length, new String[0]);
            }

            @Override
            public long getSleepMillis() {
                return 10000L;
            }
        };
        this.metricsThread.start();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void cleanupTimedOutTransactions() {
        ArrayList<TransactionEdit> invalidEdits = null;
        this.logReadLock.lock();
        try {
            TransactionManager transactionManager = this;
            synchronized (transactionManager) {
                block13: {
                    if (this.isRunning()) break block13;
                    return;
                }
                long currentTime = System.currentTimeMillis();
                ArrayList timedOut = Lists.newArrayList();
                for (Map.Entry tx : this.inProgress.entrySet()) {
                    long expiration = ((InProgressTx)tx.getValue()).getExpiration();
                    if (expiration >= 0L && currentTime > expiration) {
                        timedOut.add(tx.getKey());
                        LOG.info("Tx invalid list: added tx {} because of timeout", tx.getKey());
                        continue;
                    }
                    if (expiration >= 0L) continue;
                    LOG.warn("Transaction {} has negative expiration time {}. Likely cause is the transaction was not migrated correctly, this transaction will be expired immediately", tx.getKey(), (Object)expiration);
                    timedOut.add(tx.getKey());
                }
                if (!timedOut.isEmpty()) {
                    invalidEdits = Lists.newArrayListWithCapacity(timedOut.size());
                    this.invalid.addAll(timedOut);
                    Iterator i$ = timedOut.iterator();
                    while (i$.hasNext()) {
                        long tx = (Long)((Object)i$.next());
                        this.committingChangeSets.remove(tx);
                        this.inProgress.remove(tx);
                        invalidEdits.add(TransactionEdit.createInvalid(tx));
                    }
                    Collections.sort(this.invalid);
                    this.invalidArray = this.invalid.toLongArray();
                    LOG.info("Invalidated {} transactions due to timeout.", (Object)timedOut.size());
                }
            }
            if (invalidEdits != null) {
                this.appendToLog(invalidEdits);
            }
        }
        finally {
            this.logReadLock.unlock();
        }
    }

    public synchronized TransactionSnapshot getSnapshot() throws IOException {
        TransactionSnapshot snapshot = null;
        if (!this.isRunning() && !this.isStopping()) {
            return null;
        }
        long now = System.currentTimeMillis();
        if (now == this.lastSnapshotTime || this.currentLog != null && now == this.currentLog.getTimestamp()) {
            try {
                TimeUnit.MILLISECONDS.sleep(1L);
            }
            catch (InterruptedException ie) {
                // empty catch block
            }
        }
        snapshot = this.getCurrentState();
        LOG.info("Starting snapshot of transaction state with timestamp {}", (Object)snapshot.getTimestamp());
        LOG.info("Returning snapshot of state: " + snapshot);
        return snapshot;
    }

    public boolean takeSnapshot(OutputStream out) throws IOException {
        TransactionSnapshot snapshot = this.getSnapshot();
        if (snapshot != null) {
            this.persistor.writeSnapshot(out, snapshot);
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doSnapshot(boolean closing) throws IOException {
        long snapshotTime = 0L;
        TransactionSnapshot snapshot = null;
        TransactionLog oldLog = null;
        try {
            this.logWriteLock.lock();
            try {
                TransactionManager transactionManager = this;
                synchronized (transactionManager) {
                    block14: {
                        snapshot = this.getSnapshot();
                        if (snapshot != null || closing) break block14;
                        return;
                    }
                    if (snapshot != null) {
                        snapshotTime = snapshot.getTimestamp();
                    }
                    oldLog = this.currentLog;
                    if (!closing) {
                        this.currentLog = this.persistor.createLog(snapshot.getTimestamp());
                    }
                }
                if (oldLog != null) {
                    oldLog.close();
                }
            }
            finally {
                this.logWriteLock.unlock();
            }
            if (snapshot != null) {
                this.persistor.writeSnapshot(snapshot);
                this.lastSnapshotTime = snapshotTime;
                long oldestRetainedTimestamp = this.persistor.deleteOldSnapshots(this.snapshotRetainCount);
                this.persistor.deleteLogsOlderThan(oldestRetainedTimestamp);
            }
        }
        catch (IOException ioe) {
            this.abortService("Snapshot (timestamp " + snapshotTime + ") failed due to: " + ioe.getMessage(), ioe);
        }
    }

    public synchronized TransactionSnapshot getCurrentState() {
        return TransactionSnapshot.copyFrom(System.currentTimeMillis(), this.readPointer, this.lastWritePointer, this.invalid, this.inProgress, this.committingChangeSets, this.committedChangeSets);
    }

    public synchronized void recoverState() {
        try {
            List<TransactionLog> logs;
            TransactionSnapshot lastSnapshot = this.persistor.getLatestSnapshot();
            if (lastSnapshot != null) {
                this.restoreSnapshot(lastSnapshot);
            }
            if ((logs = this.persistor.getLogsSince(this.lastSnapshotTime)) != null) {
                this.replayLogs(logs);
            }
        }
        catch (IOException e) {
            LOG.error("Unable to read back transaction state:", e);
            throw Throwables.propagate(e);
        }
    }

    private void restoreSnapshot(TransactionSnapshot snapshot) {
        LOG.info("Restoring transaction state from snapshot at " + snapshot.getTimestamp());
        Preconditions.checkState(this.lastSnapshotTime == 0L, "lastSnapshotTime has been set!");
        Preconditions.checkState(this.readPointer == 0L, "readPointer has been set!");
        Preconditions.checkState(this.lastWritePointer == 0L, "lastWritePointer has been set!");
        Preconditions.checkState(this.invalid.isEmpty(), "invalid list should be empty!");
        Preconditions.checkState(this.inProgress.isEmpty(), "inProgress map should be empty!");
        Preconditions.checkState(this.committingChangeSets.isEmpty(), "committingChangeSets should be empty!");
        Preconditions.checkState(this.committedChangeSets.isEmpty(), "committedChangeSets should be empty!");
        LOG.info("Restoring snapshot of state: " + snapshot);
        this.lastSnapshotTime = snapshot.getTimestamp();
        this.readPointer = snapshot.getReadPointer();
        this.lastWritePointer = snapshot.getWritePointer();
        this.invalid.addAll((Collection<? extends Long>)snapshot.getInvalid());
        this.inProgress.putAll(TransactionManager.txnBackwardsCompatCheck(this.defaultLongTimeout, this.longTimeoutTolerance, snapshot.getInProgress()));
        this.committingChangeSets.putAll(snapshot.getCommittingChangeSets());
        this.committedChangeSets.putAll(snapshot.getCommittedChangeSets());
    }

    public static Map<Long, InProgressTx> txnBackwardsCompatCheck(int defaultLongTimeout, long longTimeoutTolerance, Map<Long, InProgressTx> inProgress) {
        for (Map.Entry<Long, InProgressTx> entry : inProgress.entrySet()) {
            long writePointer = entry.getKey();
            long expiration = entry.getValue().getExpiration();
            if (entry.getValue().getType() == null && (expiration < 0L || TransactionManager.getTxExpirationFromWritePointer(writePointer, defaultLongTimeout) - expiration < longTimeoutTolerance)) {
                long newExpiration = TransactionManager.getTxExpirationFromWritePointer(writePointer, defaultLongTimeout);
                InProgressTx compatTx = new InProgressTx(entry.getValue().getVisibilityUpperBound(), newExpiration, TransactionType.LONG, entry.getValue().getCheckpointWritePointers());
                entry.setValue(compatTx);
                continue;
            }
            if (entry.getValue().getType() != null) continue;
            InProgressTx compatTx = new InProgressTx(entry.getValue().getVisibilityUpperBound(), entry.getValue().getExpiration(), TransactionType.SHORT, entry.getValue().getCheckpointWritePointers());
            entry.setValue(compatTx);
        }
        return inProgress;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void resetState() {
        this.logWriteLock.lock();
        try {
            this.doSnapshot(false);
            this.clear();
            this.doSnapshot(false);
        }
        catch (IOException e) {
            LOG.error("Snapshot failed when resetting state!", e);
            e.printStackTrace();
        }
        finally {
            this.logWriteLock.unlock();
        }
    }

    private void replayLogs(Collection<TransactionLog> logs) {
        for (TransactionLog log : logs) {
            LOG.info("Replaying edits from transaction log " + log.getName());
            int editCnt = 0;
            try {
                TransactionLogReader reader = log.getReader();
                if (reader == null) continue;
                TransactionEdit edit = null;
                block13: while ((edit = reader.next()) != null) {
                    ++editCnt;
                    switch (edit.getState()) {
                        case INPROGRESS: {
                            long expiration = edit.getExpiration();
                            TransactionType type = edit.getType();
                            if (expiration < 0L) {
                                expiration = TransactionManager.getTxExpirationFromWritePointer(edit.getWritePointer(), this.defaultLongTimeout);
                                type = TransactionType.LONG;
                            } else if (type == null) {
                                type = TransactionType.SHORT;
                            }
                            this.addInProgressAndAdvance(edit.getWritePointer(), edit.getVisibilityUpperBound(), expiration, type);
                            continue block13;
                        }
                        case COMMITTING: {
                            this.addCommittingChangeSet(edit.getWritePointer(), edit.getChanges());
                            continue block13;
                        }
                        case COMMITTED: {
                            this.doCommit(edit.getWritePointer(), edit.getChanges(), edit.getCommitPointer(), edit.getCanCommit());
                            continue block13;
                        }
                        case INVALID: {
                            this.doInvalidate(edit.getWritePointer());
                            continue block13;
                        }
                        case ABORTED: {
                            TransactionType type = edit.getType();
                            if (type == null) {
                                InProgressTx inProgressTx = (InProgressTx)this.inProgress.get(edit.getWritePointer());
                                if (inProgressTx != null) {
                                    type = inProgressTx.getType();
                                } else {
                                    LOG.warn("Invalidating transaction {} as it's type cannot be determined during replay", (Object)edit.getWritePointer());
                                    this.doInvalidate(edit.getWritePointer());
                                    continue block13;
                                }
                            }
                            this.doAbort(edit.getWritePointer(), edit.getCheckpointPointers(), type);
                            continue block13;
                        }
                        case TRUNCATE_INVALID_TX: {
                            if (edit.getTruncateInvalidTxTime() != 0L) {
                                this.doTruncateInvalidTxBefore(edit.getTruncateInvalidTxTime());
                                continue block13;
                            }
                            this.doTruncateInvalidTx(edit.getTruncateInvalidTx());
                            continue block13;
                        }
                        case CHECKPOINT: {
                            this.doCheckpoint(edit.getWritePointer(), edit.getParentWritePointer());
                            continue block13;
                        }
                    }
                    throw new IllegalArgumentException("Invalid state for WAL entry: " + (Object)((Object)edit.getState()));
                }
            }
            catch (IOException ioe) {
                throw Throwables.propagate(ioe);
            }
            catch (InvalidTruncateTimeException e) {
                throw Throwables.propagate(e);
            }
            LOG.info("Read " + editCnt + " edits from log " + log.getName());
        }
    }

    @Override
    public void doStop() {
        Stopwatch timer = new Stopwatch().start();
        LOG.info("Shutting down gracefully...");
        if (this.cleanupThread != null) {
            this.cleanupThread.shutdown();
            try {
                this.cleanupThread.join(30000L);
            }
            catch (InterruptedException ie) {
                LOG.warn("Interrupted waiting for cleanup thread to stop");
                Thread.currentThread().interrupt();
            }
        }
        if (this.metricsThread != null) {
            this.metricsThread.shutdown();
            try {
                this.metricsThread.join(30000L);
            }
            catch (InterruptedException ie) {
                LOG.warn("Interrupted waiting for cleanup thread to stop");
                Thread.currentThread().interrupt();
            }
        }
        if (this.snapshotThread != null) {
            this.snapshotThread.shutdown();
            try {
                this.snapshotThread.join(30000L);
            }
            catch (InterruptedException ie) {
                LOG.warn("Interrupted waiting for snapshot thread to stop");
                Thread.currentThread().interrupt();
            }
        }
        this.persistor.stopAndWait();
        this.txMetricsCollector.stop();
        timer.stop();
        LOG.info("Took " + timer + " to stop");
        this.notifyStopped();
    }

    private void abortService(String message, Throwable error) {
        if (this.isRunning()) {
            LOG.error("Aborting transaction manager due to: " + message, error);
            this.notifyFailed(error);
        }
    }

    private void ensureAvailable() {
        Preconditions.checkState(this.isRunning(), "Transaction Manager is not running.");
    }

    public Transaction startShort() {
        return this.startShort(this.defaultTimeout);
    }

    public Transaction startShort(int timeoutInSeconds) {
        Preconditions.checkArgument(timeoutInSeconds > 0, "timeout must be positive but is %s", timeoutInSeconds);
        this.txMetricsCollector.rate("start.short");
        Stopwatch timer = new Stopwatch().start();
        long expiration = TransactionManager.getTxExpiration(timeoutInSeconds);
        Transaction tx = this.startTx(expiration, TransactionType.SHORT);
        this.txMetricsCollector.histogram("start.short.latency", (int)timer.elapsedMillis());
        return tx;
    }

    private static long getTxExpiration(long timeoutInSeconds) {
        long currentTime = System.currentTimeMillis();
        return currentTime + TimeUnit.SECONDS.toMillis(timeoutInSeconds);
    }

    public static long getTxExpirationFromWritePointer(long writePointer, long timeoutInSeconds) {
        return writePointer / 1000000L + TimeUnit.SECONDS.toMillis(timeoutInSeconds);
    }

    private long getNextWritePointer() {
        return Math.max(this.lastWritePointer + 1L, System.currentTimeMillis() * 1000000L);
    }

    public Transaction startLong() {
        this.txMetricsCollector.rate("start.long");
        Stopwatch timer = new Stopwatch().start();
        long expiration = TransactionManager.getTxExpiration(this.defaultLongTimeout);
        Transaction tx = this.startTx(expiration, TransactionType.LONG);
        this.txMetricsCollector.histogram("start.long.latency", (int)timer.elapsedMillis());
        return tx;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Transaction startTx(long expiration, TransactionType type) {
        Transaction tx = null;
        this.logReadLock.lock();
        try {
            TransactionManager transactionManager = this;
            synchronized (transactionManager) {
                this.ensureAvailable();
                long txid = this.getNextWritePointer();
                tx = this.createTransaction(txid, type);
                this.addInProgressAndAdvance(tx.getTransactionId(), tx.getVisibilityUpperBound(), expiration, type);
            }
            this.appendToLog(TransactionEdit.createStarted(tx.getTransactionId(), tx.getVisibilityUpperBound(), expiration, type));
        }
        finally {
            this.logReadLock.unlock();
        }
        return tx;
    }

    private void addInProgressAndAdvance(long writePointer, long visibilityUpperBound, long expiration, TransactionType type) {
        this.inProgress.put(writePointer, new InProgressTx(visibilityUpperBound, expiration, type));
        this.advanceWritePointer(writePointer);
    }

    private void advanceWritePointer(long writePointer) {
        if (writePointer > this.lastWritePointer) {
            this.lastWritePointer = writePointer;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean canCommit(Transaction tx, Collection<byte[]> changeIds) throws TransactionNotInProgressException {
        this.txMetricsCollector.rate("canCommit");
        Stopwatch timer = new Stopwatch().start();
        if (this.inProgress.get(tx.getTransactionId()) == null) {
            if (this.invalid.contains(tx.getTransactionId())) {
                throw new TransactionNotInProgressException(String.format("canCommit() is called for transaction %d that is not in progress (it is known to be invalid)", tx.getTransactionId()));
            }
            throw new TransactionNotInProgressException(String.format("canCommit() is called for transaction %d that is not in progress", tx.getTransactionId()));
        }
        HashSet<ChangeId> set = Sets.newHashSetWithExpectedSize(changeIds.size());
        for (byte[] change : changeIds) {
            set.add(new ChangeId(change));
        }
        if (this.hasConflicts(tx, set)) {
            return false;
        }
        this.logReadLock.lock();
        try {
            TransactionManager transactionManager = this;
            synchronized (transactionManager) {
                this.ensureAvailable();
                this.addCommittingChangeSet(tx.getTransactionId(), set);
            }
            this.appendToLog(TransactionEdit.createCommitting(tx.getTransactionId(), set));
        }
        finally {
            this.logReadLock.unlock();
        }
        this.txMetricsCollector.histogram("canCommit.latency", (int)timer.elapsedMillis());
        return true;
    }

    private void addCommittingChangeSet(long writePointer, Set<ChangeId> changes) {
        this.committingChangeSets.put(writePointer, changes);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean commit(Transaction tx) throws TransactionNotInProgressException {
        this.txMetricsCollector.rate("commit");
        Stopwatch timer = new Stopwatch().start();
        Set<ChangeId> changeSet = null;
        boolean addToCommitted = true;
        this.logReadLock.lock();
        try {
            long commitPointer;
            TransactionManager transactionManager = this;
            synchronized (transactionManager) {
                this.ensureAvailable();
                commitPointer = this.lastWritePointer + 1L;
                if (this.inProgress.get(tx.getTransactionId()) == null) {
                    if (this.invalid.contains(tx.getTransactionId())) {
                        throw new TransactionNotInProgressException(String.format("canCommit() is called for transaction %d that is not in progress (it is known to be invalid)", tx.getTransactionId()));
                    }
                    throw new TransactionNotInProgressException(String.format("canCommit() is called for transaction %d that is not in progress", tx.getTransactionId()));
                }
                changeSet = this.committingChangeSets.remove(tx.getTransactionId());
                if (changeSet != null) {
                    if (this.hasConflicts(tx, changeSet)) {
                        boolean bl = false;
                        return bl;
                    }
                } else {
                    addToCommitted = false;
                }
                this.doCommit(tx.getTransactionId(), changeSet, commitPointer, addToCommitted);
            }
            this.appendToLog(TransactionEdit.createCommitted(tx.getTransactionId(), changeSet, commitPointer, addToCommitted));
        }
        finally {
            this.logReadLock.unlock();
        }
        this.txMetricsCollector.histogram("commit.latency", (int)timer.elapsedMillis());
        return true;
    }

    private void doCommit(long writePointer, Set<ChangeId> changes, long commitPointer, boolean addToCommitted) {
        InProgressTx previous;
        this.committingChangeSets.remove(writePointer);
        if (addToCommitted && !changes.isEmpty()) {
            Set changeIds = (Set)this.committedChangeSets.get(commitPointer);
            if (changeIds != null) {
                changes.addAll(changeIds);
            }
            this.committedChangeSets.put(commitPointer, changes);
        }
        if ((previous = (InProgressTx)this.inProgress.remove(writePointer)) == null && this.invalid.rem(writePointer)) {
            this.invalidArray = this.invalid.toLongArray();
            LOG.info("Tx invalid list: removed committed tx {}", (Object)writePointer);
        }
        this.moveReadPointerIfNeeded(writePointer);
        this.committedChangeSets.headMap(TxUtils.getFirstShortInProgress(this.inProgress)).clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void abort(Transaction tx) {
        this.txMetricsCollector.rate("abort");
        Stopwatch timer = new Stopwatch().start();
        this.logReadLock.lock();
        try {
            TransactionManager transactionManager = this;
            synchronized (transactionManager) {
                this.ensureAvailable();
                this.doAbort(tx.getTransactionId(), tx.getCheckpointWritePointers(), tx.getType());
            }
            this.appendToLog(TransactionEdit.createAborted(tx.getTransactionId(), tx.getType(), tx.getCheckpointWritePointers()));
            this.txMetricsCollector.histogram("abort.latency", (int)timer.elapsedMillis());
        }
        finally {
            this.logReadLock.unlock();
        }
    }

    private void doAbort(long writePointer, long[] checkpointWritePointers, TransactionType type) {
        this.committingChangeSets.remove(writePointer);
        if (type == TransactionType.LONG) {
            this.doInvalidate(writePointer);
            return;
        }
        InProgressTx removed = (InProgressTx)this.inProgress.remove(writePointer);
        if (removed == null) {
            if (this.invalid.rem(writePointer)) {
                if (checkpointWritePointers != null) {
                    for (int i = 0; i < checkpointWritePointers.length; ++i) {
                        this.invalid.rem(checkpointWritePointers[i]);
                    }
                }
                this.invalidArray = this.invalid.toLongArray();
                LOG.info("Tx invalid list: removed aborted tx {}", (Object)writePointer);
                this.moveReadPointerIfNeeded(writePointer);
            }
        } else {
            this.moveReadPointerIfNeeded(writePointer);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean invalidate(long tx) {
        this.txMetricsCollector.rate("invalidate");
        Stopwatch timer = new Stopwatch().start();
        this.logReadLock.lock();
        try {
            boolean success;
            TransactionManager transactionManager = this;
            synchronized (transactionManager) {
                this.ensureAvailable();
                success = this.doInvalidate(tx);
            }
            this.appendToLog(TransactionEdit.createInvalid(tx));
            this.txMetricsCollector.histogram("invalidate.latency", (int)timer.elapsedMillis());
            boolean bl = success;
            return bl;
        }
        finally {
            this.logReadLock.unlock();
        }
    }

    private boolean doInvalidate(long writePointer) {
        Set<ChangeId> previousChangeSet = this.committingChangeSets.remove(writePointer);
        InProgressTx previous = (InProgressTx)this.inProgress.remove(writePointer);
        if (previous != null || previousChangeSet != null) {
            this.invalid.add(writePointer);
            if (previous == null) {
                LOG.debug("Invalidating tx {} in committing change sets but not in-progress", (Object)writePointer);
            } else {
                LongArrayList childWritePointers = previous.getCheckpointWritePointers();
                if (childWritePointers != null) {
                    for (int i = 0; i < childWritePointers.size(); ++i) {
                        this.invalid.add(childWritePointers.get(i));
                    }
                }
            }
            LOG.info("Tx invalid list: added tx {} because of invalidate", (Object)writePointer);
            Collections.sort(this.invalid);
            this.invalidArray = this.invalid.toLongArray();
            if (previous != null && !previous.isLongRunning()) {
                this.moveReadPointerIfNeeded(writePointer);
            }
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean truncateInvalidTx(Set<Long> invalidTxIds) {
        this.txMetricsCollector.rate("truncateInvalidTx");
        Stopwatch timer = new Stopwatch().start();
        this.logReadLock.lock();
        try {
            boolean success;
            TransactionManager transactionManager = this;
            synchronized (transactionManager) {
                this.ensureAvailable();
                success = this.doTruncateInvalidTx(invalidTxIds);
            }
            this.appendToLog(TransactionEdit.createTruncateInvalidTx(invalidTxIds));
            this.txMetricsCollector.histogram("truncateInvalidTx.latency", (int)timer.elapsedMillis());
            boolean bl = success;
            return bl;
        }
        finally {
            this.logReadLock.unlock();
        }
    }

    private boolean doTruncateInvalidTx(Set<Long> invalidTxIds) {
        LOG.info("Removing tx ids {} from invalid list", (Object)invalidTxIds);
        boolean success = this.invalid.removeAll(invalidTxIds);
        if (success) {
            this.invalidArray = this.invalid.toLongArray();
        }
        return success;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean truncateInvalidTxBefore(long time) throws InvalidTruncateTimeException {
        this.txMetricsCollector.rate("truncateInvalidTxBefore");
        Stopwatch timer = new Stopwatch().start();
        this.logReadLock.lock();
        try {
            boolean success;
            TransactionManager transactionManager = this;
            synchronized (transactionManager) {
                this.ensureAvailable();
                success = this.doTruncateInvalidTxBefore(time);
            }
            this.appendToLog(TransactionEdit.createTruncateInvalidTxBefore(time));
            this.txMetricsCollector.histogram("truncateInvalidTxBefore.latency", (int)timer.elapsedMillis());
            boolean bl = success;
            return bl;
        }
        finally {
            this.logReadLock.unlock();
        }
    }

    private boolean doTruncateInvalidTxBefore(long time) throws InvalidTruncateTimeException {
        long wp;
        LOG.info("Removing tx ids before {} from invalid list", (Object)time);
        long truncateWp = time * 1000000L;
        if (this.inProgress.lowerKey(truncateWp) != null) {
            throw new InvalidTruncateTimeException("Transactions started earlier than " + time + " are in-progress");
        }
        HashSet<Long> toTruncate = Sets.newHashSet();
        Iterator i$ = this.invalid.iterator();
        while (i$.hasNext() && (wp = ((Long)i$.next()).longValue()) < truncateWp) {
            toTruncate.add(wp);
        }
        return this.doTruncateInvalidTx(toTruncate);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Transaction checkpoint(Transaction originalTx) throws TransactionNotInProgressException {
        this.txMetricsCollector.rate("checkpoint");
        Stopwatch timer = new Stopwatch().start();
        Transaction checkpointedTx = null;
        long txId = originalTx.getTransactionId();
        long newWritePointer = 0L;
        this.logReadLock.lock();
        try {
            TransactionManager transactionManager = this;
            synchronized (transactionManager) {
                this.ensureAvailable();
                InProgressTx parentTx = (InProgressTx)this.inProgress.get(txId);
                if (parentTx == null) {
                    if (this.invalid.contains(txId)) {
                        throw new TransactionNotInProgressException(String.format("Transaction %d is not in progress because it was invalidated", txId));
                    }
                    throw new TransactionNotInProgressException(String.format("Transaction %d is not in progress", txId));
                }
                newWritePointer = this.getNextWritePointer();
                this.doCheckpoint(newWritePointer, txId);
                checkpointedTx = new Transaction(originalTx, newWritePointer, parentTx.getCheckpointWritePointers().toLongArray());
            }
            this.appendToLog(TransactionEdit.createCheckpoint(newWritePointer, txId));
        }
        finally {
            this.logReadLock.unlock();
        }
        this.txMetricsCollector.histogram("checkpoint.latency", (int)timer.elapsedMillis());
        return checkpointedTx;
    }

    private void doCheckpoint(long newWritePointer, long parentWritePointer) {
        InProgressTx existingTx = (InProgressTx)this.inProgress.get(parentWritePointer);
        existingTx.addCheckpointWritePointer(newWritePointer);
        this.advanceWritePointer(newWritePointer);
    }

    public int getExcludedListSize() {
        return this.invalid.size() + this.inProgress.size();
    }

    public int getInvalidSize() {
        return this.invalid.size();
    }

    int getCommittedSize() {
        return this.committedChangeSets.size();
    }

    private boolean hasConflicts(Transaction tx, Set<ChangeId> changeIds) {
        if (changeIds.isEmpty()) {
            return false;
        }
        for (Map.Entry changeSet : this.committedChangeSets.entrySet()) {
            if ((Long)changeSet.getKey() <= tx.getTransactionId() || !this.overlap((Set)changeSet.getValue(), changeIds)) continue;
            return true;
        }
        return false;
    }

    private boolean overlap(Set<ChangeId> a, Set<ChangeId> b) {
        if (a.size() > b.size()) {
            for (ChangeId change : b) {
                if (!a.contains(change)) continue;
                return true;
            }
        } else {
            for (ChangeId change : a) {
                if (!b.contains(change)) continue;
                return true;
            }
        }
        return false;
    }

    private void moveReadPointerIfNeeded(long committedWritePointer) {
        if (committedWritePointer > this.readPointer) {
            this.readPointer = committedWritePointer;
        }
    }

    private Transaction createTransaction(long writePointer, TransactionType type) {
        long firstShortTx = Long.MAX_VALUE;
        LongArrayList inProgressIds = new LongArrayList(this.inProgress.size());
        for (Map.Entry entry : this.inProgress.entrySet()) {
            long txId = (Long)entry.getKey();
            inProgressIds.add(txId);
            LongArrayList childIds = ((InProgressTx)entry.getValue()).getCheckpointWritePointers();
            if (childIds != null) {
                for (int i = 0; i < childIds.size(); ++i) {
                    inProgressIds.add(childIds.get(i));
                }
            }
            if (firstShortTx != Long.MAX_VALUE || ((InProgressTx)entry.getValue()).isLongRunning()) continue;
            firstShortTx = txId;
        }
        return new Transaction(this.readPointer, writePointer, this.invalidArray, inProgressIds.toLongArray(), firstShortTx, type);
    }

    private void appendToLog(TransactionEdit edit) {
        try {
            Stopwatch timer = new Stopwatch().start();
            this.currentLog.append(edit);
            this.txMetricsCollector.rate("wal.append.count");
            this.txMetricsCollector.histogram("wal.append.latency", (int)timer.elapsedMillis());
        }
        catch (IOException ioe) {
            this.abortService("Error appending to transaction log", ioe);
        }
    }

    private void appendToLog(List<TransactionEdit> edits) {
        try {
            Stopwatch timer = new Stopwatch().start();
            this.currentLog.append(edits);
            this.txMetricsCollector.rate("wal.append.count", edits.size());
            this.txMetricsCollector.histogram("wal.append.latency", (int)timer.elapsedMillis());
        }
        catch (IOException ioe) {
            this.abortService("Error appending to transaction log", ioe);
        }
    }

    public void logStatistics() {
        LOG.info("Transaction Statistics: write pointer = " + this.lastWritePointer + ", invalid = " + this.invalid.size() + ", in progress = " + this.inProgress.size() + ", committing = " + this.committingChangeSets.size() + ", committed = " + this.committedChangeSets.size());
    }

    public static final class InProgressTx {
        private final long visibilityUpperBound;
        private final long expiration;
        private final TransactionType type;
        private LongArrayList checkpointWritePointers = new LongArrayList();

        public InProgressTx(long visibilityUpperBound, long expiration, TransactionType type) {
            this(visibilityUpperBound, expiration, type, new LongArrayList());
        }

        public InProgressTx(long visibilityUpperBound, long expiration, TransactionType type, LongArrayList checkpointWritePointers) {
            this.visibilityUpperBound = visibilityUpperBound;
            this.expiration = expiration;
            this.type = type;
            this.checkpointWritePointers = checkpointWritePointers;
        }

        @Deprecated
        public InProgressTx(long visibilityUpperBound, long expiration) {
            this(visibilityUpperBound, expiration, null);
        }

        public long getVisibilityUpperBound() {
            return this.visibilityUpperBound;
        }

        public long getExpiration() {
            return this.expiration;
        }

        @Nullable
        public TransactionType getType() {
            return this.type;
        }

        public boolean isLongRunning() {
            if (this.type == null) {
                return this.expiration == -1L;
            }
            return this.type == TransactionType.LONG;
        }

        public void addCheckpointWritePointer(long checkpointWritePointer) {
            this.checkpointWritePointers.add(checkpointWritePointer);
        }

        public LongArrayList getCheckpointWritePointers() {
            return this.checkpointWritePointers;
        }

        public boolean equals(Object o) {
            if (o == null || !(o instanceof InProgressTx)) {
                return false;
            }
            if (this == o) {
                return true;
            }
            InProgressTx other = (InProgressTx)o;
            return Objects.equal(this.visibilityUpperBound, other.getVisibilityUpperBound()) && Objects.equal(this.expiration, other.getExpiration()) && Objects.equal((Object)this.type, (Object)other.type) && Objects.equal(this.checkpointWritePointers, other.checkpointWritePointers);
        }

        public int hashCode() {
            return Objects.hashCode(new Object[]{this.visibilityUpperBound, this.expiration, this.type, this.checkpointWritePointers});
        }

        public String toString() {
            return Objects.toStringHelper(this).add("visibilityUpperBound", this.visibilityUpperBound).add("expiration", this.expiration).add("type", (Object)this.type).toString();
        }
    }

    private static abstract class DaemonThreadExecutor
    extends Thread {
        private AtomicBoolean stopped = new AtomicBoolean(false);

        public DaemonThreadExecutor(String name) {
            super(name);
            this.setDaemon(true);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        public void run() {
            try {
                while (!this.isInterrupted() && !this.stopped.get()) {
                    this.doRun();
                    AtomicBoolean atomicBoolean = this.stopped;
                    synchronized (atomicBoolean) {
                        this.stopped.wait(this.getSleepMillis());
                    }
                }
            }
            catch (InterruptedException ie) {
                LOG.info("Interrupted thread " + this.getName());
            }
            this.onShutdown();
            LOG.info("Exiting thread " + this.getName());
        }

        public abstract void doRun();

        protected abstract long getSleepMillis();

        protected void onShutdown() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void shutdown() {
            if (this.stopped.compareAndSet(false, true)) {
                AtomicBoolean atomicBoolean = this.stopped;
                synchronized (atomicBoolean) {
                    this.stopped.notifyAll();
                }
            }
        }
    }
}

