/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.controller.rebalancer.topology;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.helix.HelixException;
import org.apache.helix.controller.rebalancer.topology.InstanceNode;
import org.apache.helix.controller.rebalancer.topology.Node;
import org.apache.helix.model.ClusterConfig;
import org.apache.helix.model.ClusterTopologyConfig;
import org.apache.helix.model.InstanceConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Topology {
    private static Logger logger = LoggerFactory.getLogger(Topology.class);
    private static final int DEFAULT_NODE_WEIGHT = 1000;
    private final MessageDigest _md;
    private final Node _root;
    private final List<String> _allInstances;
    private final List<String> _liveInstances;
    private final Map<String, InstanceConfig> _instanceConfigMap;
    private final ClusterTopologyConfig _clusterTopologyConfig;

    public Topology(List<String> allNodes, List<String> liveNodes, Map<String, InstanceConfig> instanceConfigMap, ClusterConfig clusterConfig) {
        try {
            this._md = MessageDigest.getInstance("SHA-1");
        }
        catch (NoSuchAlgorithmException ex) {
            throw new IllegalArgumentException(ex);
        }
        this._allInstances = allNodes;
        this._liveInstances = liveNodes;
        this._instanceConfigMap = instanceConfigMap;
        if (this._instanceConfigMap == null || !this._instanceConfigMap.keySet().containsAll(allNodes)) {
            throw new HelixException(String.format("Config for instances %s is not found!", this._allInstances.removeAll(this._instanceConfigMap.keySet())));
        }
        this._clusterTopologyConfig = ClusterTopologyConfig.createFromClusterConfig(clusterConfig);
        this._root = this.createClusterTree(clusterConfig);
    }

    public String getEndNodeType() {
        return this._clusterTopologyConfig.getEndNodeType();
    }

    public String getFaultZoneType() {
        return this._clusterTopologyConfig.getFaultZoneType();
    }

    public Node getRootNode() {
        return this._root;
    }

    public List<Node> getFaultZones() {
        if (this._root != null) {
            return this._root.findChildren(this.getFaultZoneType());
        }
        return Collections.emptyList();
    }

    public static List<Node> getAllLeafNodes(Node root) {
        ArrayList<Node> nodes = new ArrayList<Node>();
        if (root.isLeaf()) {
            nodes.add(root);
        } else {
            for (Node child : root.getChildren()) {
                nodes.addAll(Topology.getAllLeafNodes(child));
            }
        }
        return nodes;
    }

    public static Node clone(Node root, Map<Node, Integer> newNodeWeight, Set<Node> failedNodes) {
        Node newRoot = Topology.cloneTree(root, newNodeWeight, failedNodes);
        Topology.computeWeight(newRoot);
        return newRoot;
    }

    private static Node cloneTree(Node root, Map<Node, Integer> newNodeWeight, Set<Node> failedNodes) {
        List<Node> children;
        Node newRoot = root.clone();
        if (newNodeWeight.containsKey(root)) {
            newRoot.setWeight(newNodeWeight.get(root).intValue());
        }
        if (failedNodes.contains(root)) {
            newRoot.setFailed(true);
            newRoot.setWeight(0L);
        }
        if ((children = root.getChildren()) != null) {
            for (Node child : children) {
                Node newChild = Topology.cloneTree(child, newNodeWeight, failedNodes);
                newChild.setParent(root);
                newRoot.addChild(newChild);
            }
        }
        return newRoot;
    }

    private Node createClusterTree(ClusterConfig clusterConfig) {
        Node root = new Node();
        root.setName("root");
        root.setId(this.computeId("root"));
        root.setType(Types.ROOT.name());
        for (String instanceName : this._allInstances) {
            InstanceConfig insConfig = this._instanceConfigMap.get(instanceName);
            try {
                LinkedHashMap<String, String> instanceTopologyMap = Topology.computeInstanceTopologyMapHelper(this._clusterTopologyConfig, instanceName, insConfig, null);
                int weight = insConfig.getWeight();
                if (weight < 0 || weight == -1) {
                    weight = 1000;
                }
                this.addEndNode(root, instanceName, instanceTopologyMap, weight, this._liveInstances);
            }
            catch (IllegalArgumentException e) {
                if (Topology.isInstanceEnabled(clusterConfig, instanceName, insConfig)) {
                    throw e;
                }
                logger.warn("Topology setting {} for instance {} is unset or invalid, ignore the instance!", (Object)insConfig.getDomainAsString(), (Object)instanceName);
            }
        }
        return root;
    }

    private static boolean isInstanceEnabled(ClusterConfig clusterConfig, String instanceName, InstanceConfig instanceConfig) {
        return instanceConfig.getInstanceEnabled() && (clusterConfig.getDisabledInstances() == null || !clusterConfig.getDisabledInstances().containsKey(instanceName));
    }

    private static LinkedHashMap<String, String> computeInstanceTopologyMapHelper(ClusterTopologyConfig clusterTopologyConfig, String instanceName, InstanceConfig instanceConfig, String faultZoneForEarlyQuit) throws IllegalArgumentException {
        LinkedHashMap<String, String> instanceTopologyMap = new LinkedHashMap<String, String>();
        if (clusterTopologyConfig.isTopologyAwareEnabled()) {
            if (clusterTopologyConfig.getTopologyKeyDefaultValue().isEmpty()) {
                String zone = instanceConfig.getZoneId();
                if (zone == null) {
                    throw new IllegalArgumentException(String.format("ZONE_ID for instance %s is not set, fail the topology-aware placement!", instanceName));
                }
                instanceTopologyMap.put(Types.ZONE.name(), zone);
                if (faultZoneForEarlyQuit != null) {
                    return instanceTopologyMap;
                }
                instanceTopologyMap.put(Types.INSTANCE.name(), instanceName);
            } else {
                Map<String, String> domainAsMap = instanceConfig.getDomainAsMap();
                if (domainAsMap.isEmpty()) {
                    throw new IllegalArgumentException(String.format("Domain for instance %s is not set, fail the topology-aware placement!", instanceName));
                }
                int numOfMatchedKeys = 0;
                for (String key : clusterTopologyConfig.getTopologyKeyDefaultValue().keySet()) {
                    String value = domainAsMap.get(key);
                    if (value == null || value.length() == 0) {
                        value = clusterTopologyConfig.getTopologyKeyDefaultValue().get(key);
                    } else {
                        ++numOfMatchedKeys;
                    }
                    instanceTopologyMap.put(key, value);
                    if (!key.equals(faultZoneForEarlyQuit)) continue;
                    return instanceTopologyMap;
                }
                if (numOfMatchedKeys != domainAsMap.size()) {
                    logger.warn("Key-value pairs in InstanceConfig.Domain {} do not align with keys in ClusterConfig.Topology {}, using default domain value instead", (Object)instanceConfig.getDomainAsString(), clusterTopologyConfig.getTopologyKeyDefaultValue().keySet());
                }
            }
        } else {
            instanceTopologyMap.put(Types.INSTANCE.name(), instanceName);
        }
        return instanceTopologyMap;
    }

    public static LinkedHashMap<String, String> computeInstanceTopologyMap(ClusterConfig clusterConfig, String instanceName, InstanceConfig instanceConfig, boolean earlyQuitForFaultZone) {
        ClusterTopologyConfig clusterTopologyConfig = ClusterTopologyConfig.createFromClusterConfig(clusterConfig);
        String faultZoneForEarlyQuit = earlyQuitForFaultZone ? clusterTopologyConfig.getFaultZoneType() : null;
        return Topology.computeInstanceTopologyMapHelper(clusterTopologyConfig, instanceName, instanceConfig, faultZoneForEarlyQuit);
    }

    private void addEndNode(Node root, String instanceName, LinkedHashMap<String, String> pathNameMap, int instanceWeight, List<String> liveInstances) {
        Node current = root;
        ArrayList<Node> pathNodes = new ArrayList<Node>();
        for (Map.Entry<String, String> entry : pathNameMap.entrySet()) {
            String pathValue = entry.getValue();
            String path = entry.getKey();
            pathNodes.add(current);
            if (!current.hasChild(pathValue)) {
                this.buildNewNode(pathValue, path, current, instanceName, instanceWeight, liveInstances.contains(instanceName), pathNodes);
            } else if (path.equals(this._clusterTopologyConfig.getEndNodeType())) {
                throw new HelixException("Failed to add topology node because duplicate leaf nodes are not allowed. Duplicate node name: " + pathValue);
            }
            current = current.getChild(pathValue);
        }
    }

    private Node buildNewNode(String name, String type, Node parent, String instanceName, int instanceWeight, boolean isLiveInstance, List<Node> pathNodes) {
        Node n = new Node();
        n.setName(name);
        n.setId(this.computeId(name));
        n.setType(type);
        n.setParent(parent);
        if (type.equals(this._clusterTopologyConfig.getEndNodeType())) {
            n = new InstanceNode(n, instanceName);
            if (isLiveInstance) {
                n.setWeight(instanceWeight);
                for (Node node : pathNodes) {
                    node.addWeight(instanceWeight);
                }
            } else {
                n.setFailed(true);
                n.setWeight(0L);
            }
        }
        parent.addChild(n);
        return n;
    }

    private long computeId(String name) {
        byte[] h = this._md.digest(name.getBytes());
        return this.bstrTo32bit(h);
    }

    private static void computeWeight(Node node) {
        int weight = 0;
        for (Node child : node.getChildren()) {
            if (child.isFailed()) continue;
            weight = (int)((long)weight + child.getWeight());
        }
        node.setWeight(weight);
    }

    private long bstrTo32bit(byte[] bstr) {
        if (bstr.length < 4) {
            throw new IllegalArgumentException("hashed is less than 4 bytes!");
        }
        return (long)(this.ord(bstr[0]) << 24 | this.ord(bstr[1]) << 16 | this.ord(bstr[2]) << 8 | this.ord(bstr[3])) & 0xFFFFFFFFL;
    }

    private int ord(byte b) {
        return b & 0xFF;
    }

    public static enum Types {
        ROOT,
        ZONE,
        INSTANCE;

    }
}

