/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.task;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.helix.common.caches.TaskDataCache;
import org.apache.helix.controller.stages.CurrentStateOutput;
import org.apache.helix.model.ClusterConfig;
import org.apache.helix.model.InstanceConfig;
import org.apache.helix.model.LiveInstance;
import org.apache.helix.model.Message;
import org.apache.helix.model.Partition;
import org.apache.helix.model.Resource;
import org.apache.helix.task.JobConfig;
import org.apache.helix.task.JobContext;
import org.apache.helix.task.TaskConfig;
import org.apache.helix.task.TaskPartitionState;
import org.apache.helix.task.TaskState;
import org.apache.helix.task.TaskUtil;
import org.apache.helix.task.WorkflowConfig;
import org.apache.helix.task.WorkflowContext;
import org.apache.helix.task.assigner.AssignableInstance;
import org.apache.helix.task.assigner.TaskAssignResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AssignableInstanceManager {
    private static final Logger LOG = LoggerFactory.getLogger(AssignableInstanceManager.class);
    public static final int QUOTA_TYPE_NOT_EXIST = -1;
    private static ObjectMapper mapper = new ObjectMapper();
    private Map<String, AssignableInstance> _assignableInstanceMap = new ConcurrentHashMap<String, AssignableInstance>();
    private Map<String, TaskAssignResult> _taskAssignResultMap = new ConcurrentHashMap<String, TaskAssignResult>();
    private Map<String, Integer> _globalThreadBasedQuotaMap = new ConcurrentHashMap<String, Integer>();

    public void buildAssignableInstances(ClusterConfig clusterConfig, TaskDataCache taskDataCache, Map<String, LiveInstance> liveInstances, Map<String, InstanceConfig> instanceConfigs) {
        this._assignableInstanceMap.clear();
        this._taskAssignResultMap.clear();
        for (Map.Entry<String, LiveInstance> liveInstanceEntry : liveInstances.entrySet()) {
            String instanceName = liveInstanceEntry.getKey();
            LiveInstance liveInstance = liveInstanceEntry.getValue();
            if (!instanceConfigs.containsKey(instanceName)) continue;
            InstanceConfig instanceConfig = instanceConfigs.get(instanceName);
            AssignableInstance assignableInstance = new AssignableInstance(clusterConfig, instanceConfig, liveInstance);
            this._assignableInstanceMap.put(instanceConfig.getInstanceName(), assignableInstance);
            LOG.debug("AssignableInstance created for instance: {}", (Object)instanceName);
        }
        Map<String, JobConfig> jobConfigMap = taskDataCache.getJobConfigMap();
        for (String jobName : jobConfigMap.keySet()) {
            JobConfig jobConfig = jobConfigMap.get(jobName);
            JobContext jobContext = taskDataCache.getJobContext(jobName);
            if (jobConfig == null || jobContext == null) {
                LOG.debug("JobConfig or JobContext for this job is null. Skipping this job! Job name: {}, JobConfig: {}, JobContext: {}", new Object[]{jobName, jobConfig, jobContext});
                continue;
            }
            String workflowName = jobConfig.getWorkflow();
            WorkflowConfig workflowConfig = taskDataCache.getWorkflowConfig(workflowName);
            WorkflowContext workflowContext = taskDataCache.getWorkflowContext(workflowName);
            if (workflowConfig == null || workflowContext == null) continue;
            TaskState workflowState = workflowContext.getWorkflowState();
            TaskState jobState = workflowContext.getJobState(jobName);
            if (this.isResourceTerminalOrStopped(workflowState) || this.isResourceTerminalOrStopped(jobState)) continue;
            String quotaType = jobConfig.getJobType();
            if (quotaType == null) {
                quotaType = "DEFAULT";
            }
            Set<Integer> taskIndices = jobContext.getPartitionSet();
            for (int taskIndex : taskIndices) {
                TaskPartitionState taskState = jobContext.getPartitionState(taskIndex);
                if (taskState != TaskPartitionState.INIT && taskState != TaskPartitionState.RUNNING) continue;
                String assignedInstance = jobContext.getAssignedParticipant(taskIndex);
                String taskId = jobContext.getTaskIdForPartition(taskIndex);
                if (taskId == null) {
                    taskId = String.format("%s_%s", jobConfig.getJobId(), taskIndex);
                }
                if (assignedInstance == null) {
                    LOG.debug("This task's TaskContext does not have an assigned instance! Task will be ignored. Job: {}, TaskId: {}, TaskIndex: {}", new Object[]{jobContext.getName(), taskId, taskIndex});
                    continue;
                }
                if (this._assignableInstanceMap.containsKey(assignedInstance)) {
                    TaskConfig taskConfig = jobConfig.getTaskConfig(taskId);
                    AssignableInstance assignableInstance = this._assignableInstanceMap.get(assignedInstance);
                    TaskAssignResult taskAssignResult = assignableInstance.restoreTaskAssignResult(taskId, taskConfig, quotaType);
                    if (!taskAssignResult.isSuccessful()) continue;
                    this._taskAssignResultMap.put(taskId, taskAssignResult);
                    LOG.debug("TaskAssignResult restored for taskId: {}, assigned on instance: {}", (Object)taskId, (Object)assignedInstance);
                    continue;
                }
                LOG.debug("While building AssignableInstance map, discovered that the instance a task is assigned to is no longer a LiveInstance! TaskAssignResult will not be created and no resource will be taken up for this task. Job: {}, TaskId: {}, TaskIndex: {}, Instance: {}", new Object[]{jobContext.getName(), taskId, taskIndex, assignedInstance});
            }
        }
        LOG.info("AssignableInstanceManager built AssignableInstances from scratch based on contexts in TaskDataCache due to Controller switch or ClusterConfig change.");
        this.computeGlobalThreadBasedCapacity();
    }

    public void buildAssignableInstancesFromCurrentState(ClusterConfig clusterConfig, TaskDataCache taskDataCache, Map<String, LiveInstance> liveInstances, Map<String, InstanceConfig> instanceConfigs, CurrentStateOutput currentStateOutput, Map<String, Resource> resourceMap) {
        this._assignableInstanceMap.clear();
        this._taskAssignResultMap.clear();
        for (Map.Entry<String, LiveInstance> liveInstanceEntry : liveInstances.entrySet()) {
            String instanceName = liveInstanceEntry.getKey();
            LiveInstance liveInstance = liveInstanceEntry.getValue();
            if (!instanceConfigs.containsKey(instanceName)) continue;
            InstanceConfig instanceConfig = instanceConfigs.get(instanceName);
            AssignableInstance assignableInstance = new AssignableInstance(clusterConfig, instanceConfig, liveInstance);
            this._assignableInstanceMap.put(instanceConfig.getInstanceName(), assignableInstance);
            LOG.debug("AssignableInstance created for instance: {}", (Object)instanceName);
        }
        Map<String, JobConfig> jobConfigMap = taskDataCache.getJobConfigMap();
        for (Map.Entry<String, Resource> resourceEntry : resourceMap.entrySet()) {
            String resourceName = resourceEntry.getKey();
            if (!resourceEntry.getValue().getStateModelDefRef().equals("Task")) continue;
            JobConfig jobConfig = jobConfigMap.get(resourceName);
            JobContext jobContext = taskDataCache.getJobContext(resourceName);
            String quotaType = this.getQuotaType(jobConfig);
            Map<Partition, Map<String, String>> currentStateMap = currentStateOutput.getCurrentStateMap(resourceName);
            for (Map.Entry<Partition, Map<String, String>> currentStateMapEntry : currentStateMap.entrySet()) {
                Partition partition = currentStateMapEntry.getKey();
                String taskId = this.getTaskID(jobConfig, jobContext, partition);
                for (Map.Entry<String, String> instanceCurrentStateEntry : currentStateMapEntry.getValue().entrySet()) {
                    String assignedInstance = instanceCurrentStateEntry.getKey();
                    String taskState = instanceCurrentStateEntry.getValue();
                    if (taskState == null) {
                        LOG.warn("CurrentState is null for job {}, task {} on instance {}", new Object[]{resourceName, taskId, assignedInstance});
                    }
                    if (!TaskPartitionState.INIT.name().equals(taskState) && !TaskPartitionState.RUNNING.name().equals(taskState)) continue;
                    this.assignTaskToInstance(assignedInstance, jobConfig, taskId, quotaType);
                }
            }
            Map<Partition, Map<String, Message>> pendingMessageMap = currentStateOutput.getPendingMessageMap(resourceName);
            for (Map.Entry<Partition, Map<String, Message>> pendingMessageMapEntry : pendingMessageMap.entrySet()) {
                Partition partition = pendingMessageMapEntry.getKey();
                String taskId = this.getTaskID(jobConfig, jobContext, partition);
                for (Map.Entry<String, Message> instancePendingMessageEntry : pendingMessageMapEntry.getValue().entrySet()) {
                    String assignedInstance = instancePendingMessageEntry.getKey();
                    String messageToState = instancePendingMessageEntry.getValue().getToState();
                    if (!TaskPartitionState.RUNNING.name().equals(messageToState) || TaskPartitionState.INIT.name().equals(currentStateOutput.getCurrentState(resourceName, partition, assignedInstance)) || TaskPartitionState.RUNNING.name().equals(currentStateOutput.getCurrentState(resourceName, partition, assignedInstance))) continue;
                    this.assignTaskToInstance(assignedInstance, jobConfig, taskId, quotaType);
                }
            }
        }
        LOG.info("AssignableInstanceManager built AssignableInstances from scratch based on CurrentState.");
        this.computeGlobalThreadBasedCapacity();
    }

    private void assignTaskToInstance(String instance, JobConfig jobConfig, String taskId, String quotaType) {
        if (this._assignableInstanceMap.containsKey(instance)) {
            TaskConfig taskConfig = this.getTaskConfig(jobConfig, taskId);
            AssignableInstance assignableInstance = this._assignableInstanceMap.get(instance);
            TaskAssignResult taskAssignResult = assignableInstance.restoreTaskAssignResult(taskId, taskConfig, quotaType);
            if (taskAssignResult.isSuccessful()) {
                this._taskAssignResultMap.put(taskId, taskAssignResult);
                LOG.debug("TaskAssignResult restored for taskId: {}, assigned on instance: {}", (Object)taskId, (Object)instance);
            }
        } else {
            LOG.debug("While building AssignableInstance map, discovered that the instance a task is assigned to is no longer a LiveInstance! TaskAssignResult will not be created and no resource will be taken up for this task. TaskId: {}, Instance: {}", (Object)taskId, (Object)instance);
        }
    }

    private String getQuotaType(JobConfig jobConfig) {
        if (jobConfig == null || jobConfig.getJobType() == null) {
            return "DEFAULT";
        }
        return jobConfig.getJobType();
    }

    private String getTaskID(JobConfig jobConfig, JobContext jobContext, Partition partition) {
        if (jobConfig == null || jobContext == null) {
            return partition.getPartitionName();
        }
        int taskIndex = TaskUtil.getPartitionId(partition.getPartitionName());
        String taskId = jobContext.getTaskIdForPartition(taskIndex);
        if (taskId == null) {
            taskId = String.format("%s_%s", jobConfig.getJobId(), taskIndex);
        }
        return taskId;
    }

    private TaskConfig getTaskConfig(JobConfig jobConfig, String taskId) {
        if (jobConfig == null) {
            return new TaskConfig(null, null, taskId, null);
        }
        return jobConfig.getTaskConfig(taskId);
    }

    public void updateAssignableInstances(ClusterConfig clusterConfig, Map<String, LiveInstance> liveInstances, Map<String, InstanceConfig> instanceConfigs) {
        HashSet<AssignableInstance> staleAssignableInstances = new HashSet<AssignableInstance>(this._assignableInstanceMap.values());
        for (Map.Entry<String, LiveInstance> liveInstanceEntry : liveInstances.entrySet()) {
            String instanceName = liveInstanceEntry.getKey();
            LiveInstance liveInstance = liveInstanceEntry.getValue();
            if (!instanceConfigs.containsKey(instanceName)) continue;
            InstanceConfig instanceConfig = instanceConfigs.get(instanceName);
            if (this._assignableInstanceMap.containsKey(instanceName)) {
                this._assignableInstanceMap.get(instanceName).updateConfigs(clusterConfig, instanceConfig, liveInstance);
            } else {
                AssignableInstance assignableInstance = new AssignableInstance(clusterConfig, instanceConfig, liveInstance);
                this._assignableInstanceMap.put(instanceName, assignableInstance);
                LOG.debug("AssignableInstance created for instance: {} during updateAssignableInstances", (Object)instanceName);
            }
            staleAssignableInstances.remove(this._assignableInstanceMap.get(instanceName));
        }
        for (AssignableInstance instanceToBeRemoved : staleAssignableInstances) {
            for (String taskToRemove : instanceToBeRemoved.getCurrentAssignments()) {
                if (!this._taskAssignResultMap.containsKey(taskToRemove) || !this._taskAssignResultMap.get(taskToRemove).getAssignableInstance().getInstanceName().equals(instanceToBeRemoved.getInstanceName())) continue;
                this._taskAssignResultMap.remove(taskToRemove);
                LOG.debug("TaskAssignResult removed because its assigned instance is no longer live. TaskID: {}, instance: {}", (Object)taskToRemove, (Object)instanceToBeRemoved.getInstanceName());
            }
            this._assignableInstanceMap.remove(instanceToBeRemoved.getInstanceName());
            LOG.debug("Non-live AssignableInstance removed for instance: {} during updateAssignableInstances", (Object)instanceToBeRemoved.getInstanceName());
        }
        LOG.info("AssignableInstanceManager updated AssignableInstances due to LiveInstance/InstanceConfig change.");
        this.computeGlobalThreadBasedCapacity();
    }

    public Map<String, AssignableInstance> getAssignableInstanceMap() {
        return Collections.unmodifiableMap(this._assignableInstanceMap);
    }

    public AssignableInstance getAssignableInstance(String instanceName) {
        return this._assignableInstanceMap.get(instanceName);
    }

    public Set<AssignableInstance> getAssignableInstancesForQuotaType(String quotaType) {
        return Collections.unmodifiableSet(new HashSet<AssignableInstance>(this._assignableInstanceMap.values()));
    }

    public Map<String, TaskAssignResult> getTaskAssignResultMap() {
        return this._taskAssignResultMap;
    }

    public boolean hasGlobalCapacity(String quotaType) {
        return this._globalThreadBasedQuotaMap.containsKey(quotaType) && this._globalThreadBasedQuotaMap.get(quotaType) > 0;
    }

    public boolean hasQuotaType(String quotaType) {
        return this._globalThreadBasedQuotaMap.containsKey(quotaType);
    }

    public void release(String instanceName, TaskConfig taskConfig, String quotaType) {
        if (quotaType == null) {
            LOG.debug("Task {}'s quotaType is null. Trying to release as DEFAULT type.", (Object)taskConfig.getId());
            quotaType = "DEFAULT";
        }
        if (this._assignableInstanceMap.containsKey(instanceName)) {
            this._assignableInstanceMap.get(instanceName).release(taskConfig, quotaType);
        }
        if (this._globalThreadBasedQuotaMap.containsKey(quotaType)) {
            this._globalThreadBasedQuotaMap.put(quotaType, this._globalThreadBasedQuotaMap.get(quotaType) + 1);
        }
    }

    public TaskAssignResult tryAssign(String instanceName, TaskConfig task, String quotaType) throws IllegalArgumentException {
        if (this._assignableInstanceMap.containsKey(instanceName)) {
            return this._assignableInstanceMap.get(instanceName).tryAssign(task, quotaType);
        }
        return null;
    }

    public void assign(String instanceName, TaskAssignResult result) throws IllegalStateException {
        if (result != null && this._assignableInstanceMap.containsKey(instanceName)) {
            this._assignableInstanceMap.get(instanceName).assign(result);
            this._taskAssignResultMap.put(result.getTaskConfig().getId(), result);
        }
        if (this._globalThreadBasedQuotaMap.containsKey(result.getQuotaType())) {
            this._globalThreadBasedQuotaMap.put(result.getQuotaType(), this._globalThreadBasedQuotaMap.get(result.getQuotaType()) - 1);
        }
    }

    public Set<String> getAssignableInstanceNames() {
        return Collections.unmodifiableSet(this._assignableInstanceMap.keySet());
    }

    private boolean isResourceTerminalOrStopped(TaskState state) {
        if (state == null) {
            return true;
        }
        switch (state) {
            case ABORTED: 
            case FAILED: 
            case STOPPED: 
            case COMPLETED: 
            case TIMED_OUT: 
            case NOT_STARTED: {
                return true;
            }
        }
        return false;
    }

    public void logQuotaProfileJSON(boolean onlyDisplayIfFull) {
        ObjectNode instanceNode = mapper.createObjectNode();
        for (Map.Entry<String, AssignableInstance> instanceEntry : this._assignableInstanceMap.entrySet()) {
            AssignableInstance assignableInstance = instanceEntry.getValue();
            boolean capacityFull = false;
            ObjectNode resourceTypeNode = mapper.createObjectNode();
            for (Map.Entry<String, Map<String, Integer>> capacityEntry : assignableInstance.getTotalCapacity().entrySet()) {
                String resourceType = capacityEntry.getKey();
                Map<String, Integer> quotaTypeMap = capacityEntry.getValue();
                ObjectNode quotaTypeNode = mapper.createObjectNode();
                for (Map.Entry<String, Integer> typeEntry : quotaTypeMap.entrySet()) {
                    String quotaType = typeEntry.getKey();
                    int totalCapacity = typeEntry.getValue();
                    int usedCapacity = assignableInstance.getUsedCapacity().get(resourceType).get(quotaType);
                    if (!capacityFull) {
                        capacityFull = totalCapacity <= usedCapacity;
                    }
                    String capacityString = String.format("%d/%d", usedCapacity, totalCapacity);
                    quotaTypeNode.put(quotaType, capacityString);
                }
                resourceTypeNode.put(resourceType, (JsonNode)quotaTypeNode);
            }
            if (onlyDisplayIfFull && !capacityFull) continue;
            instanceNode.put(instanceEntry.getKey(), (JsonNode)resourceTypeNode);
        }
        if (instanceNode.size() > 0) {
            LOG.info("Current quota capacity: {}", (Object)instanceNode.toString());
        }
    }

    private void computeGlobalThreadBasedCapacity() {
        this._globalThreadBasedQuotaMap.clear();
        for (AssignableInstance assignableInstance : this._assignableInstanceMap.values()) {
            Map<String, Map<String, Integer>> capacityMap = assignableInstance.getTotalCapacity();
            for (Map.Entry<String, Integer> entry : capacityMap.get(LiveInstance.InstanceResourceType.TASK_EXEC_THREAD.name()).entrySet()) {
                int value = entry.getValue();
                if (this._globalThreadBasedQuotaMap.containsKey(entry.getKey())) {
                    value += this._globalThreadBasedQuotaMap.get(entry.getKey()).intValue();
                }
                this._globalThreadBasedQuotaMap.put(entry.getKey(), value);
            }
        }
    }
}

