/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.controller.stages;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.stream.Collectors;
import org.apache.helix.HelixException;
import org.apache.helix.HelixManager;
import org.apache.helix.HelixRebalanceException;
import org.apache.helix.controller.LogUtil;
import org.apache.helix.controller.dataproviders.ResourceControllerDataProvider;
import org.apache.helix.controller.pipeline.AbstractBaseStage;
import org.apache.helix.controller.pipeline.StageException;
import org.apache.helix.controller.rebalancer.AutoRebalancer;
import org.apache.helix.controller.rebalancer.CustomRebalancer;
import org.apache.helix.controller.rebalancer.MaintenanceRebalancer;
import org.apache.helix.controller.rebalancer.Rebalancer;
import org.apache.helix.controller.rebalancer.SemiAutoRebalancer;
import org.apache.helix.controller.rebalancer.internal.MappingCalculator;
import org.apache.helix.controller.rebalancer.waged.WagedRebalancer;
import org.apache.helix.controller.stages.AttributeName;
import org.apache.helix.controller.stages.BestPossibleStateOutput;
import org.apache.helix.controller.stages.ClusterEvent;
import org.apache.helix.controller.stages.CurrentStateOutput;
import org.apache.helix.model.ClusterConfig;
import org.apache.helix.model.IdealState;
import org.apache.helix.model.InstanceConfig;
import org.apache.helix.model.MaintenanceSignal;
import org.apache.helix.model.Partition;
import org.apache.helix.model.Resource;
import org.apache.helix.model.ResourceAssignment;
import org.apache.helix.model.StateModelDefinition;
import org.apache.helix.monitoring.mbeans.ClusterStatusMonitor;
import org.apache.helix.monitoring.mbeans.ResourceMonitor;
import org.apache.helix.util.HelixUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BestPossibleStateCalcStage
extends AbstractBaseStage {
    private static final Logger logger = LoggerFactory.getLogger((String)BestPossibleStateCalcStage.class.getName());

    @Override
    public void process(ClusterEvent event) throws Exception {
        this._eventId = event.getEventId();
        CurrentStateOutput currentStateOutput = (CurrentStateOutput)event.getAttribute(AttributeName.CURRENT_STATE.name());
        final Map resourceMap = (Map)event.getAttribute(AttributeName.RESOURCES_TO_REBALANCE.name());
        final ClusterStatusMonitor clusterStatusMonitor = (ClusterStatusMonitor)event.getAttribute(AttributeName.clusterStatusMonitor.name());
        ResourceControllerDataProvider cache = (ResourceControllerDataProvider)event.getAttribute(AttributeName.ControllerDataProvider.name());
        if (currentStateOutput == null || resourceMap == null || cache == null) {
            throw new StageException("Missing attributes in event:" + event + ". Requires CURRENT_STATE|RESOURCES|DataCache");
        }
        final BestPossibleStateOutput bestPossibleStateOutput = this.compute(event, resourceMap, currentStateOutput);
        event.addAttribute(AttributeName.BEST_POSSIBLE_STATE.name(), bestPossibleStateOutput);
        final Map<String, InstanceConfig> instanceConfigMap = cache.getInstanceConfigMap();
        final Map<String, StateModelDefinition> stateModelDefMap = cache.getStateModelDefMap();
        BestPossibleStateCalcStage.asyncExecute(cache.getAsyncTasksThreadPool(), new Callable<Object>(){

            @Override
            public Object call() {
                try {
                    if (clusterStatusMonitor != null) {
                        clusterStatusMonitor.setPerInstanceResourceStatus(bestPossibleStateOutput, instanceConfigMap, resourceMap, stateModelDefMap);
                    }
                }
                catch (Exception e) {
                    LogUtil.logError(logger, BestPossibleStateCalcStage.this._eventId, "Could not update cluster status metrics!", e);
                }
                return null;
            }
        });
    }

    private BestPossibleStateOutput compute(ClusterEvent event, Map<String, Resource> resourceMap, CurrentStateOutput currentStateOutput) {
        ResourceControllerDataProvider cache = (ResourceControllerDataProvider)event.getAttribute(AttributeName.ControllerDataProvider.name());
        BestPossibleStateOutput output = new BestPossibleStateOutput();
        HelixManager helixManager = (HelixManager)event.getAttribute(AttributeName.helixmanager.name());
        ClusterStatusMonitor clusterStatusMonitor = (ClusterStatusMonitor)event.getAttribute(AttributeName.clusterStatusMonitor.name());
        WagedRebalancer wagedRebalancer = (WagedRebalancer)event.getAttribute(AttributeName.STATEFUL_REBALANCER.name());
        boolean isValid = this.validateOfflineInstancesLimit(cache, (HelixManager)event.getAttribute(AttributeName.helixmanager.name()));
        ArrayList<String> failureResources = new ArrayList<String>();
        Map<String, Resource> calculatedResourceMap = this.computeResourceBestPossibleStateWithWagedRebalancer(wagedRebalancer, cache, currentStateOutput, resourceMap, output, failureResources);
        HashMap<String, Resource> remainingResourceMap = new HashMap<String, Resource>(resourceMap);
        remainingResourceMap.keySet().removeAll(calculatedResourceMap.keySet());
        for (Resource resource : remainingResourceMap.values()) {
            boolean result = false;
            try {
                result = this.computeSingleResourceBestPossibleState(event, cache, currentStateOutput, resource, output);
            }
            catch (HelixException ex) {
                LogUtil.logError(logger, this._eventId, String.format("Exception when calculating best possible states for %s", resource.getResourceName()), ex);
            }
            if (result) continue;
            failureResources.add(resource.getResourceName());
            LogUtil.logWarn(logger, this._eventId, String.format("Failed to calculate best possible states for %s", resource.getResourceName()));
        }
        this.updateRebalanceStatus(!isValid || !failureResources.isEmpty(), failureResources, helixManager, cache, clusterStatusMonitor, String.format("Failed to calculate best possible states for %d resources.", failureResources.size()));
        return output;
    }

    private void updateRebalanceStatus(final boolean hasFailure, final List<String> failedResources, HelixManager helixManager, ResourceControllerDataProvider cache, final ClusterStatusMonitor clusterStatusMonitor, final String errorMessage) {
        BestPossibleStateCalcStage.asyncExecute(cache.getAsyncTasksThreadPool(), new Callable<Object>(){

            @Override
            public Object call() {
                try {
                    if (hasFailure) {
                        LogUtil.logWarn(logger, BestPossibleStateCalcStage.this._eventId, errorMessage);
                    }
                    if (clusterStatusMonitor != null) {
                        clusterStatusMonitor.setRebalanceFailureGauge(hasFailure);
                        clusterStatusMonitor.setResourceRebalanceStates(failedResources, ResourceMonitor.RebalanceStatus.BEST_POSSIBLE_STATE_CAL_FAILED);
                    }
                }
                catch (Exception e) {
                    LogUtil.logError(logger, BestPossibleStateCalcStage.this._eventId, "Could not update cluster status!", e);
                }
                return null;
            }
        });
    }

    private boolean validateOfflineInstancesLimit(ResourceControllerDataProvider cache, HelixManager manager) {
        int offlineCount;
        int maxOfflineInstancesAllowed = cache.getClusterConfig().getMaxOfflineInstancesAllowed();
        if (maxOfflineInstancesAllowed >= 0 && (offlineCount = cache.getAllInstances().size() - cache.getEnabledLiveInstances().size()) > maxOfflineInstancesAllowed) {
            String errMsg = String.format("Offline Instances count %d greater than allowed count %d. Stop rebalance and put the cluster %s into maintenance mode.", offlineCount, maxOfflineInstancesAllowed, cache.getClusterName());
            if (manager != null) {
                if (manager.getHelixDataAccessor().getProperty(manager.getHelixDataAccessor().keyBuilder().maintenance()) == null) {
                    manager.getClusterManagmentTool().autoEnableMaintenanceMode(manager.getClusterName(), true, errMsg, MaintenanceSignal.AutoTriggerReason.MAX_OFFLINE_INSTANCES_EXCEEDED);
                    LogUtil.logWarn(logger, this._eventId, errMsg);
                }
            } else {
                LogUtil.logError(logger, this._eventId, "Failed to put cluster " + cache.getClusterName() + " into maintenance mode, HelixManager is not set!");
            }
            return false;
        }
        return true;
    }

    private void updateWagedRebalancer(WagedRebalancer wagedRebalancer, ClusterConfig clusterConfig) {
        if (clusterConfig != null) {
            wagedRebalancer.updateRebalancePreference(clusterConfig.getGlobalRebalancePreference());
            wagedRebalancer.setGlobalRebalanceAsyncMode(clusterConfig.isGlobalRebalanceAsyncModeEnabled());
        }
    }

    private Map<String, Resource> computeResourceBestPossibleStateWithWagedRebalancer(WagedRebalancer wagedRebalancer, ResourceControllerDataProvider cache, CurrentStateOutput currentStateOutput, Map<String, Resource> resourceMap, BestPossibleStateOutput output, List<String> failureResources) {
        if (cache.isMaintenanceModeEnabled()) {
            return Collections.emptyMap();
        }
        Map<String, Resource> wagedRebalancedResourceMap = resourceMap.entrySet().stream().filter(resourceEntry -> {
            IdealState is = cache.getIdealState((String)resourceEntry.getKey());
            return is != null && is.getRebalanceMode().equals((Object)IdealState.RebalanceMode.FULL_AUTO) && WagedRebalancer.class.getName().equals(is.getRebalancerClassName());
        }).collect(Collectors.toMap(resourceEntry -> (String)resourceEntry.getKey(), resourceEntry -> (Resource)resourceEntry.getValue()));
        HashMap<String, IdealState> newIdealStates = new HashMap<String, IdealState>();
        if (wagedRebalancer != null) {
            this.updateWagedRebalancer(wagedRebalancer, cache.getClusterConfig());
            try {
                newIdealStates.putAll(wagedRebalancer.computeNewIdealStates(cache, wagedRebalancedResourceMap, currentStateOutput));
            }
            catch (HelixRebalanceException ex) {
                LogUtil.logError(logger, this._eventId, String.format("Failed to calculate the new Ideal States using the rebalancer %s due to %s", new Object[]{wagedRebalancer.getClass().getSimpleName(), ex.getFailureType()}), ex);
            }
        } else {
            LogUtil.logWarn(logger, this._eventId, "Skip rebalancing using the WAGED rebalancer since it is not configured in the rebalance pipeline.");
        }
        for (Resource resource : wagedRebalancedResourceMap.values()) {
            IdealState is = (IdealState)newIdealStates.get(resource.getResourceName());
            if (is != null && this.checkBestPossibleStateCalculation(is)) {
                this.updateBestPossibleStateOutput(output, resource, is);
                continue;
            }
            failureResources.add(resource.getResourceName());
            LogUtil.logWarn(logger, this._eventId, String.format("Failed to calculate best possible states for %s.", resource.getResourceName()));
        }
        return wagedRebalancedResourceMap;
    }

    private void updateBestPossibleStateOutput(BestPossibleStateOutput output, Resource resource, IdealState computedIdealState) {
        output.setPreferenceLists(resource.getResourceName(), computedIdealState.getPreferenceLists());
        for (Partition partition : resource.getPartitions()) {
            Map<String, String> newStateMap = computedIdealState.getInstanceStateMap(partition.getPartitionName());
            output.setState(resource.getResourceName(), partition, newStateMap);
        }
    }

    private boolean computeSingleResourceBestPossibleState(ClusterEvent event, ResourceControllerDataProvider cache, CurrentStateOutput currentStateOutput, Resource resource, BestPossibleStateOutput output) {
        String resourceName = resource.getResourceName();
        LogUtil.logDebug(logger, this._eventId, "Processing resource:" + resourceName);
        IdealState idealState = cache.getIdealState(resourceName);
        if (idealState == null) {
            LogUtil.logInfo(logger, this._eventId, "resource:" + resourceName + " does not exist anymore");
            idealState = new IdealState(resourceName);
            idealState.setStateModelDefRef(resource.getStateModelDefRef());
        }
        if (idealState.getStateModelDefRef().equals("Task")) {
            LogUtil.logWarn(logger, this._eventId, String.format("Resource %s should not be processed by %s pipeline", resourceName, cache.getPipelineName()));
            return false;
        }
        Rebalancer<ResourceControllerDataProvider> rebalancer = this.getRebalancer(idealState, resourceName, cache.isMaintenanceModeEnabled());
        MappingCalculator<ResourceControllerDataProvider> mappingCalculator = this.getMappingCalculator(rebalancer, resourceName);
        if (rebalancer == null || mappingCalculator == null) {
            LogUtil.logError(logger, this._eventId, "Error computing assignment for resource " + resourceName + ". no rebalancer found. rebalancer: " + rebalancer + " mappingCalculator: " + mappingCalculator);
        }
        if (rebalancer != null && mappingCalculator != null) {
            ResourceAssignment partitionStateAssignment = null;
            try {
                HelixManager manager = (HelixManager)event.getAttribute(AttributeName.helixmanager.name());
                rebalancer.init(manager);
                idealState = rebalancer.computeNewIdealState(resourceName, idealState, currentStateOutput, cache);
                output.setPreferenceLists(resourceName, idealState.getPreferenceLists());
                partitionStateAssignment = mappingCalculator.computeBestPossiblePartitionState(cache, idealState, resource, currentStateOutput);
                if (partitionStateAssignment == null) {
                    LogUtil.logWarn(logger, this._eventId, "PartitionStateAssignment is null, resource: " + resourceName);
                    return false;
                }
                for (Partition partition : resource.getPartitions()) {
                    Map<String, String> newStateMap = partitionStateAssignment.getReplicaMap(partition);
                    output.setState(resourceName, partition, newStateMap);
                }
                return this.checkBestPossibleStateCalculation(idealState);
            }
            catch (HelixException e) {
                LogUtil.logError(logger, this._eventId, e.getMessage());
            }
            catch (Exception e) {
                LogUtil.logError(logger, this._eventId, "Error computing assignment for resource " + resourceName + ". Skipping.", e);
            }
        }
        return false;
    }

    private boolean checkBestPossibleStateCalculation(IdealState idealState) {
        if (idealState.getRebalanceMode() == IdealState.RebalanceMode.FULL_AUTO && !idealState.getReplicas().equals("0")) {
            Map<String, List<String>> preferenceLists = idealState.getPreferenceLists();
            if (preferenceLists == null || preferenceLists.isEmpty()) {
                return false;
            }
            int emptyListCount = 0;
            for (List<String> preferenceList : preferenceLists.values()) {
                if (!preferenceList.isEmpty()) continue;
                ++emptyListCount;
            }
            return emptyListCount != preferenceLists.values().size();
        }
        return true;
    }

    private Rebalancer<ResourceControllerDataProvider> getCustomizedRebalancer(String rebalancerClassName, String resourceName) {
        Rebalancer customizedRebalancer = null;
        if (rebalancerClassName != null) {
            if (logger.isDebugEnabled()) {
                LogUtil.logDebug(logger, this._eventId, "resource " + resourceName + " use idealStateRebalancer " + rebalancerClassName);
            }
            try {
                customizedRebalancer = (Rebalancer)Rebalancer.class.cast(HelixUtil.loadClass(this.getClass(), rebalancerClassName).newInstance());
            }
            catch (Exception e) {
                LogUtil.logError(logger, this._eventId, "Exception while invoking custom rebalancer class:" + rebalancerClassName, e);
            }
        }
        return customizedRebalancer;
    }

    private Rebalancer<ResourceControllerDataProvider> getRebalancer(IdealState idealState, String resourceName, boolean isMaintenanceModeEnabled) {
        Rebalancer<ResourceControllerDataProvider> rebalancer = null;
        switch (idealState.getRebalanceMode()) {
            case FULL_AUTO: {
                if (isMaintenanceModeEnabled) {
                    rebalancer = new MaintenanceRebalancer();
                    break;
                }
                Rebalancer<ResourceControllerDataProvider> customizedRebalancer = this.getCustomizedRebalancer(idealState.getRebalancerClassName(), resourceName);
                if (customizedRebalancer != null) {
                    rebalancer = customizedRebalancer;
                    break;
                }
                rebalancer = new AutoRebalancer();
                break;
            }
            case SEMI_AUTO: {
                rebalancer = new SemiAutoRebalancer();
                break;
            }
            case CUSTOMIZED: {
                rebalancer = new CustomRebalancer();
                break;
            }
            case USER_DEFINED: 
            case TASK: {
                rebalancer = this.getCustomizedRebalancer(idealState.getRebalancerClassName(), resourceName);
                break;
            }
            default: {
                LogUtil.logError(logger, this._eventId, "Fail to find the rebalancer, invalid rebalance mode " + (Object)((Object)idealState.getRebalanceMode()));
            }
        }
        return rebalancer;
    }

    private MappingCalculator<ResourceControllerDataProvider> getMappingCalculator(Rebalancer<ResourceControllerDataProvider> rebalancer, String resourceName) {
        SemiAutoRebalancer mappingCalculator = null;
        if (rebalancer != null) {
            try {
                mappingCalculator = (SemiAutoRebalancer)MappingCalculator.class.cast(rebalancer);
            }
            catch (ClassCastException e) {
                LogUtil.logWarn(logger, this._eventId, "Rebalancer does not have a mapping calculator, defaulting to SEMI_AUTO, resource: " + resourceName);
            }
        }
        if (mappingCalculator == null) {
            mappingCalculator = new SemiAutoRebalancer();
        }
        return mappingCalculator;
    }
}

