/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hbase;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.LongAdder;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.CompareOperator;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.ConnectionFactory;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.filter.ByteArrayComparable;
import org.apache.hadoop.hbase.filter.Filter;
import org.apache.hadoop.hbase.filter.FilterList;
import org.apache.hadoop.hbase.filter.RowFilter;
import org.apache.hadoop.hbase.filter.SingleColumnValueFilter;
import org.apache.hadoop.hbase.filter.SubstringComparator;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RegionsMerger
extends Configured
implements Tool {
    private static final Logger LOG = LoggerFactory.getLogger(RegionsMerger.class.getName());
    public static final String RESULTING_REGION_UPPER_MARK = "hbase.tools.merge.upper.mark";
    public static final String SLEEP = "hbase.tools.merge.sleep";
    public static final String MAX_ROUNDS_IDLE = "hbase.tools.max.iterations.blocked";
    private final Configuration conf;
    private final FileSystem fs;
    private final double resultSizeThreshold;
    private final int sleepBetweenCycles;
    private final long maxRoundsStuck;

    public RegionsMerger(Configuration conf) throws IOException {
        this.conf = conf;
        Path basePath = new Path(conf.get("hbase.rootdir"));
        this.fs = basePath.getFileSystem(conf);
        this.resultSizeThreshold = this.conf.getDouble(RESULTING_REGION_UPPER_MARK, 0.9) * (double)this.conf.getLong("hbase.hregion.max.filesize", 0x280000000L);
        this.sleepBetweenCycles = this.conf.getInt(SLEEP, 2000);
        this.maxRoundsStuck = this.conf.getInt(MAX_ROUNDS_IDLE, 10);
    }

    private Path getTablePath(TableName table) {
        Path basePath = new Path(this.conf.get("hbase.rootdir"));
        basePath = new Path(basePath, "data");
        Path tablePath = new Path(basePath, table.getNamespaceAsString());
        return new Path(tablePath, table.getQualifierAsString());
    }

    private long sumSizeInFS(Path parentPath) throws IOException {
        FileStatus[] files;
        long size = 0L;
        for (FileStatus f : files = this.fs.listStatus(parentPath)) {
            if (f.isFile()) {
                size += f.getLen();
                continue;
            }
            if (!f.isDirectory()) continue;
            size += this.sumSizeInFS(f.getPath());
        }
        return size;
    }

    private List<RegionInfo> getOpenRegions(Connection connection, TableName table) throws Exception {
        ArrayList<RegionInfo> regions = new ArrayList<RegionInfo>();
        Table metaTbl = connection.getTable(TableName.META_TABLE_NAME);
        String tblName = table.getNameAsString();
        RowFilter rowFilter = new RowFilter(CompareOperator.EQUAL, (ByteArrayComparable)new SubstringComparator(tblName + ","));
        SingleColumnValueFilter colFilter = new SingleColumnValueFilter(HConstants.CATALOG_FAMILY, HConstants.STATE_QUALIFIER, CompareOperator.EQUAL, Bytes.toBytes((String)"OPEN"));
        colFilter.setFilterIfMissing(true);
        Scan scan = new Scan();
        FilterList filter = new FilterList(FilterList.Operator.MUST_PASS_ALL);
        filter.addFilter((Filter)rowFilter);
        filter.addFilter((Filter)colFilter);
        scan.setFilter((Filter)filter);
        try (ResultScanner rs = metaTbl.getScanner(scan);){
            Result r;
            while ((r = rs.next()) != null) {
                RegionInfo region = RegionInfo.parseFrom((byte[])r.getValue(HConstants.CATALOG_FAMILY, HConstants.REGIONINFO_QUALIFIER));
                regions.add(region);
            }
        }
        return regions;
    }

    private boolean canMerge(Path path, RegionInfo region1, RegionInfo region2, Collection<Pair<RegionInfo, RegionInfo>> alreadyMerging) throws IOException {
        if (alreadyMerging.stream().anyMatch(regionPair -> region1.equals(regionPair.getFirst()) || region2.equals(regionPair.getFirst()) || region1.equals(regionPair.getSecond()) || region2.equals(regionPair.getSecond()))) {
            return false;
        }
        if (RegionInfo.areAdjacent((RegionInfo)region1, (RegionInfo)region2)) {
            long size2;
            boolean mergeable;
            long size1 = this.sumSizeInFS(new Path(path, region1.getEncodedName()));
            boolean bl = mergeable = this.resultSizeThreshold > (double)(size1 + (size2 = this.sumSizeInFS(new Path(path, region2.getEncodedName()))));
            if (!mergeable) {
                LOG.warn("Not merging regions {} and {} because resulting region size would get close to the {} limit. {} total size: {}; {} total size:{}", region1.getEncodedName(), region2.getEncodedName(), this.resultSizeThreshold, region1.getEncodedName(), size1, region2.getEncodedName(), size2);
            }
            return mergeable;
        }
        LOG.warn("WARNING: Can't merge regions {} and {} because those are not adjacent.", (Object)region1.getEncodedName(), (Object)region2.getEncodedName());
        return false;
    }

    public void mergeRegions(String tblName, int targetRegions) throws Exception {
        TableName table = TableName.valueOf((String)tblName);
        Path tableDir = this.getTablePath(table);
        try (Connection conn = ConnectionFactory.createConnection((Configuration)this.conf);){
            Admin admin = conn.getAdmin();
            LongAdder counter = new LongAdder();
            LongAdder lastTimeProgessed = new LongAdder();
            List regions = admin.getRegions(table);
            ConcurrentHashMap<Future, Pair> regionsMerging = new ConcurrentHashMap<Future, Pair>();
            long roundsNoProgress = 0L;
            while (regions.size() > targetRegions) {
                LOG.info("Iteration: {}", (Object)counter);
                RegionInfo previous = null;
                int regionSize = regions.size();
                LOG.info("Attempting to merge {} regions to reach the target {} ...", (Object)regionSize, (Object)targetRegions);
                regions = this.getOpenRegions(conn, table);
                for (RegionInfo current : regions) {
                    if (!current.isSplit()) {
                        if (previous != null && this.canMerge(tableDir, previous, current, regionsMerging.values())) {
                            Future f2 = admin.mergeRegionsAsync(current.getEncodedNameAsBytes(), previous.getEncodedNameAsBytes(), true);
                            Pair regionPair = new Pair((Object)previous, (Object)current);
                            regionsMerging.put(f2, regionPair);
                            previous = null;
                            if (regionSize - regionsMerging.size() > targetRegions) continue;
                            break;
                        }
                        previous = current;
                        continue;
                    }
                    LOG.debug("Skipping split region: {}", (Object)current.getEncodedName());
                }
                counter.increment();
                LOG.info("Sleeping for {} seconds before next iteration...", (Object)(this.sleepBetweenCycles / 1000));
                Thread.sleep(this.sleepBetweenCycles);
                regionsMerging.forEach((f, currentPair) -> {
                    if (f.isDone()) {
                        LOG.info("Merged regions {} and {} together.", (Object)((RegionInfo)currentPair.getFirst()).getEncodedName(), (Object)((RegionInfo)currentPair.getSecond()).getEncodedName());
                        regionsMerging.remove(f);
                        lastTimeProgessed.reset();
                        lastTimeProgessed.add(counter.longValue());
                    } else {
                        LOG.warn("Merge of regions {} and {} isn't completed yet.", currentPair.getFirst(), currentPair.getSecond());
                    }
                });
                roundsNoProgress = counter.longValue() - lastTimeProgessed.longValue();
                if (roundsNoProgress == this.maxRoundsStuck) {
                    LOG.warn("Reached {} iterations without progressing with new merges. Aborting...", (Object)roundsNoProgress);
                    break;
                }
                regions = admin.getRegions(table);
            }
        }
    }

    public int run(String[] args) {
        if (args.length != 2) {
            LOG.error("Wrong number of arguments. Arguments are: <TABLE_NAME> <TARGET_NUMBER_OF_REGIONS>");
            return 1;
        }
        try {
            this.mergeRegions(args[0], Integer.parseInt(args[1]));
        }
        catch (Exception e) {
            LOG.error("Merging regions failed:", e);
            return 2;
        }
        return 0;
    }

    public static void main(String[] args) throws Exception {
        Configuration conf = HBaseConfiguration.create();
        int errCode = ToolRunner.run((Tool)new RegionsMerger(conf), (String[])args);
        if (errCode != 0) {
            System.exit(errCode);
        }
    }
}

