/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase;

import static org.junit.Assert.assertEquals;

import java.io.IOException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptor;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptorBuilder;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.client.TableDescriptorBuilder;
import org.apache.hadoop.hbase.master.MasterFileSystem;
import org.apache.hadoop.hbase.testclassification.MediumTests;
import org.apache.hadoop.hbase.testclassification.MiscTests;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.CommonFSUtils;
import org.apache.hadoop.hbase.util.FSTableDescriptors;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.TestName;

/**
 * Verify that the ColumnFamilyDescriptor version is set correctly by default, hbase-site.xml, and
 * user input
 */
@Category({ MiscTests.class, MediumTests.class })
public class TestColumnFamilyDescriptorDefaultVersions {

  @ClassRule
  public static final HBaseClassTestRule CLASS_RULE =
    HBaseClassTestRule.forClass(TestColumnFamilyDescriptorDefaultVersions.class);

  @Rule
  public TestName name = new TestName();
  private static final HBaseTestingUtility TEST_UTIL = new HBaseTestingUtility();
  private static TableName TABLE_NAME = null;
  private static final byte[] FAMILY = Bytes.toBytes("cf0");

  /**
   * Start up a mini cluster and put a small table of empty regions into it.
   */
  @BeforeClass
  public static void beforeAllTests() throws Exception {
    TEST_UTIL.startMiniCluster(1);
  }

  @Before
  public void setup() {
    TABLE_NAME = TableName.valueOf(name.getMethodName());

  }

  @AfterClass
  public static void afterAllTests() throws Exception {
    TEST_UTIL.shutdownMiniCluster();
  }

  @Test
  public void testCreateTableWithDefault() throws IOException {
    Admin admin = TEST_UTIL.getAdmin();
    // Create a table with one family
    TableDescriptor tableDescriptor = TableDescriptorBuilder.newBuilder(TABLE_NAME)
      .setColumnFamily(ColumnFamilyDescriptorBuilder.of(FAMILY)).build();
    admin.createTable(tableDescriptor);
    admin.disableTable(TABLE_NAME);
    try {
      // Verify the column descriptor
      verifyHColumnDescriptor(1, TABLE_NAME, FAMILY);
    } finally {
      admin.deleteTable(TABLE_NAME);
    }
  }

  @Test
  public void testCreateTableWithDefaultFromConf() throws Exception {
    TEST_UTIL.shutdownMiniCluster();
    TEST_UTIL.getConfiguration().setInt("hbase.column.max.version", 3);
    TEST_UTIL.startMiniCluster(1);

    Admin admin = TEST_UTIL.getAdmin();
    // Create a table with one family
    TableDescriptor tableDescriptor = TableDescriptorBuilder.newBuilder(TABLE_NAME)
      .setColumnFamily(ColumnFamilyDescriptorBuilder.newBuilder(FAMILY)
        .setMaxVersions(TEST_UTIL.getConfiguration().getInt("hbase.column.max.version", 1)).build())
      .build();
    admin.createTable(tableDescriptor);
    admin.disableTable(TABLE_NAME);
    try {
      // Verify the column descriptor
      verifyHColumnDescriptor(3, TABLE_NAME, FAMILY);
    } finally {
      admin.deleteTable(TABLE_NAME);
    }
  }

  @Test
  public void testCreateTableWithSetVersion() throws Exception {
    TEST_UTIL.shutdownMiniCluster();
    TEST_UTIL.getConfiguration().setInt("hbase.column.max.version", 3);
    TEST_UTIL.startMiniCluster(1);

    Admin admin = TEST_UTIL.getAdmin();
    // Create a table with one family
    TableDescriptor tableDescriptor = TableDescriptorBuilder.newBuilder(TABLE_NAME)
      .setColumnFamily(ColumnFamilyDescriptorBuilder.newBuilder(FAMILY).setMaxVersions(5).build())
      .build();
    admin.createTable(tableDescriptor);
    admin.disableTable(TABLE_NAME);
    try {
      // Verify the column descriptor
      verifyHColumnDescriptor(5, TABLE_NAME, FAMILY);

    } finally {
      admin.deleteTable(TABLE_NAME);
    }
  }

  @Test
  public void testHColumnDescriptorCachedMaxVersions() throws Exception {
    ColumnFamilyDescriptor familyDescriptor =
      ColumnFamilyDescriptorBuilder.newBuilder(FAMILY).setMaxVersions(5).build();
    // Verify the max version
    assertEquals(5, familyDescriptor.getMaxVersions());

    // modify the max version
    familyDescriptor = ColumnFamilyDescriptorBuilder.newBuilder(familyDescriptor)
      .setValue(Bytes.toBytes(HConstants.VERSIONS), Bytes.toBytes("8")).build();
    // Verify the max version
    assertEquals(8, familyDescriptor.getMaxVersions());
  }

  private void verifyHColumnDescriptor(int expected, final TableName tableName,
    final byte[]... families) throws IOException {
    Admin admin = TEST_UTIL.getAdmin();

    // Verify descriptor from master
    TableDescriptor htd = admin.getDescriptor(tableName);
    ColumnFamilyDescriptor[] hcds = htd.getColumnFamilies();
    verifyColumnFamilyDescriptor(expected, hcds, tableName, families);

    // Verify descriptor from HDFS
    MasterFileSystem mfs = TEST_UTIL.getMiniHBaseCluster().getMaster().getMasterFileSystem();
    Path tableDir = CommonFSUtils.getTableDir(mfs.getRootDir(), tableName);
    TableDescriptor td = FSTableDescriptors.getTableDescriptorFromFs(mfs.getFileSystem(), tableDir);
    hcds = td.getColumnFamilies();
    verifyColumnFamilyDescriptor(expected, hcds, tableName, families);
  }

  private void verifyColumnFamilyDescriptor(int expected, final ColumnFamilyDescriptor[] hcds,
    final TableName tableName, final byte[]... families) {
    for (ColumnFamilyDescriptor hcd : hcds) {
      assertEquals(expected, hcd.getMaxVersions());
    }
  }
}
