/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import io.opentelemetry.sdk.testing.junit4.OpenTelemetryRule;
import io.opentelemetry.sdk.trace.data.SpanData;
import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.HBaseClassTestRule;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.MatcherPredicate;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.Waiter;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptor;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptorBuilder;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.RegionInfoBuilder;
import org.apache.hadoop.hbase.client.trace.StringTraceRenderer;
import org.apache.hadoop.hbase.client.trace.hamcrest.SpanDataMatchers;
import org.apache.hadoop.hbase.fs.HFileSystem;
import org.apache.hadoop.hbase.io.ByteBuffAllocator;
import org.apache.hadoop.hbase.io.HFileLink;
import org.apache.hadoop.hbase.io.compress.Compression;
import org.apache.hadoop.hbase.io.hfile.BlockCache;
import org.apache.hadoop.hbase.io.hfile.BlockCacheFactory;
import org.apache.hadoop.hbase.io.hfile.BlockCacheKey;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.CacheConfig;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFileBlock;
import org.apache.hadoop.hbase.io.hfile.HFileContext;
import org.apache.hadoop.hbase.io.hfile.HFileContextBuilder;
import org.apache.hadoop.hbase.io.hfile.PrefetchExecutor;
import org.apache.hadoop.hbase.io.hfile.RandomKeyValueUtil;
import org.apache.hadoop.hbase.regionserver.BloomType;
import org.apache.hadoop.hbase.regionserver.HRegionFileSystem;
import org.apache.hadoop.hbase.regionserver.HStoreFile;
import org.apache.hadoop.hbase.regionserver.StoreFileInfo;
import org.apache.hadoop.hbase.regionserver.StoreFileWriter;
import org.apache.hadoop.hbase.regionserver.TestHStoreFile;
import org.apache.hadoop.hbase.testclassification.IOTests;
import org.apache.hadoop.hbase.testclassification.MediumTests;
import org.apache.hadoop.hbase.trace.TraceUtil;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.CommonFSUtils;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.Assert;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Category(value={IOTests.class, MediumTests.class})
public class TestPrefetch {
    private static final Logger LOG = LoggerFactory.getLogger(TestPrefetch.class);
    @ClassRule
    public static final HBaseClassTestRule CLASS_RULE = HBaseClassTestRule.forClass(TestPrefetch.class);
    private static final HBaseTestingUtility TEST_UTIL = new HBaseTestingUtility();
    private static final int NUM_VALID_KEY_TYPES = KeyValue.Type.values().length - 2;
    private static final int DATA_BLOCK_SIZE = 2048;
    private static final int NUM_KV = 1000;
    private Configuration conf;
    private CacheConfig cacheConf;
    private FileSystem fs;
    private BlockCache blockCache;
    @Rule
    public OpenTelemetryRule otelRule = OpenTelemetryRule.create();

    @Before
    public void setUp() throws IOException {
        this.conf = TEST_UTIL.getConfiguration();
        this.conf.setBoolean("hbase.rs.prefetchblocksonopen", true);
        this.fs = HFileSystem.get((Configuration)this.conf);
        this.blockCache = BlockCacheFactory.createBlockCache((Configuration)this.conf);
        this.cacheConf = new CacheConfig(this.conf, this.blockCache);
    }

    @Test
    public void testPrefetchSetInHCDWorks() {
        ColumnFamilyDescriptor columnFamilyDescriptor = ColumnFamilyDescriptorBuilder.newBuilder((byte[])Bytes.toBytes((String)"f")).setPrefetchBlocksOnOpen(true).build();
        Configuration c = HBaseConfiguration.create();
        Assert.assertFalse((boolean)c.getBoolean("hbase.rs.prefetchblocksonopen", false));
        CacheConfig cc = new CacheConfig(c, columnFamilyDescriptor, this.blockCache, ByteBuffAllocator.HEAP);
        Assert.assertTrue((boolean)cc.shouldPrefetchOnOpen());
    }

    @Test
    public void testPrefetchBlockCacheDisabled() throws Exception {
        ScheduledThreadPoolExecutor poolExecutor = (ScheduledThreadPoolExecutor)PrefetchExecutor.getExecutorPool();
        long totalCompletedBefore = poolExecutor.getCompletedTaskCount();
        long queueBefore = poolExecutor.getQueue().size();
        ColumnFamilyDescriptor columnFamilyDescriptor = ColumnFamilyDescriptorBuilder.newBuilder((byte[])Bytes.toBytes((String)"f")).setPrefetchBlocksOnOpen(true).setBlockCacheEnabled(false).build();
        HFileContext meta = new HFileContextBuilder().withBlockSize(2048).build();
        CacheConfig cacheConfig = new CacheConfig(this.conf, columnFamilyDescriptor, this.blockCache, ByteBuffAllocator.HEAP);
        Path storeFile = this.writeStoreFile("testPrefetchBlockCacheDisabled", meta, cacheConfig);
        this.readStoreFile(storeFile, (r, o) -> {
            HFileBlock block = null;
            try {
                block = r.readBlock(o.longValue(), -1L, false, true, false, true, null, null);
            }
            catch (IOException e) {
                Assert.fail((String)e.getMessage());
            }
            return block;
        }, (key, block) -> {
            boolean isCached;
            boolean bl = isCached = this.blockCache.getBlock(key, true, false, true) != null;
            if (block.getBlockType() == BlockType.DATA || block.getBlockType() == BlockType.ROOT_INDEX || block.getBlockType() == BlockType.INTERMEDIATE_INDEX) {
                Assert.assertFalse((boolean)isCached);
            }
        }, cacheConfig);
        Assert.assertEquals((long)(totalCompletedBefore + queueBefore), (long)(poolExecutor.getCompletedTaskCount() + (long)poolExecutor.getQueue().size()));
    }

    @Test
    public void testPrefetch() throws Exception {
        TraceUtil.trace(() -> {
            Path storeFile = this.writeStoreFile("TestPrefetch");
            this.readStoreFile(storeFile);
        }, (String)"testPrefetch");
        TEST_UTIL.waitFor(TimeUnit.MINUTES.toMillis(1L), (Waiter.Predicate)new MatcherPredicate(() -> ((OpenTelemetryRule)this.otelRule).getSpans(), Matchers.hasItems((Matcher[])new Matcher[]{SpanDataMatchers.hasName((String)"testPrefetch"), SpanDataMatchers.hasName((String)"PrefetchExecutor.request")})));
        List spans = this.otelRule.getSpans();
        if (LOG.isDebugEnabled()) {
            StringTraceRenderer renderer = new StringTraceRenderer((Collection)spans);
            renderer.render(arg_0 -> ((Logger)LOG).debug(arg_0));
        }
        SpanData testSpan = spans.stream().filter(arg_0 -> ((Matcher)SpanDataMatchers.hasName((String)"testPrefetch")).matches(arg_0)).findFirst().orElseThrow(AssertionError::new);
        MatcherAssert.assertThat((String)"prefetch spans happen on their own threads, detached from file open.", (Object)spans, (Matcher)Matchers.hasItem((Matcher)Matchers.allOf((Matcher)SpanDataMatchers.hasName((String)"PrefetchExecutor.request"), (Matcher)Matchers.not((Matcher)SpanDataMatchers.hasParentSpanId((SpanData)testSpan)))));
    }

    @Test
    public void testPrefetchRace() throws Exception {
        for (int i = 0; i < 10; ++i) {
            Path storeFile = this.writeStoreFile("TestPrefetchRace-" + i);
            this.readStoreFileLikeScanner(storeFile);
        }
    }

    private void readStoreFileLikeScanner(Path storeFilePath) throws Exception {
        HFile.Reader reader = HFile.createReader((FileSystem)this.fs, (Path)storeFilePath, (CacheConfig)this.cacheConf, (boolean)true, (Configuration)this.conf);
        do {
            HFileBlock block;
            for (long offset = 0L; offset < reader.getTrailer().getLoadOnOpenDataOffset(); offset += (long)block.getOnDiskSizeWithHeader()) {
                block = reader.readBlock(offset, -1L, false, false, false, true, null, null);
            }
        } while (!reader.prefetchComplete());
    }

    private void readStoreFile(Path storeFilePath) throws Exception {
        this.readStoreFile(storeFilePath, (r, o) -> {
            HFileBlock block = null;
            try {
                block = r.readBlock(o.longValue(), -1L, false, true, false, true, null, null);
            }
            catch (IOException e) {
                Assert.fail((String)e.getMessage());
            }
            return block;
        }, (key, block) -> {
            boolean isCached;
            boolean bl = isCached = this.blockCache.getBlock(key, true, false, true) != null;
            if (block.getBlockType() == BlockType.DATA || block.getBlockType() == BlockType.ROOT_INDEX || block.getBlockType() == BlockType.INTERMEDIATE_INDEX) {
                Assert.assertTrue((boolean)isCached);
            }
        });
    }

    private void readStoreFileCacheOnly(Path storeFilePath) throws Exception {
        this.readStoreFile(storeFilePath, (r, o) -> {
            HFileBlock block = null;
            try {
                block = r.readBlock(o.longValue(), -1L, false, true, false, true, null, null, true);
            }
            catch (IOException e) {
                Assert.fail((String)e.getMessage());
            }
            return block;
        }, (key, block) -> {
            boolean isCached;
            boolean bl = isCached = this.blockCache.getBlock(key, true, false, true) != null;
            if (block.getBlockType() == BlockType.DATA) {
                Assert.assertFalse((boolean)block.isUnpacked());
            } else if (block.getBlockType() == BlockType.ROOT_INDEX || block.getBlockType() == BlockType.INTERMEDIATE_INDEX) {
                Assert.assertTrue((boolean)block.isUnpacked());
            }
            Assert.assertTrue((boolean)isCached);
        });
    }

    private void readStoreFile(Path storeFilePath, BiFunction<HFile.Reader, Long, HFileBlock> readFunction, BiConsumer<BlockCacheKey, HFileBlock> validationFunction) throws Exception {
        this.readStoreFile(storeFilePath, readFunction, validationFunction, this.cacheConf);
    }

    private void readStoreFile(Path storeFilePath, BiFunction<HFile.Reader, Long, HFileBlock> readFunction, BiConsumer<BlockCacheKey, HFileBlock> validationFunction, CacheConfig cacheConfig) throws Exception {
        HFileBlock block;
        HFile.Reader reader = HFile.createReader((FileSystem)this.fs, (Path)storeFilePath, (CacheConfig)cacheConfig, (boolean)true, (Configuration)this.conf);
        while (!reader.prefetchComplete()) {
            Thread.sleep(1000L);
        }
        for (long offset = 0L; offset < reader.getTrailer().getLoadOnOpenDataOffset(); offset += (long)block.getOnDiskSizeWithHeader()) {
            block = readFunction.apply(reader, offset);
            BlockCacheKey blockCacheKey = new BlockCacheKey(reader.getName(), offset);
            validationFunction.accept(blockCacheKey, block);
        }
    }

    @Test
    public void testPrefetchCompressed() throws Exception {
        this.conf.setBoolean("hbase.block.data.cachecompressed", true);
        this.cacheConf = new CacheConfig(this.conf, this.blockCache);
        HFileContext context = new HFileContextBuilder().withCompression(Compression.Algorithm.GZ).withBlockSize(2048).build();
        Path storeFile = this.writeStoreFile("TestPrefetchCompressed", context);
        this.readStoreFileCacheOnly(storeFile);
        this.conf.setBoolean("hbase.block.data.cachecompressed", false);
    }

    @Test
    public void testPrefetchDoesntSkipHFileLink() throws Exception {
        this.testPrefetchWhenHFileLink(c -> {
            boolean isCached = c != null;
            Assert.assertTrue((boolean)isCached);
        });
    }

    private void testPrefetchWhenHFileLink(Consumer<Cacheable> test) throws Exception {
        HFileBlock block;
        this.cacheConf = new CacheConfig(this.conf, this.blockCache);
        HFileContext context = new HFileContextBuilder().withBlockSize(2048).build();
        Path testDir = TEST_UTIL.getDataTestDir("testPrefetchWhenHFileLink");
        RegionInfo hri = RegionInfoBuilder.newBuilder((TableName)TableName.valueOf((String)"testPrefetchWhenHFileLink")).build();
        Configuration testConf = new Configuration(this.conf);
        CommonFSUtils.setRootDir((Configuration)testConf, (Path)testDir);
        HRegionFileSystem regionFs = HRegionFileSystem.createRegionOnFileSystem((Configuration)testConf, (FileSystem)this.fs, (Path)CommonFSUtils.getTableDir((Path)testDir, (TableName)hri.getTable()), (RegionInfo)hri);
        StoreFileWriter writer = new StoreFileWriter.Builder(this.conf, this.cacheConf, this.fs).withFilePath(regionFs.createTempName()).withFileContext(context).build();
        TestHStoreFile.writeStoreFile(writer, Bytes.toBytes((String)"testPrefetchWhenHFileLink"), Bytes.toBytes((String)"testPrefetchWhenHFileLink"));
        Path storeFilePath = regionFs.commitStoreFile("cf", writer.getPath());
        Path dstPath = new Path(regionFs.getTableDir(), new Path("test-region", "cf"));
        HFileLink.create((Configuration)testConf, (FileSystem)this.fs, (Path)dstPath, (RegionInfo)hri, (String)storeFilePath.getName());
        Path linkFilePath = new Path(dstPath, HFileLink.createHFileLinkName((RegionInfo)hri, (String)storeFilePath.getName()));
        StoreFileInfo storeFileInfo = new StoreFileInfo(testConf, this.fs, linkFilePath, true);
        HStoreFile hsf = new HStoreFile(storeFileInfo, BloomType.NONE, this.cacheConf);
        Assert.assertTrue((boolean)storeFileInfo.isLink());
        hsf.initReader();
        HFile.Reader reader = hsf.getReader().getHFileReader();
        while (!reader.prefetchComplete()) {
            Thread.sleep(1000L);
        }
        for (long offset = 0L; offset < reader.getTrailer().getLoadOnOpenDataOffset(); offset += (long)block.getOnDiskSizeWithHeader()) {
            block = reader.readBlock(offset, -1L, false, true, false, true, null, null, true);
            BlockCacheKey blockCacheKey = new BlockCacheKey(reader.getName(), offset);
            if (block.getBlockType() != BlockType.DATA) continue;
            test.accept(this.blockCache.getBlock(blockCacheKey, true, false, true));
        }
    }

    private Path writeStoreFile(String fname) throws IOException {
        HFileContext meta = new HFileContextBuilder().withBlockSize(2048).build();
        return this.writeStoreFile(fname, meta);
    }

    private Path writeStoreFile(String fname, HFileContext context) throws IOException {
        return this.writeStoreFile(fname, context, this.cacheConf);
    }

    private Path writeStoreFile(String fname, HFileContext context, CacheConfig cacheConfig) throws IOException {
        Path storeFileParentDir = new Path(TEST_UTIL.getDataTestDir(), fname);
        StoreFileWriter sfw = new StoreFileWriter.Builder(this.conf, cacheConfig, this.fs).withOutputDir(storeFileParentDir).withFileContext(context).build();
        ThreadLocalRandom rand = ThreadLocalRandom.current();
        int rowLen = 32;
        for (int i = 0; i < 1000; ++i) {
            byte[] k = RandomKeyValueUtil.randomOrderedKey(rand, i);
            byte[] v = RandomKeyValueUtil.randomValue(rand);
            int cfLen = ((Random)rand).nextInt(k.length - 32 + 1);
            KeyValue kv = new KeyValue(k, 0, 32, k, 32, cfLen, k, 32 + cfLen, k.length - 32 - cfLen, ((Random)rand).nextLong(), TestPrefetch.generateKeyType(rand), v, 0, v.length);
            sfw.append((Cell)kv);
        }
        sfw.close();
        return sfw.getPath();
    }

    public static KeyValue.Type generateKeyType(Random rand) {
        if (rand.nextBoolean()) {
            return KeyValue.Type.Put;
        }
        KeyValue.Type keyType = KeyValue.Type.values()[1 + rand.nextInt(NUM_VALID_KEY_TYPES)];
        if (keyType == KeyValue.Type.Minimum || keyType == KeyValue.Type.Maximum) {
            throw new RuntimeException("Generated an invalid key type: " + keyType + ". Probably the layout of KeyValue.Type has changed.");
        }
        return keyType;
    }
}

