/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.util.test;

import java.io.IOException;
import java.util.Random;
import java.util.Set;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Mutation;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.LoadTestKVGenerator;
import org.apache.yetus.audience.InterfaceAudience;

/**
 * A generator of random data (keys/cfs/columns/values) for load testing. Contains
 * LoadTestKVGenerator as a matter of convenience...
 */
@InterfaceAudience.Private
public abstract class LoadTestDataGenerator {
  protected LoadTestKVGenerator kvGenerator;

  // The mutate info column stores information
  // about update done to this column family this row.
  public final static byte[] MUTATE_INFO = Bytes.toBytes("mutate_info");

  // The increment column always has a long value,
  // which can be incremented later on during updates.
  public final static byte[] INCREMENT = Bytes.toBytes("increment");

  protected String[] args;

  public LoadTestDataGenerator() {

  }

  /**
   * Initializes the object.
   * @param minValueSize minimum size of the value generated by
   *                     {@link #generateValue(byte[], byte[], byte[])}.
   * @param maxValueSize maximum size of the value generated by
   *                     {@link #generateValue(byte[], byte[], byte[])}.
   */
  public LoadTestDataGenerator(int minValueSize, int maxValueSize) {
    this.kvGenerator = new LoadTestKVGenerator(minValueSize, maxValueSize);
  }

  public static byte[] generateData(final Random r, int length) {
    byte[] b = new byte[length];
    int i = 0;

    for (i = 0; i < (length - 8); i += 8) {
      b[i] = (byte) (65 + r.nextInt(26));
      b[i + 1] = b[i];
      b[i + 2] = b[i];
      b[i + 3] = b[i];
      b[i + 4] = b[i];
      b[i + 5] = b[i];
      b[i + 6] = b[i];
      b[i + 7] = b[i];
    }

    byte a = (byte) (65 + r.nextInt(26));
    for (; i < length; i++) {
      b[i] = a;
    }
    return b;
  }

  /**
   * initialize the LoadTestDataGenerator n * init args
   */
  public void initialize(String[] args) {
    this.args = args;
  }

  /**
   * Generates a deterministic, unique hashed row key from a number. That way, the user can keep
   * track of numbers, without messing with byte array and ensuring key distribution.
   * @param keyBase Base number for a key, such as a loop counter.
   */
  public abstract byte[] getDeterministicUniqueKey(long keyBase);

  /**
   * Gets column families for the load test table.
   * @return The array of byte[]s representing column family names.
   */
  public abstract byte[][] getColumnFamilies();

  /**
   * Generates an applicable set of columns to be used for a particular key and family.
   * @param rowKey The row key to generate for.
   * @param cf     The column family name to generate for.
   * @return The array of byte[]s representing column names.
   */
  public abstract byte[][] generateColumnsForCf(byte[] rowKey, byte[] cf);

  /**
   * Generates a value to be used for a particular row/cf/column.
   * @param rowKey The row key to generate for.
   * @param cf     The column family name to generate for.
   * @param column The column name to generate for.
   * @return The value to use.
   */
  public abstract byte[] generateValue(byte[] rowKey, byte[] cf, byte[] column);

  /**
   * Checks that columns for a rowKey and cf are valid if generated via
   * {@link #generateColumnsForCf(byte[], byte[])}
   * @param rowKey    The row key to verify for.
   * @param cf        The column family name to verify for.
   * @param columnSet The column set (for example, encountered by read).
   * @return True iff valid.
   */
  public abstract boolean verify(byte[] rowKey, byte[] cf, Set<byte[]> columnSet);

  /**
   * Checks that value for a rowKey/cf/column is valid if generated via
   * {@link #generateValue(byte[], byte[], byte[])}
   * @param rowKey The row key to verify for.
   * @param cf     The column family name to verify for.
   * @param column The column name to verify for.
   * @param value  The value (for example, encountered by read).
   * @return True iff valid.
   */
  public abstract boolean verify(byte[] rowKey, byte[] cf, byte[] column, byte[] value);

  /**
   * Giving a chance for the LoadTestDataGenerator to change the Mutation load. nn * @return updated
   * Mutation n
   */
  public Mutation beforeMutate(long rowkeyBase, Mutation m) throws IOException {
    return m;
  }

  /**
   * Giving a chance for the LoadTestDataGenerator to change the Get load. nn * @return updated Get
   * n
   */
  public Get beforeGet(long rowkeyBase, Get get) throws IOException {
    return get;
  }

  /** Returns the arguments passed to the generator as a list of objects. */
  public String[] getArgs() {
    return this.args;
  }
}
