/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.rsgroup;

import com.google.protobuf.BlockingRpcChannel;
import com.google.protobuf.ServiceException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.OptionalLong;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.NamespaceDescriptor;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptorBuilder;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.CoprocessorDescriptorBuilder;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Mutation;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.client.TableDescriptorBuilder;
import org.apache.hadoop.hbase.constraint.ConstraintException;
import org.apache.hadoop.hbase.coprocessor.MultiRowMutationEndpoint;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.ipc.CoprocessorRpcChannel;
import org.apache.hadoop.hbase.master.ClusterSchema;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.ServerListener;
import org.apache.hadoop.hbase.master.TableStateManager;
import org.apache.hadoop.hbase.master.procedure.CreateTableProcedure;
import org.apache.hadoop.hbase.master.procedure.MasterProcedureUtil;
import org.apache.hadoop.hbase.net.Address;
import org.apache.hadoop.hbase.procedure2.Procedure;
import org.apache.hadoop.hbase.protobuf.ProtobufMagic;
import org.apache.hadoop.hbase.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.protobuf.generated.ClientProtos;
import org.apache.hadoop.hbase.protobuf.generated.MultiRowMutationProtos;
import org.apache.hadoop.hbase.protobuf.generated.RSGroupProtos;
import org.apache.hadoop.hbase.regionserver.DisabledRegionSplitPolicy;
import org.apache.hadoop.hbase.rsgroup.RSGroupInfo;
import org.apache.hadoop.hbase.rsgroup.RSGroupInfoManager;
import org.apache.hadoop.hbase.rsgroup.RSGroupProtobufUtil;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZKWatcher;
import org.apache.hadoop.hbase.zookeeper.ZNodePaths;
import org.apache.hadoop.util.Shell;
import org.apache.hbase.thirdparty.com.google.common.collect.Lists;
import org.apache.hbase.thirdparty.com.google.common.collect.Maps;
import org.apache.hbase.thirdparty.com.google.common.collect.Sets;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
final class RSGroupInfoManagerImpl
implements RSGroupInfoManager {
    private static final Logger LOG = LoggerFactory.getLogger(RSGroupInfoManagerImpl.class);
    private static final TableDescriptor RSGROUP_TABLE_DESC;
    private volatile Map<String, RSGroupInfo> rsGroupMap = Collections.emptyMap();
    private volatile Map<TableName, String> tableMap = Collections.emptyMap();
    private final MasterServices masterServices;
    private final Connection conn;
    private final ZKWatcher watcher;
    private final RSGroupStartupWorker rsGroupStartupWorker;
    private Set<String> prevRSGroups = new HashSet<String>();
    private final ServerEventsListenerThread serverEventsListenerThread = new ServerEventsListenerThread();
    RSGroupMappingScript script;

    private RSGroupInfoManagerImpl(MasterServices masterServices) throws IOException {
        this.masterServices = masterServices;
        this.watcher = masterServices.getZooKeeper();
        this.conn = masterServices.getConnection();
        this.rsGroupStartupWorker = new RSGroupStartupWorker();
        this.script = new RSGroupMappingScript(masterServices.getConfiguration());
    }

    private synchronized void init() throws IOException {
        this.refresh();
        this.serverEventsListenerThread.start();
        this.masterServices.getServerManager().registerListener((ServerListener)this.serverEventsListenerThread);
    }

    static RSGroupInfoManager getInstance(MasterServices master) throws IOException {
        RSGroupInfoManagerImpl instance = new RSGroupInfoManagerImpl(master);
        instance.init();
        return instance;
    }

    @Override
    public void start() {
        this.rsGroupStartupWorker.start();
    }

    @Override
    public synchronized void addRSGroup(RSGroupInfo rsGroupInfo) throws IOException {
        this.checkGroupName(rsGroupInfo.getName());
        if (this.rsGroupMap.get(rsGroupInfo.getName()) != null || rsGroupInfo.getName().equals("default")) {
            throw new DoNotRetryIOException("Group already exists: " + rsGroupInfo.getName());
        }
        HashMap newGroupMap = Maps.newHashMap(this.rsGroupMap);
        newGroupMap.put(rsGroupInfo.getName(), rsGroupInfo);
        this.flushConfig(newGroupMap);
    }

    private RSGroupInfo getRSGroupInfo(String groupName) throws DoNotRetryIOException {
        RSGroupInfo rsGroupInfo = this.getRSGroup(groupName);
        if (rsGroupInfo == null) {
            throw new DoNotRetryIOException("RSGroup " + groupName + " does not exist");
        }
        return rsGroupInfo;
    }

    private static Set<Address> getOnlineServers(MasterServices master) {
        HashSet<Address> onlineServers = new HashSet<Address>();
        if (master == null) {
            return onlineServers;
        }
        for (ServerName server : master.getServerManager().getOnlineServers().keySet()) {
            onlineServers.add(server.getAddress());
        }
        return onlineServers;
    }

    @Override
    public synchronized Set<Address> moveServers(Set<Address> servers, String srcGroup, String dstGroup) throws IOException {
        RSGroupInfo src = this.getRSGroupInfo(srcGroup);
        RSGroupInfo dst = this.getRSGroupInfo(dstGroup);
        HashSet<Address> movedServers = new HashSet<Address>();
        Set<Address> onlineServers = dst.getName().equals("default") ? RSGroupInfoManagerImpl.getOnlineServers(this.masterServices) : null;
        for (Address el : servers) {
            src.removeServer(el);
            if (onlineServers != null && !onlineServers.contains(el)) {
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Dropping " + el + " during move-to-default rsgroup because not online");
                continue;
            }
            dst.addServer(el);
            movedServers.add(el);
        }
        HashMap newGroupMap = Maps.newHashMap(this.rsGroupMap);
        newGroupMap.put(src.getName(), src);
        newGroupMap.put(dst.getName(), dst);
        this.flushConfig(newGroupMap);
        return movedServers;
    }

    @Override
    public RSGroupInfo getRSGroupOfServer(Address serverHostPort) throws IOException {
        for (RSGroupInfo info : this.rsGroupMap.values()) {
            if (!info.containsServer(serverHostPort)) continue;
            return info;
        }
        return null;
    }

    @Override
    public RSGroupInfo getRSGroup(String groupName) {
        return this.rsGroupMap.get(groupName);
    }

    @Override
    public String getRSGroupOfTable(TableName tableName) {
        return this.tableMap.get(tableName);
    }

    @Override
    public synchronized void moveTables(Set<TableName> tableNames, String groupName) throws IOException {
        if (groupName != null && !this.rsGroupMap.containsKey(groupName)) {
            throw new DoNotRetryIOException("Group " + groupName + " does not exist");
        }
        HashMap newGroupMap = Maps.newHashMap(this.rsGroupMap);
        for (TableName tableName : tableNames) {
            if (!this.tableMap.containsKey(tableName)) continue;
            RSGroupInfo src = new RSGroupInfo((RSGroupInfo)newGroupMap.get(this.tableMap.get(tableName)));
            src.removeTable(tableName);
            newGroupMap.put(src.getName(), src);
        }
        if (groupName != null) {
            RSGroupInfo dstGroup = new RSGroupInfo((RSGroupInfo)newGroupMap.get(groupName));
            dstGroup.addAllTables(tableNames);
            newGroupMap.put(dstGroup.getName(), dstGroup);
        }
        this.flushConfig(newGroupMap);
    }

    @Override
    public synchronized void removeRSGroup(String groupName) throws IOException {
        if (!this.rsGroupMap.containsKey(groupName) || groupName.equals("default")) {
            throw new DoNotRetryIOException("Group " + groupName + " does not exist or is a reserved group");
        }
        HashMap newGroupMap = Maps.newHashMap(this.rsGroupMap);
        newGroupMap.remove(groupName);
        this.flushConfig(newGroupMap);
    }

    @Override
    public List<RSGroupInfo> listRSGroups() {
        return Lists.newLinkedList(this.rsGroupMap.values());
    }

    @Override
    public boolean isOnline() {
        return this.rsGroupStartupWorker.isOnline();
    }

    @Override
    public void moveServersAndTables(Set<Address> servers, Set<TableName> tables, String srcGroup, String dstGroup) throws IOException {
        RSGroupInfo srcGroupInfo = this.getRSGroupInfo(srcGroup);
        RSGroupInfo dstGroupInfo = this.getRSGroupInfo(dstGroup);
        for (Address el : servers) {
            srcGroupInfo.removeServer(el);
            dstGroupInfo.addServer(el);
        }
        for (TableName tableName : tables) {
            srcGroupInfo.removeTable(tableName);
            dstGroupInfo.addTable(tableName);
        }
        HashMap newGroupMap = Maps.newHashMap(this.rsGroupMap);
        newGroupMap.put(srcGroupInfo.getName(), srcGroupInfo);
        newGroupMap.put(dstGroupInfo.getName(), dstGroupInfo);
        this.flushConfig(newGroupMap);
    }

    @Override
    public synchronized void removeServers(Set<Address> servers) throws IOException {
        HashMap<String, RSGroupInfo> rsGroupInfos = new HashMap<String, RSGroupInfo>();
        for (Address el : servers) {
            RSGroupInfo rsGroupInfo = this.getRSGroupOfServer(el);
            if (rsGroupInfo != null) {
                RSGroupInfo newRsGroupInfo = (RSGroupInfo)rsGroupInfos.get(rsGroupInfo.getName());
                if (newRsGroupInfo == null) {
                    rsGroupInfo.removeServer(el);
                    rsGroupInfos.put(rsGroupInfo.getName(), rsGroupInfo);
                    continue;
                }
                newRsGroupInfo.removeServer(el);
                rsGroupInfos.put(newRsGroupInfo.getName(), newRsGroupInfo);
                continue;
            }
            LOG.warn("Server " + el + " does not belong to any rsgroup.");
        }
        if (rsGroupInfos.size() > 0) {
            HashMap newGroupMap = Maps.newHashMap(this.rsGroupMap);
            newGroupMap.putAll(rsGroupInfos);
            this.flushConfig(newGroupMap);
        }
    }

    @Override
    public void renameRSGroup(String oldName, String newName) throws IOException {
        this.checkGroupName(oldName);
        this.checkGroupName(newName);
        if (oldName.equals("default")) {
            throw new ConstraintException("Can't rename default rsgroup");
        }
        RSGroupInfo oldGroup = this.getRSGroup(oldName);
        if (oldGroup == null) {
            throw new ConstraintException("RSGroup " + oldName + " does not exist");
        }
        if (this.rsGroupMap.containsKey(newName)) {
            throw new ConstraintException("Group already exists: " + newName);
        }
        HashMap newGroupMap = Maps.newHashMap(this.rsGroupMap);
        newGroupMap.remove(oldName);
        RSGroupInfo newGroup = new RSGroupInfo(newName, (SortedSet)oldGroup.getServers(), oldGroup.getTables());
        newGroupMap.put(newName, newGroup);
        this.flushConfig(newGroupMap);
    }

    @Override
    public RSGroupInfo determineRSGroupInfoForTable(TableName tableName) throws IOException {
        RSGroupInfo groupFromOldRSGroupInfo = this.getRSGroup(this.getRSGroupOfTable(tableName));
        if (groupFromOldRSGroupInfo != null) {
            return groupFromOldRSGroupInfo;
        }
        RSGroupInfo groupDeterminedByAdmin = this.getRSGroup(this.script.getRSGroup(tableName.getNamespaceAsString(), tableName.getQualifierAsString()));
        if (groupDeterminedByAdmin != null) {
            return groupDeterminedByAdmin;
        }
        ClusterSchema clusterSchema = this.masterServices.getClusterSchema();
        if (clusterSchema == null) {
            if (TableName.isMetaTableName((TableName)tableName)) {
                LOG.info("Can not get the namespace rs group config for meta table, since the meta table is not online yet, will use default group to assign meta first");
            } else {
                LOG.warn("ClusterSchema is null, can only use default rsgroup, should not happen?");
            }
        } else {
            NamespaceDescriptor nd = clusterSchema.getNamespace(tableName.getNamespaceAsString());
            RSGroupInfo groupNameOfNs = this.getRSGroup(nd.getConfigurationValue("hbase.rsgroup.name"));
            if (groupNameOfNs != null) {
                return groupNameOfNs;
            }
        }
        return this.getRSGroup("default");
    }

    @Override
    public void updateRSGroupConfig(String groupName, Map<String, String> configuration) throws IOException {
        if ("default".equals(groupName)) {
            throw new ConstraintException("configuration of default can't be stored persistently");
        }
        RSGroupInfo rsGroupInfo = this.getRSGroupInfo(groupName);
        new HashSet(rsGroupInfo.getConfiguration().keySet()).forEach(arg_0 -> ((RSGroupInfo)rsGroupInfo).removeConfiguration(arg_0));
        configuration.forEach((arg_0, arg_1) -> ((RSGroupInfo)rsGroupInfo).setConfiguration(arg_0, arg_1));
        this.flushConfig();
    }

    List<RSGroupInfo> retrieveGroupListFromGroupTable() throws IOException {
        ArrayList rsGroupInfoList = Lists.newArrayList();
        try (Table table = this.conn.getTable(RSGROUP_TABLE_NAME);
             ResultScanner scanner = table.getScanner(new Scan());){
            Result result;
            while ((result = scanner.next()) != null) {
                RSGroupProtos.RSGroupInfo proto = RSGroupProtos.RSGroupInfo.parseFrom((byte[])result.getValue(META_FAMILY_BYTES, META_QUALIFIER_BYTES));
                rsGroupInfoList.add(RSGroupProtobufUtil.toGroupInfo(proto));
            }
        }
        return rsGroupInfoList;
    }

    List<RSGroupInfo> retrieveGroupListFromZookeeper() throws IOException {
        String groupBasePath = ZNodePaths.joinZNode((String)this.watcher.getZNodePaths().baseZNode, (String)"rsgroup");
        ArrayList RSGroupInfoList = Lists.newArrayList();
        try {
            if (ZKUtil.checkExists((ZKWatcher)this.watcher, (String)groupBasePath) != -1) {
                List children = ZKUtil.listChildrenAndWatchForNewChildren((ZKWatcher)this.watcher, (String)groupBasePath);
                if (children == null) {
                    return RSGroupInfoList;
                }
                for (String znode : children) {
                    byte[] data = ZKUtil.getData((ZKWatcher)this.watcher, (String)ZNodePaths.joinZNode((String)groupBasePath, (String)znode));
                    if (data == null || data.length <= 0) continue;
                    ProtobufUtil.expectPBMagicPrefix((byte[])data);
                    ByteArrayInputStream bis = new ByteArrayInputStream(data, ProtobufUtil.lengthOfPBMagic(), data.length);
                    RSGroupInfoList.add(RSGroupProtobufUtil.toGroupInfo(RSGroupProtos.RSGroupInfo.parseFrom((InputStream)bis)));
                }
                LOG.debug("Read ZK GroupInfo count:" + RSGroupInfoList.size());
            }
        }
        catch (InterruptedException | DeserializationException | KeeperException e) {
            throw new IOException("Failed to read rsGroupZNode", e);
        }
        return RSGroupInfoList;
    }

    @Override
    public void refresh() throws IOException {
        this.refresh(false);
    }

    private synchronized void refresh(boolean forceOnline) throws IOException {
        LinkedList<RSGroupInfo> groupList = new LinkedList<RSGroupInfo>();
        if (forceOnline || this.isOnline()) {
            LOG.debug("Refreshing in Online mode.");
            groupList.addAll(this.retrieveGroupListFromGroupTable());
        } else {
            LOG.debug("Refreshing in Offline mode.");
            groupList.addAll(this.retrieveGroupListFromZookeeper());
        }
        TreeSet<TableName> orphanTables = new TreeSet<TableName>();
        for (String entry : this.masterServices.getTableDescriptors().getAll().keySet()) {
            orphanTables.add(TableName.valueOf((String)entry));
        }
        for (RSGroupInfo group : groupList) {
            if (group.getName().equals("default")) continue;
            orphanTables.removeAll(group.getTables());
        }
        groupList.add(new RSGroupInfo("default", this.getDefaultServers(groupList), orphanTables));
        HashMap newGroupMap = Maps.newHashMap();
        HashMap newTableMap = Maps.newHashMap();
        for (RSGroupInfo group : groupList) {
            newGroupMap.put(group.getName(), group);
            for (TableName table : group.getTables()) {
                newTableMap.put(table, group.getName());
            }
        }
        this.resetRSGroupAndTableMaps(newGroupMap, newTableMap);
        this.updateCacheOfRSGroups(this.rsGroupMap.keySet());
    }

    private synchronized Map<TableName, String> flushConfigTable(Map<String, RSGroupInfo> groupMap) throws IOException {
        HashMap newTableMap = Maps.newHashMap();
        ArrayList mutations = Lists.newArrayList();
        for (String groupName : this.prevRSGroups) {
            if (groupMap.containsKey(groupName)) continue;
            Delete d = new Delete(Bytes.toBytes((String)groupName));
            mutations.add(d);
        }
        for (RSGroupInfo RSGroupInfo2 : groupMap.values()) {
            RSGroupProtos.RSGroupInfo proto = RSGroupProtobufUtil.toProtoGroupInfo(RSGroupInfo2);
            Put p = new Put(Bytes.toBytes((String)RSGroupInfo2.getName()));
            p.addColumn(META_FAMILY_BYTES, META_QUALIFIER_BYTES, proto.toByteArray());
            mutations.add(p);
            for (TableName entry : RSGroupInfo2.getTables()) {
                newTableMap.put(entry, RSGroupInfo2.getName());
            }
        }
        if (mutations.size() > 0) {
            this.multiMutate(mutations);
        }
        return newTableMap;
    }

    private synchronized void flushConfig() throws IOException {
        this.flushConfig(this.rsGroupMap);
    }

    private synchronized void flushConfig(Map<String, RSGroupInfo> newGroupMap) throws IOException {
        if (!this.isOnline()) {
            if (newGroupMap == this.rsGroupMap) {
                return;
            }
            HashMap oldGroupMap = Maps.newHashMap(this.rsGroupMap);
            RSGroupInfo oldDefaultGroup = (RSGroupInfo)oldGroupMap.remove("default");
            RSGroupInfo newDefaultGroup = newGroupMap.remove("default");
            if (!oldGroupMap.equals(newGroupMap) || !oldDefaultGroup.getTables().equals(newDefaultGroup.getTables())) {
                throw new IOException("Only servers in default group can be updated during offline mode");
            }
            newGroupMap.put("default", newDefaultGroup);
            this.rsGroupMap = newGroupMap;
            return;
        }
        Map<TableName, String> newTableMap = this.flushConfigTable(newGroupMap);
        this.resetRSGroupAndTableMaps(newGroupMap, newTableMap);
        try {
            String znode;
            String groupBasePath = ZNodePaths.joinZNode((String)this.watcher.getZNodePaths().baseZNode, (String)"rsgroup");
            ZKUtil.createAndFailSilent((ZKWatcher)this.watcher, (String)groupBasePath, (byte[])ProtobufMagic.PB_MAGIC);
            ArrayList<ZKUtil.ZKUtilOp> zkOps = new ArrayList<ZKUtil.ZKUtilOp>(newGroupMap.size());
            for (String groupName : this.prevRSGroups) {
                if (newGroupMap.containsKey(groupName)) continue;
                znode = ZNodePaths.joinZNode((String)groupBasePath, (String)groupName);
                zkOps.add(ZKUtil.ZKUtilOp.deleteNodeFailSilent((String)znode));
            }
            for (RSGroupInfo RSGroupInfo2 : newGroupMap.values()) {
                znode = ZNodePaths.joinZNode((String)groupBasePath, (String)RSGroupInfo2.getName());
                RSGroupProtos.RSGroupInfo proto = RSGroupProtobufUtil.toProtoGroupInfo(RSGroupInfo2);
                LOG.debug("Updating znode: " + znode);
                ZKUtil.createAndFailSilent((ZKWatcher)this.watcher, (String)znode);
                zkOps.add(ZKUtil.ZKUtilOp.deleteNodeFailSilent((String)znode));
                zkOps.add(ZKUtil.ZKUtilOp.createAndFailSilent((String)znode, (byte[])ProtobufUtil.prependPBMagic((byte[])proto.toByteArray())));
            }
            LOG.debug("Writing ZK GroupInfo count: " + zkOps.size());
            ZKUtil.multiOrSequential((ZKWatcher)this.watcher, zkOps, (boolean)false);
        }
        catch (KeeperException e) {
            LOG.error("Failed to write to rsGroupZNode", (Throwable)e);
            this.masterServices.abort("Failed to write to rsGroupZNode", (Throwable)e);
            throw new IOException("Failed to write to rsGroupZNode", e);
        }
        this.updateCacheOfRSGroups(newGroupMap.keySet());
    }

    private void resetRSGroupAndTableMaps(Map<String, RSGroupInfo> newRSGroupMap, Map<TableName, String> newTableMap) {
        this.rsGroupMap = Collections.unmodifiableMap(newRSGroupMap);
        this.tableMap = Collections.unmodifiableMap(newTableMap);
    }

    private void updateCacheOfRSGroups(Set<String> currentGroups) {
        this.prevRSGroups.clear();
        this.prevRSGroups.addAll(currentGroups);
    }

    private List<ServerName> getOnlineRS() throws IOException {
        if (this.masterServices != null) {
            return this.masterServices.getServerManager().getOnlineServersList();
        }
        LOG.debug("Reading online RS from zookeeper");
        LinkedList<ServerName> servers = new LinkedList<ServerName>();
        try {
            for (String el : ZKUtil.listChildrenNoWatch((ZKWatcher)this.watcher, (String)this.watcher.getZNodePaths().rsZNode)) {
                servers.add(ServerName.parseServerName((String)el));
            }
        }
        catch (KeeperException e) {
            throw new IOException("Failed to retrieve server list from zookeeper", e);
        }
        return servers;
    }

    private SortedSet<Address> getDefaultServers() throws IOException {
        return this.getDefaultServers(this.listRSGroups());
    }

    private SortedSet<Address> getDefaultServers(List<RSGroupInfo> rsGroupInfoList) throws IOException {
        HashSet serversInOtherGroup = new HashSet();
        for (RSGroupInfo group : rsGroupInfoList) {
            if ("default".equals(group.getName())) continue;
            serversInOtherGroup.addAll(group.getServers());
        }
        TreeSet defaultServers = Sets.newTreeSet();
        for (ServerName serverName : this.getOnlineRS()) {
            Address server = Address.fromParts((String)serverName.getHostname(), (int)serverName.getPort());
            if (serversInOtherGroup.contains(server)) continue;
            defaultServers.add(server);
        }
        return defaultServers;
    }

    private synchronized void updateDefaultServers(SortedSet<Address> servers) throws IOException {
        RSGroupInfo info = this.rsGroupMap.get("default");
        RSGroupInfo newInfo = new RSGroupInfo(info.getName(), servers, info.getTables());
        HashMap newGroupMap = Maps.newHashMap(this.rsGroupMap);
        newGroupMap.put(newInfo.getName(), newInfo);
        this.flushConfig(newGroupMap);
    }

    private static boolean isMasterRunning(MasterServices masterServices) {
        return !masterServices.isAborted() && !masterServices.isStopped();
    }

    private void multiMutate(List<Mutation> mutations) throws IOException {
        try (Table table = this.conn.getTable(RSGROUP_TABLE_NAME);){
            CoprocessorRpcChannel channel = table.coprocessorService(ROW_KEY);
            MultiRowMutationProtos.MutateRowsRequest.Builder mmrBuilder = MultiRowMutationProtos.MutateRowsRequest.newBuilder();
            for (Mutation mutation : mutations) {
                if (mutation instanceof Put) {
                    mmrBuilder.addMutationRequest(ProtobufUtil.toMutation((ClientProtos.MutationProto.MutationType)ClientProtos.MutationProto.MutationType.PUT, (Mutation)mutation));
                    continue;
                }
                if (mutation instanceof Delete) {
                    mmrBuilder.addMutationRequest(ProtobufUtil.toMutation((ClientProtos.MutationProto.MutationType)ClientProtos.MutationProto.MutationType.DELETE, (Mutation)mutation));
                    continue;
                }
                throw new DoNotRetryIOException("multiMutate doesn't support " + mutation.getClass().getName());
            }
            MultiRowMutationProtos.MultiRowMutationService.BlockingInterface service = MultiRowMutationProtos.MultiRowMutationService.newBlockingStub((BlockingRpcChannel)channel);
            try {
                service.mutateRows(null, mmrBuilder.build());
            }
            catch (ServiceException ex) {
                ProtobufUtil.toIOException((ServiceException)ex);
            }
        }
    }

    private void checkGroupName(String groupName) throws ConstraintException {
        if (!groupName.matches("[a-zA-Z0-9_]+")) {
            throw new ConstraintException("RSGroup name should only contain alphanumeric characters");
        }
    }

    static {
        TableDescriptorBuilder builder = TableDescriptorBuilder.newBuilder((TableName)RSGROUP_TABLE_NAME).setColumnFamily(ColumnFamilyDescriptorBuilder.of((byte[])META_FAMILY_BYTES)).setRegionSplitPolicyClassName(DisabledRegionSplitPolicy.class.getName());
        try {
            builder.setCoprocessor(CoprocessorDescriptorBuilder.newBuilder((String)MultiRowMutationEndpoint.class.getName()).setPriority(0x1FFFFFFF).build());
        }
        catch (IOException ex) {
            throw new Error(ex);
        }
        RSGROUP_TABLE_DESC = builder.build();
    }

    private class RSGroupStartupWorker
    extends Thread {
        private final Logger LOG;
        private volatile boolean online;

        RSGroupStartupWorker() {
            super(RSGroupStartupWorker.class.getName() + "-" + RSGroupInfoManagerImpl.this.masterServices.getServerName());
            this.LOG = LoggerFactory.getLogger(RSGroupStartupWorker.class);
            this.online = false;
            this.setDaemon(true);
        }

        @Override
        public void run() {
            if (this.waitForGroupTableOnline()) {
                this.LOG.info("GroupBasedLoadBalancer is now online");
            } else {
                this.LOG.warn("Quit without making region group table online");
            }
        }

        private boolean waitForGroupTableOnline() {
            while (RSGroupInfoManagerImpl.isMasterRunning(RSGroupInfoManagerImpl.this.masterServices)) {
                try {
                    TableStateManager tsm = RSGroupInfoManagerImpl.this.masterServices.getTableStateManager();
                    if (!tsm.isTablePresent(RSGroupInfoManager.RSGROUP_TABLE_NAME)) {
                        this.createRSGroupTable();
                    }
                    try (Table table = RSGroupInfoManagerImpl.this.conn.getTable(RSGroupInfoManager.RSGROUP_TABLE_NAME);){
                        table.get(new Get(RSGroupInfoManager.ROW_KEY));
                    }
                    this.LOG.info("RSGroup table=" + RSGroupInfoManager.RSGROUP_TABLE_NAME + " is online, refreshing cached information");
                    RSGroupInfoManagerImpl.this.refresh(true);
                    this.online = true;
                    RSGroupInfoManagerImpl.this.flushConfig();
                    return true;
                }
                catch (Exception e) {
                    this.LOG.warn("Failed to perform check", (Throwable)e);
                    Threads.sleepWithoutInterrupt((long)100L);
                }
            }
            return false;
        }

        private void createRSGroupTable() throws IOException {
            int tries;
            OptionalLong optProcId = RSGroupInfoManagerImpl.this.masterServices.getProcedures().stream().filter(p -> p instanceof CreateTableProcedure).map(p -> (CreateTableProcedure)p).filter(p -> p.getTableName().equals((Object)RSGroupInfoManager.RSGROUP_TABLE_NAME)).mapToLong(Procedure::getProcId).findFirst();
            long procId = optProcId.isPresent() ? optProcId.getAsLong() : RSGroupInfoManagerImpl.this.masterServices.createSystemTable(RSGROUP_TABLE_DESC);
            for (tries = 600; !RSGroupInfoManagerImpl.this.masterServices.getMasterProcedureExecutor().isFinished(procId) && RSGroupInfoManagerImpl.this.masterServices.getMasterProcedureExecutor().isRunning() && tries > 0; --tries) {
                try {
                    Thread.sleep(100L);
                    continue;
                }
                catch (InterruptedException e) {
                    throw new IOException("Wait interrupted ", e);
                }
            }
            if (tries <= 0) {
                throw new IOException("Failed to create group table in a given time.");
            }
            Procedure result = RSGroupInfoManagerImpl.this.masterServices.getMasterProcedureExecutor().getResult(procId);
            if (result != null && result.isFailed()) {
                throw new IOException("Failed to create group table. " + MasterProcedureUtil.unwrapRemoteIOException((Procedure)result));
            }
        }

        public boolean isOnline() {
            return this.online;
        }
    }

    private class ServerEventsListenerThread
    extends Thread
    implements ServerListener {
        private final Logger LOG = LoggerFactory.getLogger(ServerEventsListenerThread.class);
        private boolean changed = false;

        ServerEventsListenerThread() {
            this.setDaemon(true);
        }

        public void serverAdded(ServerName serverName) {
            this.serverChanged();
        }

        public void serverRemoved(ServerName serverName) {
            this.serverChanged();
        }

        private synchronized void serverChanged() {
            this.changed = true;
            this.notify();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            this.setName(ServerEventsListenerThread.class.getName() + "-" + RSGroupInfoManagerImpl.this.masterServices.getServerName());
            SortedSet prevDefaultServers = new TreeSet();
            while (RSGroupInfoManagerImpl.isMasterRunning(RSGroupInfoManagerImpl.this.masterServices)) {
                try {
                    this.LOG.info("Updating default servers.");
                    SortedSet servers = RSGroupInfoManagerImpl.this.getDefaultServers();
                    if (!servers.equals(prevDefaultServers)) {
                        RSGroupInfoManagerImpl.this.updateDefaultServers(servers);
                        prevDefaultServers = servers;
                        this.LOG.info("Updated with servers: " + servers.size());
                    }
                    try {
                        ServerEventsListenerThread serverEventsListenerThread = this;
                        synchronized (serverEventsListenerThread) {
                            while (!this.changed) {
                                this.wait();
                            }
                            this.changed = false;
                        }
                    }
                    catch (InterruptedException e) {
                        this.LOG.warn("Interrupted", (Throwable)e);
                    }
                }
                catch (IOException e) {
                    this.LOG.warn("Failed to update default servers", (Throwable)e);
                }
            }
        }
    }

    static class RSGroupMappingScript {
        static final String RS_GROUP_MAPPING_SCRIPT = "hbase.rsgroup.table.mapping.script";
        static final String RS_GROUP_MAPPING_SCRIPT_TIMEOUT = "hbase.rsgroup.table.mapping.script.timeout";
        private Shell.ShellCommandExecutor rsgroupMappingScript;

        RSGroupMappingScript(Configuration conf) {
            String script = conf.get(RS_GROUP_MAPPING_SCRIPT);
            if (script == null || script.isEmpty()) {
                return;
            }
            this.rsgroupMappingScript = new Shell.ShellCommandExecutor(new String[]{script, "", ""}, null, null, conf.getLong(RS_GROUP_MAPPING_SCRIPT_TIMEOUT, 5000L));
        }

        String getRSGroup(String namespace, String tablename) {
            if (this.rsgroupMappingScript == null) {
                return null;
            }
            String[] exec = this.rsgroupMappingScript.getExecString();
            exec[1] = namespace;
            exec[2] = tablename;
            try {
                this.rsgroupMappingScript.execute();
            }
            catch (IOException e) {
                LOG.error("Failed to get RSGroup from script for table {}:{}", new Object[]{namespace, tablename, e});
                return null;
            }
            return this.rsgroupMappingScript.getOutput().trim();
        }
    }
}

