/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.normalizer;

import java.io.IOException;
import java.time.Instant;
import java.time.Period;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.function.BooleanSupplier;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.RegionMetrics;
import org.apache.hadoop.hbase.ServerMetrics;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.Size;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.MasterSwitchType;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.RegionState;
import org.apache.hadoop.hbase.master.assignment.RegionStates;
import org.apache.hadoop.hbase.master.normalizer.MergeNormalizationPlan;
import org.apache.hadoop.hbase.master.normalizer.NormalizationPlan;
import org.apache.hadoop.hbase.master.normalizer.RegionNormalizer;
import org.apache.hadoop.hbase.master.normalizer.SplitNormalizationPlan;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hbase.thirdparty.org.apache.commons.collections4.CollectionUtils;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.LimitedPrivate(value={"Configuration"})
public class SimpleRegionNormalizer
implements RegionNormalizer {
    private static final Logger LOG = LoggerFactory.getLogger(SimpleRegionNormalizer.class);
    static final String SPLIT_ENABLED_KEY = "hbase.normalizer.split.enabled";
    static final boolean DEFAULT_SPLIT_ENABLED = true;
    static final String MERGE_ENABLED_KEY = "hbase.normalizer.merge.enabled";
    static final boolean DEFAULT_MERGE_ENABLED = true;
    static final String MIN_REGION_COUNT_KEY = "hbase.normalizer.min.region.count";
    static final int DEFAULT_MIN_REGION_COUNT = 3;
    static final String MERGE_MIN_REGION_AGE_DAYS_KEY = "hbase.normalizer.merge.min_region_age.days";
    static final int DEFAULT_MERGE_MIN_REGION_AGE_DAYS = 3;
    static final String MERGE_MIN_REGION_SIZE_MB_KEY = "hbase.normalizer.merge.min_region_size.mb";
    static final int DEFAULT_MERGE_MIN_REGION_SIZE_MB = 1;
    private final long[] skippedCount = new long[NormalizationPlan.PlanType.values().length];
    private Configuration conf;
    private MasterServices masterServices;
    private boolean splitEnabled = true;
    private boolean mergeEnabled = true;
    private int minRegionCount = 3;
    private Period mergeMinRegionAge = Period.ofDays(3);
    private int mergeMinRegionSizeMb = 1;

    public Configuration getConf() {
        return this.conf;
    }

    public void setConf(Configuration conf) {
        if (conf == null) {
            return;
        }
        this.conf = conf;
        this.splitEnabled = conf.getBoolean(SPLIT_ENABLED_KEY, true);
        this.mergeEnabled = conf.getBoolean(MERGE_ENABLED_KEY, true);
        this.minRegionCount = SimpleRegionNormalizer.parseMinRegionCount(conf);
        this.mergeMinRegionAge = SimpleRegionNormalizer.parseMergeMinRegionAge(conf);
        this.mergeMinRegionSizeMb = SimpleRegionNormalizer.parseMergeMinRegionSizeMb(conf);
    }

    private static int parseMinRegionCount(Configuration conf) {
        int settledValue;
        int parsedValue = conf.getInt(MIN_REGION_COUNT_KEY, 3);
        if (parsedValue != (settledValue = Math.max(1, parsedValue))) {
            SimpleRegionNormalizer.warnInvalidValue(MIN_REGION_COUNT_KEY, parsedValue, settledValue);
        }
        return settledValue;
    }

    private static Period parseMergeMinRegionAge(Configuration conf) {
        int settledValue;
        int parsedValue = conf.getInt(MERGE_MIN_REGION_AGE_DAYS_KEY, 3);
        if (parsedValue != (settledValue = Math.max(0, parsedValue))) {
            SimpleRegionNormalizer.warnInvalidValue(MERGE_MIN_REGION_AGE_DAYS_KEY, parsedValue, settledValue);
        }
        return Period.ofDays(settledValue);
    }

    private static int parseMergeMinRegionSizeMb(Configuration conf) {
        int settledValue;
        int parsedValue = conf.getInt(MERGE_MIN_REGION_SIZE_MB_KEY, 1);
        if (parsedValue != (settledValue = Math.max(0, parsedValue))) {
            SimpleRegionNormalizer.warnInvalidValue(MERGE_MIN_REGION_SIZE_MB_KEY, parsedValue, settledValue);
        }
        return settledValue;
    }

    private static <T> void warnInvalidValue(String key, T parsedValue, T settledValue) {
        LOG.warn("Configured value {}={} is invalid. Setting value to {}.", new Object[]{key, parsedValue, settledValue});
    }

    public boolean isSplitEnabled() {
        return this.splitEnabled;
    }

    public boolean isMergeEnabled() {
        return this.mergeEnabled;
    }

    public int getMinRegionCount() {
        return this.minRegionCount;
    }

    public Period getMergeMinRegionAge() {
        return this.mergeMinRegionAge;
    }

    public int getMergeMinRegionSizeMb() {
        return this.mergeMinRegionSizeMb;
    }

    @Override
    public void setMasterServices(MasterServices masterServices) {
        this.masterServices = masterServices;
    }

    @Override
    public void planSkipped(RegionInfo hri, NormalizationPlan.PlanType type) {
        int n = type.ordinal();
        this.skippedCount[n] = this.skippedCount[n] + 1L;
    }

    @Override
    public long getSkippedCount(NormalizationPlan.PlanType type) {
        return this.skippedCount[type.ordinal()];
    }

    @Override
    public List<NormalizationPlan> computePlansForTable(TableName table) {
        if (table == null) {
            return Collections.emptyList();
        }
        if (table.isSystemTable()) {
            LOG.debug("Normalization of system table {} isn't allowed", (Object)table);
            return Collections.emptyList();
        }
        boolean proceedWithSplitPlanning = this.proceedWithSplitPlanning();
        boolean proceedWithMergePlanning = this.proceedWithMergePlanning();
        if (!proceedWithMergePlanning && !proceedWithSplitPlanning) {
            LOG.debug("Both split and merge are disabled. Skipping normalization of table: {}", (Object)table);
            return Collections.emptyList();
        }
        NormalizeContext ctx = new NormalizeContext(table);
        if (CollectionUtils.isEmpty(ctx.getTableRegions())) {
            return Collections.emptyList();
        }
        LOG.debug("Computing normalization plan for table:  {}, number of regions: {}", (Object)table, (Object)ctx.getTableRegions().size());
        ArrayList<NormalizationPlan> plans = new ArrayList<NormalizationPlan>();
        int splitPlansCount = 0;
        if (proceedWithSplitPlanning) {
            List<NormalizationPlan> splitPlans = this.computeSplitNormalizationPlans(ctx);
            splitPlansCount = splitPlans.size();
            plans.addAll(splitPlans);
        }
        int mergePlansCount = 0;
        if (proceedWithMergePlanning) {
            List<NormalizationPlan> mergePlans = this.computeMergeNormalizationPlans(ctx);
            mergePlansCount = mergePlans.size();
            plans.addAll(mergePlans);
        }
        LOG.debug("Computed normalization plans for table {}. Total plans: {}, split plans: {}, merge plans: {}", new Object[]{table, plans.size(), splitPlansCount, mergePlansCount});
        return plans;
    }

    private long getRegionSizeMB(RegionInfo hri) {
        ServerName sn = this.masterServices.getAssignmentManager().getRegionStates().getRegionServerOfRegion(hri);
        if (sn == null) {
            LOG.debug("{} region was not found on any Server", (Object)hri.getRegionNameAsString());
            return -1L;
        }
        ServerMetrics serverMetrics = this.masterServices.getServerManager().getLoad(sn);
        if (serverMetrics == null) {
            LOG.debug("server {} was not found in ServerManager", (Object)sn.getServerName());
            return -1L;
        }
        RegionMetrics regionLoad = serverMetrics.getRegionMetrics().get(hri.getRegionName());
        if (regionLoad == null) {
            LOG.debug("{} was not found in RegionsLoad", (Object)hri.getRegionNameAsString());
            return -1L;
        }
        return (long)regionLoad.getStoreFileSize().get(Size.Unit.MEGABYTE);
    }

    private boolean isMasterSwitchEnabled(MasterSwitchType masterSwitchType) {
        return this.masterServices.isSplitOrMergeEnabled(masterSwitchType);
    }

    private boolean proceedWithSplitPlanning() {
        return this.isSplitEnabled() && this.isMasterSwitchEnabled(MasterSwitchType.SPLIT);
    }

    private boolean proceedWithMergePlanning() {
        return this.isMergeEnabled() && this.isMasterSwitchEnabled(MasterSwitchType.MERGE);
    }

    private double getAverageRegionSizeMb(List<RegionInfo> tableRegions) {
        double avgRegionSize;
        if (CollectionUtils.isEmpty(tableRegions)) {
            throw new IllegalStateException("Cannot calculate average size of a table without any regions.");
        }
        TableName table = tableRegions.get(0).getTable();
        int targetRegionCount = -1;
        long targetRegionSize = -1L;
        try {
            TableDescriptor tableDescriptor = this.masterServices.getTableDescriptors().get(table);
            if (tableDescriptor != null) {
                targetRegionCount = tableDescriptor.getNormalizerTargetRegionCount();
                targetRegionSize = tableDescriptor.getNormalizerTargetRegionSize();
                LOG.debug("Table {} configured with target region count {}, target region size {} MB", new Object[]{table, targetRegionCount, targetRegionSize});
            }
        }
        catch (IOException e) {
            LOG.warn("TableDescriptor for {} unavailable, table-level target region count and size configurations cannot be considered.", (Object)table, (Object)e);
        }
        if (targetRegionSize > 0L) {
            avgRegionSize = targetRegionSize;
        } else {
            int regionCount = tableRegions.size();
            long totalSizeMb = tableRegions.stream().mapToLong(this::getRegionSizeMB).sum();
            avgRegionSize = targetRegionCount > 0 ? (double)totalSizeMb / (double)targetRegionCount : (double)totalSizeMb / (double)regionCount;
            LOG.debug("Table {}, total aggregated regions size: {} MB and average region size {} MB", new Object[]{table, totalSizeMb, String.format("%.3f", avgRegionSize)});
        }
        return avgRegionSize;
    }

    private boolean skipForMerge(RegionStates regionStates, RegionInfo regionInfo) {
        RegionState state = regionStates.getRegionState(regionInfo);
        String name = regionInfo.getEncodedName();
        return SimpleRegionNormalizer.logTraceReason(() -> state == null, "skipping merge of region {} because no state information is available.", name) || SimpleRegionNormalizer.logTraceReason(() -> !Objects.equals((Object)state.getState(), (Object)RegionState.State.OPEN), "skipping merge of region {} because it is not open.", name) || SimpleRegionNormalizer.logTraceReason(() -> !this.isOldEnoughForMerge(regionInfo), "skipping merge of region {} because it is not old enough.", name) || SimpleRegionNormalizer.logTraceReason(() -> !this.isLargeEnoughForMerge(regionInfo), "skipping merge region {} because it is not large enough.", name);
    }

    private List<NormalizationPlan> computeMergeNormalizationPlans(NormalizeContext ctx) {
        if (ctx.getTableRegions().size() < this.minRegionCount) {
            LOG.debug("Table {} has {} regions, required min number of regions for normalizer to run is {}, not computing merge plans.", new Object[]{ctx.getTableName(), ctx.getTableRegions().size(), this.minRegionCount});
            return Collections.emptyList();
        }
        double avgRegionSizeMb = ctx.getAverageRegionSizeMb();
        LOG.debug("Computing normalization plan for table {}. average region size: {} MB, number of regions: {}.", new Object[]{ctx.getTableName(), avgRegionSizeMb, ctx.getTableRegions().size()});
        ArrayList<NormalizationPlan> plans = new ArrayList<NormalizationPlan>();
        for (int candidateIdx = 0; candidateIdx < ctx.getTableRegions().size() - 1; ++candidateIdx) {
            RegionInfo current = ctx.getTableRegions().get(candidateIdx);
            RegionInfo next = ctx.getTableRegions().get(candidateIdx + 1);
            if (this.skipForMerge(ctx.getRegionStates(), current) || this.skipForMerge(ctx.getRegionStates(), next)) continue;
            long currentSizeMb = this.getRegionSizeMB(current);
            long nextSizeMb = this.getRegionSizeMB(next);
            if (currentSizeMb != 0L && nextSizeMb != 0L && !((double)(currentSizeMb + nextSizeMb) < avgRegionSizeMb)) continue;
            plans.add(new MergeNormalizationPlan(current, next));
            ++candidateIdx;
        }
        return plans;
    }

    private static boolean skipForSplit(RegionState state, RegionInfo regionInfo) {
        String name = regionInfo.getEncodedName();
        return SimpleRegionNormalizer.logTraceReason(() -> state == null, "skipping split of region {} because no state information is available.", name) || SimpleRegionNormalizer.logTraceReason(() -> !Objects.equals((Object)state.getState(), (Object)RegionState.State.OPEN), "skipping merge of region {} because it is not open.", name);
    }

    private List<NormalizationPlan> computeSplitNormalizationPlans(NormalizeContext ctx) {
        double avgRegionSize = ctx.getAverageRegionSizeMb();
        LOG.debug("Table {}, average region size: {} MB", (Object)ctx.getTableName(), (Object)String.format("%.3f", avgRegionSize));
        ArrayList<NormalizationPlan> plans = new ArrayList<NormalizationPlan>();
        for (RegionInfo hri : ctx.getTableRegions()) {
            long regionSize;
            if (SimpleRegionNormalizer.skipForSplit(ctx.getRegionStates().getRegionState(hri), hri) || !((double)(regionSize = this.getRegionSizeMB(hri)) > 2.0 * avgRegionSize)) continue;
            LOG.info("Table {}, large region {} has size {} MB, more than twice avg size {} MB, splitting", new Object[]{ctx.getTableName(), hri.getRegionNameAsString(), regionSize, String.format("%.3f", avgRegionSize)});
            plans.add(new SplitNormalizationPlan(hri));
        }
        return plans;
    }

    private boolean isOldEnoughForMerge(RegionInfo regionInfo) {
        Instant currentTime = Instant.ofEpochMilli(EnvironmentEdgeManager.currentTime());
        Instant regionCreateTime = Instant.ofEpochMilli(regionInfo.getRegionId());
        return currentTime.isAfter(regionCreateTime.plus(this.mergeMinRegionAge));
    }

    private boolean isLargeEnoughForMerge(RegionInfo regionInfo) {
        return this.getRegionSizeMB(regionInfo) >= (long)this.mergeMinRegionSizeMb;
    }

    private static boolean logTraceReason(BooleanSupplier predicate, String fmtWhenTrue, Object ... args) {
        boolean value = predicate.getAsBoolean();
        if (value) {
            LOG.trace(fmtWhenTrue, args);
        }
        return value;
    }

    private class NormalizeContext {
        private final TableName tableName;
        private final RegionStates regionStates;
        private final List<RegionInfo> tableRegions;
        private final double averageRegionSizeMb;

        public NormalizeContext(TableName tableName) {
            this.tableName = tableName;
            this.regionStates = SimpleRegionNormalizer.this.masterServices.getAssignmentManager().getRegionStates();
            this.tableRegions = this.regionStates.getRegionsOfTable(tableName);
            this.tableRegions.sort(RegionInfo.COMPARATOR);
            this.averageRegionSizeMb = SimpleRegionNormalizer.this.getAverageRegionSizeMb(this.tableRegions);
        }

        public TableName getTableName() {
            return this.tableName;
        }

        public RegionStates getRegionStates() {
            return this.regionStates;
        }

        public List<RegionInfo> getTableRegions() {
            return this.tableRegions;
        }

        public double getAverageRegionSizeMb() {
            return this.averageRegionSizeMb;
        }
    }
}

