/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.ipc;

import java.util.Iterator;
import java.util.LinkedList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.net.Address;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.Pair;

/**
 * A class to manage a list of servers that failed recently.
 */
@InterfaceAudience.Private
public class FailedServers {
  private final LinkedList<Pair<Long, Address>> failedServers = new
      LinkedList<Pair<Long, Address>>();
  private final int recheckServersTimeout;
  private static final Log LOG = LogFactory.getLog(FailedServers.class);

  public FailedServers(Configuration conf) {
    this.recheckServersTimeout = conf.getInt(
        RpcClient.FAILED_SERVER_EXPIRY_KEY, RpcClient.FAILED_SERVER_EXPIRY_DEFAULT);
  }

  /**
   * Add an address to the list of the failed servers list.
   */
  public synchronized void addToFailedServers(Address address, Throwable throwable) {
    final long expiry = EnvironmentEdgeManager.currentTime() + recheckServersTimeout;
    failedServers.addFirst(new Pair<Long, Address>(expiry, address));
    if (LOG.isDebugEnabled()) {
      LOG.debug(
        "Added failed server with address " + address + " to list caused by "
            + throwable.toString());
    }
  }

  /**
   * Check if the server should be considered as bad. Clean the old entries of the list.
   *
   * @return true if the server is in the failed servers list
   */
  public synchronized boolean isFailedServer(final Address address) {
    if (failedServers.isEmpty()) {
      return false;
    }

    final long now = EnvironmentEdgeManager.currentTime();

    // iterate, looking for the search entry and cleaning expired entries
    Iterator<Pair<Long, Address>> it = failedServers.iterator();
    while (it.hasNext()) {
      Pair<Long, Address> cur = it.next();
      if (cur.getFirst() < now) {
        it.remove();
      } else {
        if (address.equals(cur.getSecond())) {
          return true;
        }
      }
    }

    return false;
  }
}
