/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import com.google.common.primitives.Ints;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.NotImplementedException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.DeprecatedTableDescriptor;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.TableDescriptors;
import org.apache.hadoop.hbase.TableInfoMissingException;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.protobuf.generated.ZooKeeperProtos;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZooKeeperWatcher;
import org.apache.zookeeper.KeeperException;

@InterfaceAudience.Private
public class FSTableDescriptors
implements TableDescriptors {
    private static final Log LOG = LogFactory.getLog(FSTableDescriptors.class);
    private final FileSystem fs;
    private final Path rootdir;
    private final boolean fsreadonly;
    private volatile boolean usecache;
    private volatile boolean fsvisited;
    long cachehits = 0L;
    long invocations = 0L;
    static final String TABLEINFO_FILE_PREFIX = ".tableinfo";
    static final String TABLEINFO_DIR = ".tabledesc";
    static final String TMP_DIR = ".tmp";
    private final Map<TableName, HTableDescriptor> cache = new ConcurrentHashMap<TableName, HTableDescriptor>();
    private final HTableDescriptor metaTableDescriptor;
    static final Comparator<FileStatus> TABLEINFO_FILESTATUS_COMPARATOR = new Comparator<FileStatus>(){

        @Override
        public int compare(FileStatus left, FileStatus right) {
            return right.compareTo(left);
        }
    };
    private static final PathFilter TABLEINFO_PATHFILTER = new PathFilter(){

        public boolean accept(Path p) {
            return p.getName().startsWith(FSTableDescriptors.TABLEINFO_FILE_PREFIX);
        }
    };
    static final int WIDTH_OF_SEQUENCE_ID = 10;
    private static final Pattern TABLEINFO_FILE_REGEX = Pattern.compile(".tableinfo(\\.([0-9]{10}))?$");

    public FSTableDescriptors(Configuration conf) throws IOException {
        this(conf, FSUtils.getCurrentFileSystem((Configuration)conf), FSUtils.getRootDir((Configuration)conf));
    }

    public FSTableDescriptors(Configuration conf, FileSystem fs, Path rootdir) throws IOException {
        this(conf, fs, rootdir, false, true);
    }

    public FSTableDescriptors(Configuration conf, FileSystem fs, Path rootdir, boolean fsreadonly, boolean usecache) throws IOException {
        this.fs = fs;
        this.rootdir = rootdir;
        this.fsreadonly = fsreadonly;
        this.usecache = usecache;
        this.metaTableDescriptor = HTableDescriptor.metaTableDescriptor((Configuration)conf);
    }

    @Override
    public void setCacheOn() throws IOException {
        this.cache.clear();
        this.usecache = true;
    }

    @Override
    public void setCacheOff() throws IOException {
        this.usecache = false;
        this.cache.clear();
    }

    public boolean isUsecache() {
        return this.usecache;
    }

    @Override
    public HTableDescriptor get(TableName tablename) throws IOException {
        HTableDescriptor cachedtdm;
        ++this.invocations;
        if (TableName.META_TABLE_NAME.equals((Object)tablename)) {
            ++this.cachehits;
            return this.metaTableDescriptor;
        }
        if (HConstants.HBASE_NON_USER_TABLE_DIRS.contains(tablename.getNameAsString())) {
            throw new IOException("No descriptor found for non table = " + tablename);
        }
        if (this.usecache && (cachedtdm = this.cache.get(tablename)) != null) {
            ++this.cachehits;
            return cachedtdm;
        }
        HTableDescriptor tdmt = null;
        try {
            tdmt = FSTableDescriptors.getTableDescriptorFromFs(this.fs, this.rootdir, tablename, !this.fsreadonly);
        }
        catch (NullPointerException e) {
            LOG.debug((Object)("Exception during readTableDecriptor. Current table name = " + tablename), (Throwable)e);
        }
        catch (TableInfoMissingException e) {
        }
        catch (IOException ioe) {
            LOG.debug((Object)("Exception during readTableDecriptor. Current table name = " + tablename), (Throwable)ioe);
        }
        if (this.usecache && tdmt != null) {
            this.cache.put(tablename, tdmt);
        }
        return tdmt;
    }

    @Override
    public Map<String, HTableDescriptor> getAll() throws IOException {
        TreeMap<String, HTableDescriptor> htds = new TreeMap<String, HTableDescriptor>();
        if (this.fsvisited && this.usecache) {
            for (Map.Entry<TableName, HTableDescriptor> entry : this.cache.entrySet()) {
                htds.put(entry.getKey().toString(), entry.getValue());
            }
            htds.put(HTableDescriptor.META_TABLEDESC.getTableName().getNameAsString(), HTableDescriptor.META_TABLEDESC);
        } else {
            LOG.debug((Object)"Fetching table descriptors from the filesystem.");
            boolean allvisited = true;
            for (Path d : FSUtils.getTableDirs(this.fs, this.rootdir)) {
                HTableDescriptor htd = null;
                try {
                    htd = this.get(FSUtils.getTableName((Path)d));
                }
                catch (FileNotFoundException fnfe) {
                    LOG.warn((Object)"Trouble retrieving htd", (Throwable)fnfe);
                }
                if (htd == null) {
                    allvisited = false;
                    continue;
                }
                htds.put(htd.getTableName().getNameAsString(), htd);
                this.fsvisited = allvisited;
            }
        }
        return htds;
    }

    @Override
    public Map<String, HTableDescriptor> getByNamespace(String name) throws IOException {
        TreeMap<String, HTableDescriptor> htds = new TreeMap<String, HTableDescriptor>();
        List<Path> tableDirs = FSUtils.getLocalTableDirs(this.fs, FSUtils.getNamespaceDir((Path)this.rootdir, (String)name));
        for (Path d : tableDirs) {
            HTableDescriptor htd = null;
            try {
                htd = this.get(FSUtils.getTableName((Path)d));
            }
            catch (FileNotFoundException fnfe) {
                LOG.warn((Object)"Trouble retrieving htd", (Throwable)fnfe);
            }
            if (htd == null) continue;
            htds.put(FSUtils.getTableName((Path)d).getNameAsString(), htd);
        }
        return htds;
    }

    @Override
    public void add(HTableDescriptor htd) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot add a table descriptor - in read only mode");
        }
        if (TableName.META_TABLE_NAME.equals((Object)htd.getTableName())) {
            throw new NotImplementedException();
        }
        if (HConstants.HBASE_NON_USER_TABLE_DIRS.contains(htd.getTableName().getNameAsString())) {
            throw new NotImplementedException("Cannot add a table descriptor for a reserved subdirectory name: " + htd.getNameAsString());
        }
        this.updateTableDescriptor(htd);
    }

    @Override
    public HTableDescriptor remove(TableName tablename) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot remove a table descriptor - in read only mode");
        }
        Path tabledir = this.getTableDir(tablename);
        if (this.fs.exists(tabledir) && !this.fs.delete(tabledir, true)) {
            throw new IOException("Failed delete of " + tabledir.toString());
        }
        HTableDescriptor descriptor = this.cache.remove(tablename);
        if (descriptor == null) {
            return null;
        }
        return descriptor;
    }

    public boolean isTableInfoExists(TableName tableName) throws IOException {
        return this.getTableInfoPath(tableName) != null;
    }

    private FileStatus getTableInfoPath(TableName tableName) throws IOException {
        Path tableDir = this.getTableDir(tableName);
        return this.getTableInfoPath(tableDir);
    }

    private FileStatus getTableInfoPath(Path tableDir) throws IOException {
        return FSTableDescriptors.getTableInfoPath(this.fs, tableDir, !this.fsreadonly);
    }

    public static FileStatus getTableInfoPath(FileSystem fs, Path tableDir) throws IOException {
        return FSTableDescriptors.getTableInfoPath(fs, tableDir, false);
    }

    private static FileStatus getTableInfoPath(FileSystem fs, Path tableDir, boolean removeOldFiles) throws IOException {
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        return FSTableDescriptors.getCurrentTableInfoStatus(fs, tableInfoDir, removeOldFiles);
    }

    static FileStatus getCurrentTableInfoStatus(FileSystem fs, Path dir, boolean removeOldFiles) throws IOException {
        FileStatus[] status = FSUtils.listStatus((FileSystem)fs, (Path)dir, (PathFilter)TABLEINFO_PATHFILTER);
        if (status == null || status.length < 1) {
            return null;
        }
        FileStatus mostCurrent = null;
        for (FileStatus file : status) {
            if (mostCurrent != null && TABLEINFO_FILESTATUS_COMPARATOR.compare(file, mostCurrent) >= 0) continue;
            mostCurrent = file;
        }
        if (removeOldFiles && status.length > 1) {
            for (FileStatus file : status) {
                Path path = file.getPath();
                if (file.equals((Object)mostCurrent)) continue;
                if (!fs.delete(file.getPath(), false)) {
                    LOG.warn((Object)("Failed cleanup of " + path));
                    continue;
                }
                LOG.debug((Object)("Cleaned up old tableinfo file " + path));
            }
        }
        return mostCurrent;
    }

    Path getTableDir(TableName tableName) {
        return FSUtils.getTableDir((Path)this.rootdir, (TableName)tableName);
    }

    private static String formatTableInfoSequenceId(int number) {
        byte[] b = new byte[10];
        int d = Math.abs(number);
        for (int i = b.length - 1; i >= 0; --i) {
            b[i] = (byte)(d % 10 + 48);
            d /= 10;
        }
        return Bytes.toString((byte[])b);
    }

    static int getTableInfoSequenceId(Path p) {
        if (p == null) {
            return 0;
        }
        Matcher m = TABLEINFO_FILE_REGEX.matcher(p.getName());
        if (!m.matches()) {
            throw new IllegalArgumentException(p.toString());
        }
        String suffix = m.group(2);
        if (suffix == null || suffix.length() <= 0) {
            return 0;
        }
        return Integer.parseInt(m.group(2));
    }

    static String getTableInfoFileName(int sequenceid) {
        return ".tableinfo." + FSTableDescriptors.formatTableInfoSequenceId(sequenceid);
    }

    public static HTableDescriptor getTableDescriptorFromFs(FileSystem fs, Path hbaseRootDir, TableName tableName) throws IOException {
        Path tableDir = FSUtils.getTableDir((Path)hbaseRootDir, (TableName)tableName);
        return FSTableDescriptors.getTableDescriptorFromFs(fs, tableDir);
    }

    public static HTableDescriptor getTableDescriptorFromFs(FileSystem fs, Path hbaseRootDir, TableName tableName, boolean rewritePb) throws IOException {
        Path tableDir = FSUtils.getTableDir((Path)hbaseRootDir, (TableName)tableName);
        return FSTableDescriptors.getTableDescriptorFromFs(fs, tableDir, rewritePb);
    }

    public static HTableDescriptor getTableDescriptorFromFs(FileSystem fs, Path tableDir) throws IOException {
        return FSTableDescriptors.getTableDescriptorFromFs(fs, tableDir, false);
    }

    public static HTableDescriptor getTableDescriptorFromFs(FileSystem fs, Path tableDir, boolean rewritePb) throws IOException {
        FileStatus status = FSTableDescriptors.getTableInfoPath(fs, tableDir, false);
        if (status == null) {
            throw new TableInfoMissingException("No table descriptor file under " + tableDir);
        }
        return (HTableDescriptor)FSTableDescriptors.readTableDescriptor((FileSystem)fs, (FileStatus)status, (boolean)rewritePb).first;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static Pair<HTableDescriptor, ZooKeeperProtos.Table> readTableDescriptor(FileSystem fs, FileStatus status, boolean rewritePb) throws IOException {
        int len = Ints.checkedCast((long)status.getLen());
        byte[] content = new byte[len];
        try (FSDataInputStream fsDataInputStream = fs.open(status.getPath());){
            fsDataInputStream.readFully(content);
        }
        HTableDescriptor htd = null;
        ZooKeeperProtos.Table tableState = null;
        try {
            htd = HTableDescriptor.parseFrom((byte[])content);
        }
        catch (DeserializationException e) {
            try {
                DeprecatedTableDescriptor dtd = DeprecatedTableDescriptor.parseFrom((byte[])content);
                htd = dtd.getHTableDescriptor();
                tableState = dtd.getTableState();
                LOG.warn((Object)("Found incompatible table descriptor from 1.7.0 version: " + dtd.getHTableDescriptor().getTableName() + " state: " + tableState.getState().name()));
                if (rewritePb) {
                    LOG.warn((Object)("converting to new format for table " + htd.getTableName()));
                    FSTableDescriptors.rewriteTableDescriptor(fs, status, htd);
                    rewritePb = false;
                }
            }
            catch (DeserializationException e1) {
                throw new IOException("content=" + Bytes.toShort((byte[])content), e1);
            }
        }
        if (rewritePb && !ProtobufUtil.isPBMagicPrefix((byte[])content)) {
            FSTableDescriptors.rewriteTableDescriptor(fs, status, htd);
        }
        return new Pair((Object)htd, tableState);
    }

    private static void rewriteTableDescriptor(FileSystem fs, FileStatus status, HTableDescriptor td) throws IOException {
        Path tableInfoDir = status.getPath().getParent();
        Path tableDir = tableInfoDir.getParent();
        FSTableDescriptors.writeTableDescriptor(fs, td, tableDir, status);
    }

    Path updateTableDescriptor(HTableDescriptor htd) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot update a table descriptor - in read only mode");
        }
        Path tableDir = this.getTableDir(htd.getTableName());
        Path p = FSTableDescriptors.writeTableDescriptor(this.fs, htd, tableDir, this.getTableInfoPath(tableDir));
        if (p == null) {
            throw new IOException("Failed update");
        }
        LOG.info((Object)("Updated tableinfo=" + p));
        if (this.usecache) {
            this.cache.put(htd.getTableName(), htd);
        }
        return p;
    }

    public void deleteTableDescriptorIfExists(TableName tableName) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot delete a table descriptor - in read only mode");
        }
        Path tableDir = this.getTableDir(tableName);
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        FSTableDescriptors.deleteTableDescriptorFiles(this.fs, tableInfoDir, Integer.MAX_VALUE);
    }

    private static void deleteTableDescriptorFiles(FileSystem fs, Path dir, int maxSequenceId) throws IOException {
        FileStatus[] status;
        for (FileStatus file : status = FSUtils.listStatus((FileSystem)fs, (Path)dir, (PathFilter)TABLEINFO_PATHFILTER)) {
            Path path = file.getPath();
            int sequenceId = FSTableDescriptors.getTableInfoSequenceId(path);
            if (sequenceId > maxSequenceId) continue;
            boolean success = FSUtils.delete((FileSystem)fs, (Path)path, (boolean)false);
            if (success) {
                LOG.debug((Object)("Deleted table descriptor at " + path));
                continue;
            }
            LOG.error((Object)("Failed to delete descriptor at " + path));
        }
    }

    private static Path writeTableDescriptor(FileSystem fs, HTableDescriptor htd, Path tableDir, FileStatus currentDescriptorFile) throws IOException {
        int currentSequenceId;
        Path tmpTableDir = new Path(tableDir, TMP_DIR);
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        int newSequenceId = currentSequenceId = currentDescriptorFile == null ? 0 : FSTableDescriptors.getTableInfoSequenceId(currentDescriptorFile.getPath());
        int retries = 10;
        int retrymax = currentSequenceId + retries;
        Path tableInfoDirPath = null;
        do {
            String filename;
            Path tempPath;
            if (fs.exists(tempPath = new Path(tmpTableDir, filename = FSTableDescriptors.getTableInfoFileName(++newSequenceId)))) {
                LOG.debug((Object)(tempPath + " exists; retrying up to " + retries + " times"));
                continue;
            }
            tableInfoDirPath = new Path(tableInfoDir, filename);
            try {
                FSTableDescriptors.writeHTD(fs, tempPath, htd);
                fs.mkdirs(tableInfoDirPath.getParent());
                if (!fs.rename(tempPath, tableInfoDirPath)) {
                    throw new IOException("Failed rename of " + tempPath + " to " + tableInfoDirPath);
                }
                LOG.debug((Object)("Wrote descriptor into: " + tableInfoDirPath));
                break;
            }
            catch (IOException ioe) {
                LOG.debug((Object)"Failed write and/or rename; retrying", (Throwable)ioe);
                if (!FSUtils.deleteDirectory((FileSystem)fs, (Path)tempPath)) {
                    LOG.warn((Object)("Failed cleanup of " + tempPath));
                }
                tableInfoDirPath = null;
            }
        } while (newSequenceId < retrymax);
        if (tableInfoDirPath != null) {
            FSTableDescriptors.deleteTableDescriptorFiles(fs, tableInfoDir, newSequenceId - 1);
        }
        return tableInfoDirPath;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void writeHTD(FileSystem fs, Path p, HTableDescriptor htd) throws IOException {
        try (FSDataOutputStream out = fs.create(p, false);){
            out.write(htd.toByteArray());
        }
    }

    public boolean createTableDescriptor(HTableDescriptor htd) throws IOException {
        return this.createTableDescriptor(htd, false);
    }

    public boolean createTableDescriptor(HTableDescriptor htd, boolean forceCreation) throws IOException {
        Path tableDir = this.getTableDir(htd.getTableName());
        return this.createTableDescriptorForTableDirectory(tableDir, htd, forceCreation);
    }

    public boolean createTableDescriptorForTableDirectory(Path tableDir, HTableDescriptor htd, boolean forceCreation) throws IOException {
        Path p;
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot create a table descriptor - in read only mode");
        }
        FileStatus status = FSTableDescriptors.getTableInfoPath(this.fs, tableDir);
        if (status != null) {
            LOG.debug((Object)("Current tableInfoPath = " + status.getPath()));
            if (!forceCreation && this.fs.exists(status.getPath()) && status.getLen() > 0L && ((HTableDescriptor)FSTableDescriptors.readTableDescriptor((FileSystem)this.fs, (FileStatus)status, (boolean)false).first).equals((Object)htd)) {
                LOG.debug((Object)"TableInfo already exists.. Skipping creation");
                return false;
            }
        }
        return (p = FSTableDescriptors.writeTableDescriptor(this.fs, htd, tableDir, status)) != null;
    }

    public void repairHBase170TableDescriptors(ZooKeeperWatcher zkw) throws IOException, KeeperException {
        LOG.info((Object)"Attempting to repair HBase 1.7.0 tables, if any.");
        for (Path tableDir : FSUtils.getTableDirs(this.fs, this.rootdir)) {
            FileStatus status = FSTableDescriptors.getTableInfoPath(this.fs, tableDir, false);
            if (status == null) {
                LOG.warn((Object)("No table descriptor file under " + tableDir));
                continue;
            }
            Pair<HTableDescriptor, ZooKeeperProtos.Table> result = FSTableDescriptors.readTableDescriptor(this.fs, status, true);
            if (result.second == null) continue;
            TableName tableName = ((HTableDescriptor)result.first).getTableName();
            ZooKeeperProtos.Table tableState = (ZooKeeperProtos.Table)result.second;
            LOG.warn((Object)("Rewriting ZK Table state for table " + tableName));
            String znode = ZKUtil.joinZNode((String)zkw.tableZNode, (String)tableName.getNameAsString());
            if (ZKUtil.checkExists((ZooKeeperWatcher)zkw, (String)znode) != -1) {
                LOG.warn((Object)("Table state znode already exists for table: " + tableName + ". Ignoring."));
                continue;
            }
            ZKUtil.createAndFailSilent((ZooKeeperWatcher)zkw, (String)znode);
            byte[] data = ProtobufUtil.prependPBMagic((byte[])tableState.toByteArray());
            ZKUtil.setData((ZooKeeperWatcher)zkw, (String)znode, (byte[])data);
            LOG.info((Object)("Repaired ZK table state for table: " + tableName));
        }
    }
}

