/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.volume;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.util.Properties;
import java.util.UUID;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.GetSpaceUsed;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.server.datanode.StorageLocation;
import org.apache.hadoop.ozone.common.InconsistentStorageStateException;
import org.apache.hadoop.ozone.container.common.DataNodeLayoutVersion;
import org.apache.hadoop.ozone.container.common.helpers.DatanodeVersionFile;
import org.apache.hadoop.ozone.container.common.impl.ChunkLayOutVersion;
import org.apache.hadoop.ozone.container.common.utils.HddsVolumeUtil;
import org.apache.hadoop.ozone.container.common.volume.VolumeIOStats;
import org.apache.hadoop.ozone.container.common.volume.VolumeInfo;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class HddsVolume {
    private static final Logger LOG = LoggerFactory.getLogger(HddsVolume.class);
    public static final String HDDS_VOLUME_DIR = "hdds";
    private final File hddsRootDir;
    private final VolumeInfo volumeInfo;
    private VolumeState state;
    private final VolumeIOStats volumeIOStats;
    private String storageID;
    private String clusterID;
    private String datanodeUuid;
    private long cTime;
    private int layoutVersion;

    private HddsVolume(Builder b) throws IOException {
        if (!b.failedVolume) {
            StorageLocation location = StorageLocation.parse((String)b.volumeRootStr);
            this.hddsRootDir = new File(location.getUri().getPath(), HDDS_VOLUME_DIR);
            this.state = VolumeState.NOT_INITIALIZED;
            this.clusterID = b.clusterID;
            this.datanodeUuid = b.datanodeUuid;
            this.volumeIOStats = new VolumeIOStats();
            VolumeInfo.Builder volumeBuilder = new VolumeInfo.Builder(b.volumeRootStr, b.conf).storageType(b.storageType).configuredCapacity(b.configuredCapacity);
            this.volumeInfo = volumeBuilder.build();
            LOG.info("Creating Volume: " + this.hddsRootDir + " of  storage type : " + b.storageType + " and capacity : " + this.volumeInfo.getCapacity());
            this.initialize();
        } else {
            this.hddsRootDir = new File(b.volumeRootStr);
            this.volumeIOStats = null;
            this.volumeInfo = null;
            this.storageID = UUID.randomUUID().toString();
            this.state = VolumeState.FAILED;
        }
    }

    public VolumeInfo getVolumeInfo() {
        return this.volumeInfo;
    }

    private void initialize() throws IOException {
        VolumeState intialVolumeState = this.analyzeVolumeState();
        switch (intialVolumeState) {
            case NON_EXISTENT: {
                if (!this.hddsRootDir.mkdir()) {
                    throw new IOException("Cannot create directory " + this.hddsRootDir);
                }
                this.setState(VolumeState.NOT_FORMATTED);
                this.createVersionFile();
                break;
            }
            case NOT_FORMATTED: {
                this.createVersionFile();
                break;
            }
            case NOT_INITIALIZED: {
                this.readVersionFile();
                this.setState(VolumeState.NORMAL);
                break;
            }
            case INCONSISTENT: {
                throw new IOException("Volume is in an " + (Object)((Object)VolumeState.INCONSISTENT) + " state. Skipped loading volume: " + this.hddsRootDir.getPath());
            }
            default: {
                throw new IOException("Unrecognized initial state : " + (Object)((Object)intialVolumeState) + "of volume : " + this.hddsRootDir);
            }
        }
    }

    private VolumeState analyzeVolumeState() {
        if (!this.hddsRootDir.exists()) {
            return VolumeState.NON_EXISTENT;
        }
        if (!this.hddsRootDir.isDirectory()) {
            return VolumeState.INCONSISTENT;
        }
        File[] files = this.hddsRootDir.listFiles();
        if (files == null || files.length == 0) {
            return VolumeState.NOT_FORMATTED;
        }
        if (!this.getVersionFile().exists()) {
            return VolumeState.INCONSISTENT;
        }
        return VolumeState.NOT_INITIALIZED;
    }

    public void format(String cid) throws IOException {
        Preconditions.checkNotNull((Object)cid, (Object)"clusterID cannot be null while formatting Volume");
        this.clusterID = cid;
        this.initialize();
    }

    private void createVersionFile() throws IOException {
        this.storageID = HddsVolumeUtil.generateUuid();
        this.cTime = Time.now();
        this.layoutVersion = ChunkLayOutVersion.getLatestVersion().getVersion();
        if (this.clusterID == null || this.datanodeUuid == null) {
            LOG.debug("ClusterID not available. Cannot format the volume {}", (Object)this.hddsRootDir.getPath());
            this.setState(VolumeState.NOT_FORMATTED);
        } else {
            this.writeVersionFile();
            this.setState(VolumeState.NORMAL);
        }
    }

    private void writeVersionFile() throws IOException {
        Preconditions.checkNotNull((Object)this.storageID, (Object)"StorageID cannot be null in Version File");
        Preconditions.checkNotNull((Object)this.clusterID, (Object)"ClusterID cannot be null in Version File");
        Preconditions.checkNotNull((Object)this.datanodeUuid, (Object)"DatanodeUUID cannot be null in Version File");
        Preconditions.checkArgument((this.cTime > 0L ? 1 : 0) != 0, (Object)"Creation Time should be positive");
        Preconditions.checkArgument((this.layoutVersion == DataNodeLayoutVersion.getLatestVersion().getVersion() ? 1 : 0) != 0, (Object)"Version File should have the latest LayOutVersion");
        File versionFile = this.getVersionFile();
        LOG.debug("Writing Version file to disk, {}", (Object)versionFile);
        DatanodeVersionFile dnVersionFile = new DatanodeVersionFile(this.storageID, this.clusterID, this.datanodeUuid, this.cTime, this.layoutVersion);
        dnVersionFile.createVersionFile(versionFile);
    }

    private void readVersionFile() throws IOException {
        File versionFile = this.getVersionFile();
        Properties props = DatanodeVersionFile.readFrom(versionFile);
        if (props.isEmpty()) {
            throw new InconsistentStorageStateException("Version file " + versionFile + " is missing");
        }
        LOG.debug("Reading Version file from disk, {}", (Object)versionFile);
        this.storageID = HddsVolumeUtil.getStorageID(props, versionFile);
        this.clusterID = HddsVolumeUtil.getClusterID(props, versionFile, this.clusterID);
        this.datanodeUuid = HddsVolumeUtil.getDatanodeUUID(props, versionFile, this.datanodeUuid);
        this.cTime = HddsVolumeUtil.getCreationTime(props, versionFile);
        this.layoutVersion = HddsVolumeUtil.getLayOutVersion(props, versionFile);
    }

    private File getVersionFile() {
        return HddsVolumeUtil.getVersionFile(this.hddsRootDir);
    }

    public File getHddsRootDir() {
        return this.hddsRootDir;
    }

    public StorageType getStorageType() {
        if (this.volumeInfo != null) {
            return this.volumeInfo.getStorageType();
        }
        return StorageType.DEFAULT;
    }

    public String getStorageID() {
        return this.storageID;
    }

    public String getClusterID() {
        return this.clusterID;
    }

    public String getDatanodeUuid() {
        return this.datanodeUuid;
    }

    public long getCTime() {
        return this.cTime;
    }

    public int getLayoutVersion() {
        return this.layoutVersion;
    }

    public VolumeState getStorageState() {
        return this.state;
    }

    public long getCapacity() throws IOException {
        if (this.volumeInfo != null) {
            return this.volumeInfo.getCapacity();
        }
        return 0L;
    }

    public long getAvailable() throws IOException {
        if (this.volumeInfo != null) {
            return this.volumeInfo.getAvailable();
        }
        return 0L;
    }

    public void setState(VolumeState state) {
        this.state = state;
    }

    public boolean isFailed() {
        return this.state == VolumeState.FAILED;
    }

    public VolumeIOStats getVolumeIOStats() {
        return this.volumeIOStats;
    }

    public void failVolume() {
        this.setState(VolumeState.FAILED);
        if (this.volumeInfo != null) {
            this.volumeInfo.shutdownUsageThread();
        }
    }

    public void shutdown() {
        this.state = VolumeState.NON_EXISTENT;
        if (this.volumeInfo != null) {
            this.volumeInfo.shutdownUsageThread();
        }
    }

    @VisibleForTesting
    public void setScmUsageForTesting(GetSpaceUsed scmUsageForTest) {
        if (this.volumeInfo != null) {
            this.volumeInfo.setScmUsageForTesting(scmUsageForTest);
        }
    }

    public static enum VolumeState {
        NORMAL,
        FAILED,
        NON_EXISTENT,
        INCONSISTENT,
        NOT_FORMATTED,
        NOT_INITIALIZED;

    }

    public static class Builder {
        private final String volumeRootStr;
        private Configuration conf;
        private StorageType storageType;
        private long configuredCapacity;
        private String datanodeUuid;
        private String clusterID;
        private boolean failedVolume = false;

        public Builder(String rootDirStr) {
            this.volumeRootStr = rootDirStr;
        }

        public Builder conf(Configuration config) {
            this.conf = config;
            return this;
        }

        public Builder storageType(StorageType st) {
            this.storageType = st;
            return this;
        }

        public Builder configuredCapacity(long capacity) {
            this.configuredCapacity = capacity;
            return this;
        }

        public Builder datanodeUuid(String datanodeUUID) {
            this.datanodeUuid = datanodeUUID;
            return this;
        }

        public Builder clusterID(String cid) {
            this.clusterID = cid;
            return this;
        }

        public Builder failedVolume(boolean failed) {
            this.failedVolume = failed;
            return this;
        }

        public HddsVolume build() throws IOException {
            return new HddsVolume(this);
        }
    }
}

