/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.server;

import com.google.common.annotations.VisibleForTesting;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.ContainerInfo;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.server.SCMDatanodeProtocolServer;
import org.apache.hadoop.hdds.server.events.EventHandler;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.hdds.server.events.EventQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCMChillModeManager
implements EventHandler<SCMDatanodeProtocolServer.NodeRegistrationContainerReport> {
    private static final Logger LOG = LoggerFactory.getLogger(SCMChillModeManager.class);
    private AtomicBoolean inChillMode = new AtomicBoolean(true);
    private AtomicLong containerWithMinReplicas = new AtomicLong(0L);
    private Map<String, ChillModeExitRule> exitRules = new HashMap<String, ChillModeExitRule>(1);
    private Configuration config;
    private static final String CONT_EXIT_RULE = "ContainerChillModeRule";
    private static final String DN_EXIT_RULE = "DataNodeChillModeRule";
    private final EventQueue eventPublisher;

    SCMChillModeManager(Configuration conf, List<ContainerInfo> allContainers, EventQueue eventQueue) {
        this.config = conf;
        this.eventPublisher = eventQueue;
        this.exitRules.put(CONT_EXIT_RULE, new ContainerChillModeRule(this.config, allContainers));
        this.exitRules.put(DN_EXIT_RULE, new DataNodeChillModeRule(this.config));
        if (!conf.getBoolean("hdds.scm.chillmode.enabled", true)) {
            this.exitChillMode((EventPublisher)eventQueue);
        }
        this.emitChillModeStatus();
    }

    @VisibleForTesting
    public void emitChillModeStatus() {
        this.eventPublisher.fireEvent(SCMEvents.CHILL_MODE_STATUS, (Object)this.inChillMode.get());
    }

    private void validateChillModeExitRules(EventPublisher eventQueue) {
        for (ChillModeExitRule exitRule : this.exitRules.values()) {
            if (exitRule.validate()) continue;
            return;
        }
        this.exitChillMode(eventQueue);
    }

    @VisibleForTesting
    public void exitChillMode(EventPublisher eventQueue) {
        LOG.info("SCM exiting chill mode.");
        this.setInChillMode(false);
        for (ChillModeExitRule e : this.exitRules.values()) {
            e.cleanup();
        }
        this.emitChillModeStatus();
    }

    public void onMessage(SCMDatanodeProtocolServer.NodeRegistrationContainerReport nodeRegistrationContainerReport, EventPublisher publisher) {
        if (this.getInChillMode()) {
            this.exitRules.get(CONT_EXIT_RULE).process(nodeRegistrationContainerReport);
            this.exitRules.get(DN_EXIT_RULE).process(nodeRegistrationContainerReport);
            this.validateChillModeExitRules(publisher);
        }
    }

    public boolean getInChillMode() {
        return this.inChillMode.get();
    }

    public void setInChillMode(boolean inChillMode) {
        this.inChillMode.set(inChillMode);
    }

    @VisibleForTesting
    public static Logger getLogger() {
        return LOG;
    }

    @VisibleForTesting
    public double getCurrentContainerThreshold() {
        return ((ContainerChillModeRule)this.exitRules.get(CONT_EXIT_RULE)).getCurrentContainerThreshold();
    }

    public static class ChillModeRestrictedOps {
        private static EnumSet restrictedOps = EnumSet.noneOf(HddsProtos.ScmOps.class);

        public static boolean isRestrictedInChillMode(HddsProtos.ScmOps opName) {
            return restrictedOps.contains(opName);
        }

        static {
            restrictedOps.add(HddsProtos.ScmOps.allocateBlock);
            restrictedOps.add(HddsProtos.ScmOps.allocateContainer);
        }
    }

    public class DataNodeChillModeRule
    implements ChillModeExitRule<SCMDatanodeProtocolServer.NodeRegistrationContainerReport> {
        private int requiredDns;
        private int registeredDns = 0;
        private HashSet<UUID> registeredDnSet;

        public DataNodeChillModeRule(Configuration conf) {
            this.requiredDns = conf.getInt("hdds.scm.chillmode.min.datanode", 1);
            this.registeredDnSet = new HashSet(this.requiredDns * 2);
        }

        @Override
        public boolean validate() {
            return this.registeredDns >= this.requiredDns;
        }

        @VisibleForTesting
        public double getRegisteredDataNodes() {
            return this.registeredDns;
        }

        @Override
        public void process(SCMDatanodeProtocolServer.NodeRegistrationContainerReport reportsProto) {
            if (this.requiredDns == 0) {
                return;
            }
            if (SCMChillModeManager.this.inChillMode.get()) {
                this.registeredDnSet.add(reportsProto.getDatanodeDetails().getUuid());
                this.registeredDns = this.registeredDnSet.size();
                LOG.info("SCM in chill mode. {} DataNodes registered, {} required.", (Object)this.registeredDns, (Object)this.requiredDns);
            }
        }

        @Override
        public void cleanup() {
            this.registeredDnSet.clear();
        }
    }

    public class ContainerChillModeRule
    implements ChillModeExitRule<SCMDatanodeProtocolServer.NodeRegistrationContainerReport> {
        private double chillModeCutoff;
        private Map<Long, ContainerInfo> containerMap;
        private double maxContainer;

        public ContainerChillModeRule(Configuration conf, List<ContainerInfo> containers) {
            this.chillModeCutoff = conf.getDouble("hdds.scm.chillmode.threshold.pct", 0.99);
            this.containerMap = new ConcurrentHashMap<Long, ContainerInfo>();
            if (containers != null) {
                containers.forEach(c -> {
                    if (c != null && c.getState() != null && !c.getState().equals((Object)HddsProtos.LifeCycleState.ALLOCATED)) {
                        this.containerMap.put(c.getContainerID(), (ContainerInfo)c);
                    }
                });
                this.maxContainer = this.containerMap.size();
            }
        }

        @Override
        public boolean validate() {
            if (this.maxContainer == 0.0) {
                return true;
            }
            return this.getCurrentContainerThreshold() >= this.chillModeCutoff;
        }

        @VisibleForTesting
        public double getCurrentContainerThreshold() {
            if (this.maxContainer == 0.0) {
                return 1.0;
            }
            return SCMChillModeManager.this.containerWithMinReplicas.doubleValue() / this.maxContainer;
        }

        @Override
        public void process(SCMDatanodeProtocolServer.NodeRegistrationContainerReport reportsProto) {
            if (this.maxContainer == 0.0) {
                return;
            }
            ((StorageContainerDatanodeProtocolProtos.ContainerReportsProto)reportsProto.getReport()).getReportsList().forEach(c -> {
                if (this.containerMap.containsKey(c.getContainerID()) && this.containerMap.remove(c.getContainerID()) != null) {
                    SCMChillModeManager.this.containerWithMinReplicas.getAndAdd(1L);
                }
            });
            if (SCMChillModeManager.this.inChillMode.get()) {
                LOG.info("SCM in chill mode. {} % containers have at least one reported replica.", (Object)((double)SCMChillModeManager.this.containerWithMinReplicas.get() / this.maxContainer * 100.0));
            }
        }

        @Override
        public void cleanup() {
            this.containerMap.clear();
        }
    }

    public static interface ChillModeExitRule<T> {
        public boolean validate();

        public void process(T var1);

        public void cleanup();
    }
}

