/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.pipelines;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.StorageUnit;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.common.helpers.Pipeline;
import org.apache.hadoop.hdds.scm.container.common.helpers.PipelineID;
import org.apache.hadoop.hdds.scm.container.placement.algorithms.ContainerPlacementPolicy;
import org.apache.hadoop.hdds.scm.container.placement.algorithms.SCMContainerPlacementRandom;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.exceptions.SCMException;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.pipelines.PipelineManager;
import org.apache.hadoop.hdds.scm.pipelines.PipelineStateManager;
import org.apache.hadoop.hdds.scm.pipelines.ratis.RatisManagerImpl;
import org.apache.hadoop.hdds.scm.pipelines.standalone.StandaloneManagerImpl;
import org.apache.hadoop.hdds.server.ServerUtils;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.ozone.lease.Lease;
import org.apache.hadoop.ozone.lease.LeaseException;
import org.apache.hadoop.ozone.lease.LeaseManager;
import org.apache.hadoop.utils.MetadataStore;
import org.apache.hadoop.utils.MetadataStoreBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PipelineSelector {
    private static final Logger LOG = LoggerFactory.getLogger(PipelineSelector.class);
    private final ContainerPlacementPolicy placementPolicy;
    private final Map<HddsProtos.ReplicationType, PipelineManager> pipelineManagerMap;
    private final Configuration conf;
    private final EventPublisher eventPublisher;
    private final long containerSize;
    private final MetadataStore pipelineStore;
    private final PipelineStateManager stateManager;
    private final NodeManager nodeManager;
    private final Map<PipelineID, HashSet<ContainerID>> pipeline2ContainerMap;
    private final Map<PipelineID, Pipeline> pipelineMap;
    private final LeaseManager<Pipeline> pipelineLeaseManager;

    public PipelineSelector(NodeManager nodeManager, Configuration conf, EventPublisher eventPublisher, int cacheSizeMB) throws IOException {
        this.conf = conf;
        this.eventPublisher = eventPublisher;
        this.placementPolicy = PipelineSelector.createContainerPlacementPolicy(nodeManager, conf);
        this.containerSize = (long)this.conf.getStorageSize("ozone.scm.container.size", "5GB", StorageUnit.BYTES);
        this.pipelineMap = new ConcurrentHashMap<PipelineID, Pipeline>();
        this.pipelineManagerMap = new HashMap<HddsProtos.ReplicationType, PipelineManager>();
        this.pipelineManagerMap.put(HddsProtos.ReplicationType.STAND_ALONE, new StandaloneManagerImpl(nodeManager, this.placementPolicy, this.containerSize));
        this.pipelineManagerMap.put(HddsProtos.ReplicationType.RATIS, new RatisManagerImpl(nodeManager, this.placementPolicy, this.containerSize, conf));
        long pipelineCreationLeaseTimeout = conf.getTimeDuration("ozone.scm.pipeline.creation.lease.timeout", "60s", TimeUnit.MILLISECONDS);
        this.pipelineLeaseManager = new LeaseManager("PipelineCreation", pipelineCreationLeaseTimeout);
        this.pipelineLeaseManager.start();
        this.stateManager = new PipelineStateManager();
        this.nodeManager = nodeManager;
        this.pipeline2ContainerMap = new HashMap<PipelineID, HashSet<ContainerID>>();
        File metaDir = ServerUtils.getOzoneMetaDirPath((Configuration)conf);
        File containerDBPath = new File(metaDir, "scm-pipeline.db");
        this.pipelineStore = MetadataStoreBuilder.newBuilder().setConf(conf).setDbFile(containerDBPath).setCacheSize((long)cacheSizeMB * 0x100000L).build();
        this.reloadExistingPipelines();
    }

    private void reloadExistingPipelines() throws IOException {
        if (this.pipelineStore.isEmpty()) {
            return;
        }
        List range = this.pipelineStore.getSequentialRangeKVs(null, Integer.MAX_VALUE, null);
        for (Map.Entry entry : range) {
            Pipeline pipeline = Pipeline.getFromProtoBuf((HddsProtos.Pipeline)((HddsProtos.Pipeline)HddsProtos.Pipeline.PARSER.parseFrom((byte[])entry.getValue())));
            Preconditions.checkNotNull((Object)pipeline);
            this.addExistingPipeline(pipeline);
        }
    }

    @VisibleForTesting
    public Set<ContainerID> getOpenContainerIDsByPipeline(PipelineID pipelineID) {
        return this.pipeline2ContainerMap.get(pipelineID);
    }

    public void addContainerToPipeline(PipelineID pipelineID, long containerID) {
        this.pipeline2ContainerMap.get(pipelineID).add(ContainerID.valueof((long)containerID));
    }

    public void removeContainerFromPipeline(PipelineID pipelineID, long containerID) throws IOException {
        if (this.pipeline2ContainerMap.containsKey(pipelineID)) {
            this.pipeline2ContainerMap.get(pipelineID).remove(ContainerID.valueof((long)containerID));
            this.closePipelineIfNoOpenContainers(this.pipelineMap.get(pipelineID));
        } else {
            LOG.warn("Cannot remove container #{} from pipeline. Pipeline #{} not found.", (Object)containerID, (Object)pipelineID.getId());
        }
    }

    public static Pipeline newPipelineFromNodes(List<DatanodeDetails> nodes, HddsProtos.ReplicationType replicationType, HddsProtos.ReplicationFactor replicationFactor, PipelineID id) {
        Preconditions.checkNotNull(nodes);
        Preconditions.checkArgument((nodes.size() > 0 ? 1 : 0) != 0);
        String leaderId = nodes.get(0).getUuidString();
        Pipeline pipeline = new Pipeline(leaderId, HddsProtos.LifeCycleState.ALLOCATED, replicationType, replicationFactor, id);
        for (DatanodeDetails node : nodes) {
            pipeline.addMember(node);
        }
        return pipeline;
    }

    private static ContainerPlacementPolicy createContainerPlacementPolicy(NodeManager nodeManager, Configuration conf) {
        Class implClass = conf.getClass("ozone.scm.container.placement.impl", SCMContainerPlacementRandom.class);
        try {
            Constructor ctor = implClass.getDeclaredConstructor(NodeManager.class, Configuration.class);
            return (ContainerPlacementPolicy)ctor.newInstance(nodeManager, conf);
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (InvocationTargetException e) {
            throw new RuntimeException(implClass.getName() + " could not be constructed.", e.getCause());
        }
        catch (Exception e) {
            LOG.error("Unhandled exception occurred, Placement policy will not be functional.");
            throw new IllegalArgumentException("Unable to load ContainerPlacementPolicy", e);
        }
    }

    public Pipeline getReplicationPipeline(HddsProtos.ReplicationType replicationType, HddsProtos.ReplicationFactor replicationFactor) throws IOException {
        PipelineManager manager = this.pipelineManagerMap.get(replicationType);
        Preconditions.checkNotNull((Object)manager, (Object)"Found invalid pipeline manager");
        LOG.debug("Getting replication pipeline forReplicationType {} : ReplicationFactor {}", (Object)replicationType.toString(), (Object)replicationFactor.toString());
        Pipeline pipeline = manager.createPipeline(replicationFactor, replicationType);
        if (pipeline == null) {
            PipelineID pipelineId = manager.getPipeline(replicationFactor, replicationType);
            if (pipelineId == null) {
                throw new SCMException(SCMException.ResultCodes.FAILED_TO_FIND_ACTIVE_PIPELINE);
            }
            pipeline = this.pipelineMap.get(pipelineId);
            Preconditions.checkArgument((pipeline.getLifeCycleState() == HddsProtos.LifeCycleState.OPEN ? 1 : 0) != 0);
        } else {
            this.pipelineStore.put(pipeline.getId().getProtobuf().toByteArray(), pipeline.getProtobufMessage().toByteArray());
            this.updatePipelineState(pipeline, HddsProtos.LifeCycleEvent.CREATE);
            manager.initializePipeline(pipeline);
            this.updatePipelineState(pipeline, HddsProtos.LifeCycleEvent.CREATED);
        }
        return pipeline;
    }

    public Pipeline getPipeline(PipelineID pipelineID) {
        return this.pipelineMap.get(pipelineID);
    }

    public void finalizePipeline(Pipeline pipeline) throws IOException {
        PipelineManager manager = this.pipelineManagerMap.get(pipeline.getType());
        Preconditions.checkNotNull((Object)manager, (Object)"Found invalid pipeline manager");
        if (pipeline.getLifeCycleState() == HddsProtos.LifeCycleState.CLOSING || pipeline.getLifeCycleState() == HddsProtos.LifeCycleState.CLOSED) {
            LOG.debug("pipeline:{} already in closing state, skipping", (Object)pipeline.getId());
            return;
        }
        if (manager.finalizePipeline(pipeline)) {
            LOG.info("Finalizing pipeline. pipelineID: {}", (Object)pipeline.getId());
            this.updatePipelineState(pipeline, HddsProtos.LifeCycleEvent.FINALIZE);
            this.closePipelineIfNoOpenContainers(pipeline);
        }
    }

    private void closePipelineIfNoOpenContainers(Pipeline pipeline) throws IOException {
        if (pipeline.getLifeCycleState() != HddsProtos.LifeCycleState.CLOSING) {
            return;
        }
        HashSet<ContainerID> containerIDS = this.pipeline2ContainerMap.get(pipeline.getId());
        if (containerIDS.size() == 0) {
            this.updatePipelineState(pipeline, HddsProtos.LifeCycleEvent.CLOSE);
            LOG.info("Closing pipeline. pipelineID: {}", (Object)pipeline.getId());
        }
    }

    private void closePipeline(Pipeline pipeline) throws IOException {
        PipelineManager manager = this.pipelineManagerMap.get(pipeline.getType());
        Preconditions.checkNotNull((Object)manager, (Object)"Found invalid pipeline manager");
        LOG.debug("Closing pipeline. pipelineID: {}", (Object)pipeline.getId());
        HashSet<ContainerID> containers = this.pipeline2ContainerMap.get(pipeline.getId());
        Preconditions.checkArgument((containers.size() == 0 ? 1 : 0) != 0);
        manager.closePipeline(pipeline);
    }

    public List<Pipeline> listPipelines() {
        return Collections.unmodifiableList(new ArrayList<Pipeline>(this.pipelineMap.values()));
    }

    public void closePipeline(PipelineID pipelineID) throws IOException {
        Pipeline pipeline = this.pipelineMap.get(pipelineID);
        if (pipeline == null) {
            LOG.warn("Cannot close the pipeline. {} not found!", (Object)pipelineID);
            return;
        }
        LOG.info("Closing pipeline. pipelineID: {}", (Object)pipeline.getId());
        this.finalizePipeline(pipeline);
        if (pipeline.getLifeCycleState() != HddsProtos.LifeCycleState.CLOSED) {
            this.pipelineManagerMap.get(pipeline.getType()).closePipeline(pipeline);
            this.pipeline2ContainerMap.remove(pipeline.getId());
            this.nodeManager.removePipeline(pipeline);
            this.pipelineMap.remove(pipeline.getId());
        }
        this.pipelineStore.delete(pipelineID.getProtobuf().toByteArray());
    }

    private void addOpenPipeline(Pipeline pipeline) {
        PipelineManager manager = this.pipelineManagerMap.get(pipeline.getType());
        Preconditions.checkNotNull((Object)manager, (Object)"Found invalid pipeline manager");
        LOG.debug("Adding Open pipeline. pipelineID: {}", (Object)pipeline.getId());
        manager.addOpenPipeline(pipeline);
    }

    private void closeContainersByPipeline(Pipeline pipeline) {
        HashSet<ContainerID> containers = this.pipeline2ContainerMap.get(pipeline.getId());
        for (ContainerID id : containers) {
            this.eventPublisher.fireEvent(SCMEvents.CLOSE_CONTAINER, (Object)id);
        }
    }

    private void addExistingPipeline(Pipeline pipeline) throws IOException {
        HddsProtos.LifeCycleState state = pipeline.getLifeCycleState();
        switch (state) {
            case ALLOCATED: {
                break;
            }
            case CREATING: 
            case OPEN: 
            case CLOSING: {
                this.pipelineMap.put(pipeline.getId(), pipeline);
                this.pipeline2ContainerMap.put(pipeline.getId(), new HashSet());
                this.nodeManager.addPipeline(pipeline);
                pipeline.resetPipeline();
                break;
            }
            case CLOSED: {
                break;
            }
            default: {
                throw new IOException("invalid pipeline state:" + state);
            }
        }
    }

    public void handleStaleNode(DatanodeDetails dn) {
        Set<PipelineID> pipelineIDs = this.nodeManager.getPipelineByDnID(dn.getUuid());
        for (PipelineID id : pipelineIDs) {
            LOG.info("closing pipeline {}.", (Object)id);
            this.eventPublisher.fireEvent(SCMEvents.PIPELINE_CLOSE, (Object)id);
        }
    }

    void processPipelineReport(DatanodeDetails dn, StorageContainerDatanodeProtocolProtos.PipelineReportsProto pipelineReport) {
        HashSet reportedPipelines = new HashSet();
        pipelineReport.getPipelineReportList().forEach(p -> reportedPipelines.add(this.processPipelineReport(p.getPipelineID(), dn)));
    }

    private PipelineID processPipelineReport(HddsProtos.PipelineID id, DatanodeDetails dn) {
        PipelineID pipelineID = PipelineID.getFromProtobuf((HddsProtos.PipelineID)id);
        Pipeline pipeline = this.pipelineMap.get(pipelineID);
        if (pipeline != null) {
            this.pipelineManagerMap.get(pipeline.getType()).processPipelineReport(pipeline, dn);
        }
        return pipelineID;
    }

    public void updatePipelineState(Pipeline pipeline, HddsProtos.LifeCycleEvent event) throws IOException {
        try {
            switch (event) {
                case CREATE: {
                    this.pipelineMap.put(pipeline.getId(), pipeline);
                    this.pipeline2ContainerMap.put(pipeline.getId(), new HashSet());
                    this.nodeManager.addPipeline(pipeline);
                    Lease pipelineLease = this.pipelineLeaseManager.acquire((Object)pipeline);
                    pipelineLease.registerCallBack(() -> {
                        this.updatePipelineState(pipeline, HddsProtos.LifeCycleEvent.TIMEOUT);
                        return null;
                    });
                    break;
                }
                case CREATED: {
                    this.pipelineLeaseManager.release((Object)pipeline);
                    this.addOpenPipeline(pipeline);
                    break;
                }
                case FINALIZE: {
                    this.closeContainersByPipeline(pipeline);
                    break;
                }
                case CLOSE: 
                case TIMEOUT: {
                    this.closePipeline(pipeline);
                    this.pipeline2ContainerMap.remove(pipeline.getId());
                    this.nodeManager.removePipeline(pipeline);
                    this.pipelineMap.remove(pipeline.getId());
                    break;
                }
                default: {
                    throw new SCMException("Unsupported pipeline LifeCycleEvent.", SCMException.ResultCodes.FAILED_TO_CHANGE_PIPELINE_STATE);
                }
            }
            this.stateManager.updatePipelineState(pipeline, event);
            this.pipelineStore.put(pipeline.getId().getProtobuf().toByteArray(), pipeline.getProtobufMessage().toByteArray());
        }
        catch (LeaseException e) {
            throw new IOException("Lease Exception.", e);
        }
    }

    public void shutdown() throws IOException {
        if (this.pipelineLeaseManager != null) {
            this.pipelineLeaseManager.shutdown();
        }
        if (this.pipelineStore != null) {
            this.pipelineStore.close();
        }
    }
}

