/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container.states;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.common.helpers.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.states.ContainerAttribute;
import org.apache.hadoop.hdds.scm.container.states.ContainerQueryKey;
import org.apache.hadoop.hdds.scm.exceptions.SCMException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerStateMap {
    private static final Logger LOG = LoggerFactory.getLogger(ContainerStateMap.class);
    private final ContainerAttribute<HddsProtos.LifeCycleState> lifeCycleStateMap = new ContainerAttribute();
    private final ContainerAttribute<String> ownerMap = new ContainerAttribute();
    private final ContainerAttribute<HddsProtos.ReplicationFactor> factorMap = new ContainerAttribute();
    private final ContainerAttribute<HddsProtos.ReplicationType> typeMap = new ContainerAttribute();
    private final Map<ContainerID, ContainerInfo> containerMap = new HashMap<ContainerID, ContainerInfo>();
    private final Map<ContainerID, Set<DatanodeDetails>> contReplicaMap;
    private static final NavigableSet<ContainerID> EMPTY_SET = Collections.unmodifiableNavigableSet(new TreeSet());
    private final Map<ContainerQueryKey, NavigableSet<ContainerID>> resultCache;
    private final ReadWriteLock lock = new ReentrantReadWriteLock();

    public ContainerStateMap() {
        this.contReplicaMap = new HashMap<ContainerID, Set<DatanodeDetails>>();
        this.resultCache = new ConcurrentHashMap<ContainerQueryKey, NavigableSet<ContainerID>>();
    }

    public void addContainer(ContainerInfo info) throws SCMException {
        Preconditions.checkNotNull((Object)info, (Object)"Container Info cannot be null");
        Preconditions.checkArgument((info.getReplicationFactor().getNumber() > 0 ? 1 : 0) != 0, (Object)"ExpectedReplicaCount should be greater than 0");
        this.lock.writeLock().lock();
        try {
            ContainerID id = ContainerID.valueof((long)info.getContainerID());
            if (this.containerMap.putIfAbsent(id, info) != null) {
                LOG.debug("Duplicate container ID detected. {}", (Object)id);
                throw new SCMException("Duplicate container ID detected.", SCMException.ResultCodes.CONTAINER_EXISTS);
            }
            this.lifeCycleStateMap.insert(info.getState(), id);
            this.ownerMap.insert(info.getOwner(), id);
            this.factorMap.insert(info.getReplicationFactor(), id);
            this.typeMap.insert(info.getReplicationType(), id);
            this.flushCache(info);
            LOG.trace("Created container with {} successfully.", (Object)id);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    public ContainerInfo getContainerInfo(ContainerInfo info) {
        return this.getContainerInfo(info.getContainerID());
    }

    public ContainerInfo getContainerInfo(long containerID) {
        return this.getContainerInfo(ContainerID.valueof((long)containerID));
    }

    public ContainerInfo getContainerInfo(ContainerID containerID) {
        this.lock.readLock().lock();
        try {
            ContainerInfo containerInfo = this.containerMap.get(containerID);
            return containerInfo;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public Set<DatanodeDetails> getContainerReplicas(ContainerID containerID) throws SCMException {
        Preconditions.checkNotNull((Object)containerID);
        this.lock.readLock().lock();
        try {
            if (this.contReplicaMap.containsKey(containerID)) {
                Set<DatanodeDetails> set = Collections.unmodifiableSet(this.contReplicaMap.get(containerID));
                return set;
            }
        }
        finally {
            this.lock.readLock().unlock();
        }
        throw new SCMException("No entry exist for containerId: " + containerID + " in replica map.", SCMException.ResultCodes.NO_REPLICA_FOUND);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addContainerReplica(ContainerID containerID, DatanodeDetails ... dnList) {
        Preconditions.checkNotNull((Object)containerID);
        this.lock.writeLock().lock();
        try {
            for (DatanodeDetails dn : dnList) {
                Preconditions.checkNotNull((Object)dn);
                if (this.contReplicaMap.containsKey(containerID)) {
                    if (this.contReplicaMap.get(containerID).add(dn)) continue;
                    LOG.debug("ReplicaMap already contains entry for container Id: {},DataNode: {}", (Object)containerID, (Object)dn);
                    continue;
                }
                HashSet<DatanodeDetails> dnSet = new HashSet<DatanodeDetails>();
                dnSet.add(dn);
                this.contReplicaMap.put(containerID, dnSet);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean removeContainerReplica(ContainerID containerID, DatanodeDetails dn) throws SCMException {
        Preconditions.checkNotNull((Object)containerID);
        Preconditions.checkNotNull((Object)dn);
        this.lock.writeLock().lock();
        try {
            if (this.contReplicaMap.containsKey(containerID)) {
                boolean bl = this.contReplicaMap.get(containerID).remove(dn);
                return bl;
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
        throw new SCMException("No entry exist for containerId: " + containerID + " in replica map.", SCMException.ResultCodes.FAILED_TO_FIND_CONTAINER);
    }

    @VisibleForTesting
    public static Logger getLOG() {
        return LOG;
    }

    public Map<ContainerID, ContainerInfo> getContainerMap() {
        this.lock.readLock().lock();
        try {
            Map<ContainerID, ContainerInfo> map = Collections.unmodifiableMap(this.containerMap);
            return map;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public void updateContainerInfo(ContainerInfo info) throws SCMException {
        Preconditions.checkNotNull((Object)info);
        ContainerInfo currentInfo = null;
        this.lock.writeLock().lock();
        try {
            currentInfo = this.containerMap.get(ContainerID.valueof((long)info.getContainerID()));
            if (currentInfo == null) {
                throw new SCMException("No such container.", SCMException.ResultCodes.FAILED_TO_FIND_CONTAINER);
            }
            this.flushCache(info, currentInfo);
            this.containerMap.put(info.containerID(), info);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateState(ContainerInfo info, HddsProtos.LifeCycleState currentState, HddsProtos.LifeCycleState newState) throws SCMException {
        Preconditions.checkNotNull((Object)currentState);
        Preconditions.checkNotNull((Object)newState);
        ContainerID id = new ContainerID(info.getContainerID());
        ContainerInfo currentInfo = null;
        this.lock.writeLock().lock();
        try {
            try {
                ContainerInfo newInfo = new ContainerInfo(info);
                newInfo.setState(newState);
                this.flushCache(newInfo, info);
                currentInfo = this.containerMap.get(id);
                if (currentInfo == null) {
                    throw new SCMException("No such container.", SCMException.ResultCodes.FAILED_TO_FIND_CONTAINER);
                }
                info.setState(newState);
                this.containerMap.put(id, info);
                this.lifeCycleStateMap.update(currentState, newState, id);
                LOG.trace("Updated the container {} to new state. Old = {}, new = {}", new Object[]{id, currentState, newState});
            }
            catch (SCMException ex) {
                LOG.error("Unable to update the container state. {}", (Throwable)ex);
                LOG.info("Reverting the update to lifecycle state. Moving back to old state. Old = {}, Attempted state = {}", (Object)currentState, (Object)newState);
                this.containerMap.put(id, currentInfo);
                this.lifeCycleStateMap.update(newState, currentState, id);
                throw new SCMException("Updating the container map failed.", ex, SCMException.ResultCodes.FAILED_TO_CHANGE_CONTAINER_STATE);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    NavigableSet<ContainerID> getContainerIDsByOwner(String ownerName) {
        Preconditions.checkNotNull((Object)ownerName);
        this.lock.readLock().lock();
        try {
            NavigableSet<ContainerID> navigableSet = this.ownerMap.getCollection(ownerName);
            return navigableSet;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    NavigableSet<ContainerID> getContainerIDsByType(HddsProtos.ReplicationType type) {
        Preconditions.checkNotNull((Object)type);
        this.lock.readLock().lock();
        try {
            NavigableSet<ContainerID> navigableSet = this.typeMap.getCollection(type);
            return navigableSet;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    NavigableSet<ContainerID> getContainerIDsByFactor(HddsProtos.ReplicationFactor factor) {
        Preconditions.checkNotNull((Object)factor);
        this.lock.readLock().lock();
        try {
            NavigableSet<ContainerID> navigableSet = this.factorMap.getCollection(factor);
            return navigableSet;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public NavigableSet<ContainerID> getContainerIDsByState(HddsProtos.LifeCycleState state) {
        Preconditions.checkNotNull((Object)state);
        this.lock.readLock().lock();
        try {
            NavigableSet<ContainerID> navigableSet = this.lifeCycleStateMap.getCollection(state);
            return navigableSet;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NavigableSet<ContainerID> getMatchingContainerIDs(HddsProtos.LifeCycleState state, String owner, HddsProtos.ReplicationFactor factor, HddsProtos.ReplicationType type) {
        Preconditions.checkNotNull((Object)state, (Object)"State cannot be null");
        Preconditions.checkNotNull((Object)owner, (Object)"Owner cannot be null");
        Preconditions.checkNotNull((Object)factor, (Object)"Factor cannot be null");
        Preconditions.checkNotNull((Object)type, (Object)"Type cannot be null");
        this.lock.readLock().lock();
        try {
            ContainerQueryKey queryKey = new ContainerQueryKey(state, owner, factor, type);
            if (this.resultCache.containsKey(queryKey)) {
                NavigableSet<ContainerID> navigableSet = this.resultCache.get(queryKey);
                return navigableSet;
            }
            NavigableSet<ContainerID> stateSet = this.lifeCycleStateMap.getCollection(state);
            if (stateSet.size() == 0) {
                NavigableSet<ContainerID> navigableSet = EMPTY_SET;
                return navigableSet;
            }
            NavigableSet<ContainerID> ownerSet = this.ownerMap.getCollection(owner);
            if (ownerSet.size() == 0) {
                NavigableSet<ContainerID> navigableSet = EMPTY_SET;
                return navigableSet;
            }
            NavigableSet<ContainerID> factorSet = this.factorMap.getCollection(factor);
            if (factorSet.size() == 0) {
                NavigableSet<ContainerID> navigableSet = EMPTY_SET;
                return navigableSet;
            }
            NavigableSet<ContainerID> typeSet = this.typeMap.getCollection(type);
            if (typeSet.size() == 0) {
                NavigableSet<ContainerID> navigableSet = EMPTY_SET;
                return navigableSet;
            }
            NavigableSet<ContainerID>[] sets = this.sortBySize(stateSet, ownerSet, factorSet, typeSet);
            NavigableSet<ContainerID> currentSet = sets[0];
            for (int x = 1; x < sets.length; ++x) {
                currentSet = this.intersectSets(currentSet, sets[x]);
            }
            this.resultCache.put(queryKey, currentSet);
            NavigableSet<ContainerID> navigableSet = currentSet;
            return navigableSet;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    private NavigableSet<ContainerID> intersectSets(NavigableSet<ContainerID> smaller, NavigableSet<ContainerID> bigger) {
        Preconditions.checkState((smaller.size() <= bigger.size() ? 1 : 0) != 0, (Object)"This function assumes the first set is lesser or equal to second set");
        TreeSet<ContainerID> resultSet = new TreeSet<ContainerID>();
        for (ContainerID id : smaller) {
            if (!bigger.contains(id)) continue;
            resultSet.add(id);
        }
        return resultSet;
    }

    private NavigableSet<ContainerID>[] sortBySize(NavigableSet<ContainerID> ... sets) {
        for (int x = 0; x < sets.length - 1; ++x) {
            for (int y = 0; y < sets.length - x - 1; ++y) {
                if (sets[y].size() <= sets[y + 1].size()) continue;
                NavigableSet<ContainerID> temp = sets[y];
                sets[y] = sets[y + 1];
                sets[y + 1] = temp;
            }
        }
        return sets;
    }

    private void flushCache(ContainerInfo ... containerInfos) {
        for (ContainerInfo containerInfo : containerInfos) {
            ContainerQueryKey key = new ContainerQueryKey(containerInfo.getState(), containerInfo.getOwner(), containerInfo.getReplicationFactor(), containerInfo.getReplicationType());
            this.resultCache.remove(key);
        }
    }
}

