/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.primitives.Longs;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.StorageUnit;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.block.PendingDeleteStatusList;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerStateManager;
import org.apache.hadoop.hdds.scm.container.common.helpers.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.common.helpers.ContainerWithPipeline;
import org.apache.hadoop.hdds.scm.container.common.helpers.Pipeline;
import org.apache.hadoop.hdds.scm.container.common.helpers.PipelineID;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.exceptions.SCMException;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.pipelines.PipelineSelector;
import org.apache.hadoop.hdds.server.ServerUtils;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.ozone.lease.Lease;
import org.apache.hadoop.ozone.lease.LeaseException;
import org.apache.hadoop.ozone.lease.LeaseManager;
import org.apache.hadoop.utils.BatchOperation;
import org.apache.hadoop.utils.MetadataStore;
import org.apache.hadoop.utils.MetadataStoreBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCMContainerManager
implements ContainerManager {
    private static final Logger LOG = LoggerFactory.getLogger(SCMContainerManager.class);
    private final NodeManager nodeManager;
    private final long cacheSize;
    private final Lock lock;
    private final Charset encoding = Charset.forName("UTF-8");
    private final MetadataStore containerStore;
    private final PipelineSelector pipelineSelector;
    private final ContainerStateManager containerStateManager;
    private final LeaseManager<ContainerInfo> containerLeaseManager;
    private final EventPublisher eventPublisher;
    private final long size;

    public SCMContainerManager(Configuration conf, NodeManager nodeManager, int cacheSizeMB, EventPublisher eventPublisher) throws IOException {
        this.nodeManager = nodeManager;
        this.cacheSize = cacheSizeMB;
        File metaDir = ServerUtils.getOzoneMetaDirPath((Configuration)conf);
        File containerDBPath = new File(metaDir, "scm-container.db");
        this.containerStore = MetadataStoreBuilder.newBuilder().setConf(conf).setDbFile(containerDBPath).setCacheSize(this.cacheSize * 0x100000L).build();
        this.lock = new ReentrantLock();
        this.size = (long)conf.getStorageSize("ozone.scm.container.size", "5GB", StorageUnit.BYTES);
        this.pipelineSelector = new PipelineSelector(nodeManager, conf, eventPublisher, cacheSizeMB);
        this.containerStateManager = new ContainerStateManager(conf, this, this.pipelineSelector);
        LOG.trace("Container State Manager created.");
        this.eventPublisher = eventPublisher;
        long containerCreationLeaseTimeout = conf.getTimeDuration("ozone.scm.container.creation.lease.timeout", "60s", TimeUnit.MILLISECONDS);
        this.containerLeaseManager = new LeaseManager("ContainerCreation", containerCreationLeaseTimeout);
        this.containerLeaseManager.start();
        this.loadExistingContainers();
    }

    private void loadExistingContainers() {
        List<ContainerInfo> containerList;
        try {
            containerList = this.listContainer(0L, Integer.MAX_VALUE);
            if (containerList == null || containerList.size() == 0) {
                LOG.info("No containers to load for this cluster.");
                return;
            }
        }
        catch (IOException e) {
            if (!e.getMessage().equals("No container exists in current db")) {
                LOG.error("Could not list the containers", (Throwable)e);
            }
            return;
        }
        try {
            for (ContainerInfo container : containerList) {
                this.containerStateManager.addExistingContainer(container);
                if (!container.isContainerOpen()) continue;
                this.pipelineSelector.addContainerToPipeline(container.getPipelineID(), container.getContainerID());
            }
        }
        catch (SCMException ex) {
            LOG.error("Unable to create a container information. ", (Throwable)ex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ContainerInfo getContainer(long containerID) throws IOException {
        this.lock.lock();
        try {
            ContainerInfo containerInfo;
            byte[] containerBytes = this.containerStore.get(Longs.toByteArray((long)containerID));
            if (containerBytes == null) {
                throw new SCMException("Specified key does not exist. key : " + containerID, SCMException.ResultCodes.FAILED_TO_FIND_CONTAINER);
            }
            HddsProtos.SCMContainerInfo temp = (HddsProtos.SCMContainerInfo)HddsProtos.SCMContainerInfo.PARSER.parseFrom(containerBytes);
            ContainerInfo containerInfo2 = containerInfo = ContainerInfo.fromProtobuf((HddsProtos.SCMContainerInfo)temp);
            return containerInfo2;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ContainerWithPipeline getContainerWithPipeline(long containerID) throws IOException {
        this.lock.lock();
        try {
            Pipeline pipeline;
            byte[] containerBytes = this.containerStore.get(Longs.toByteArray((long)containerID));
            if (containerBytes == null) {
                throw new SCMException("Specified key does not exist. key : " + containerID, SCMException.ResultCodes.FAILED_TO_FIND_CONTAINER);
            }
            HddsProtos.SCMContainerInfo temp = (HddsProtos.SCMContainerInfo)HddsProtos.SCMContainerInfo.PARSER.parseFrom(containerBytes);
            ContainerInfo contInfo = ContainerInfo.fromProtobuf((HddsProtos.SCMContainerInfo)temp);
            String leaderId = "";
            if (contInfo.isContainerOpen()) {
                pipeline = this.pipelineSelector.getPipeline(contInfo.getPipelineID());
            } else {
                Set<DatanodeDetails> dnWithReplicas = this.containerStateManager.getContainerReplicas(contInfo.containerID());
                if (!dnWithReplicas.isEmpty()) {
                    leaderId = dnWithReplicas.iterator().next().getUuidString();
                }
                pipeline = new Pipeline(leaderId, contInfo.getState(), HddsProtos.ReplicationType.STAND_ALONE, contInfo.getReplicationFactor(), PipelineID.randomId());
                dnWithReplicas.forEach(arg_0 -> ((Pipeline)pipeline).addMember(arg_0));
            }
            ContainerWithPipeline containerWithPipeline = new ContainerWithPipeline(contInfo, pipeline);
            return containerWithPipeline;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<ContainerInfo> listContainer(long startContainerID, int count) throws IOException {
        ArrayList<ContainerInfo> containerList = new ArrayList<ContainerInfo>();
        this.lock.lock();
        try {
            if (this.containerStore.isEmpty()) {
                throw new IOException("No container exists in current db");
            }
            byte[] startKey = startContainerID <= 0L ? null : Longs.toByteArray((long)startContainerID);
            List range = this.containerStore.getSequentialRangeKVs(startKey, count, null);
            for (Map.Entry entry : range) {
                ContainerInfo containerInfo = ContainerInfo.fromProtobuf((HddsProtos.SCMContainerInfo)((HddsProtos.SCMContainerInfo)HddsProtos.SCMContainerInfo.PARSER.parseFrom((byte[])entry.getValue())));
                Preconditions.checkNotNull((Object)containerInfo);
                containerList.add(containerInfo);
            }
        }
        finally {
            this.lock.unlock();
        }
        return containerList;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ContainerWithPipeline allocateContainer(HddsProtos.ReplicationType type, HddsProtos.ReplicationFactor replicationFactor, String owner) throws IOException {
        ContainerWithPipeline containerWithPipeline;
        this.lock.lock();
        try {
            containerWithPipeline = this.containerStateManager.allocateContainer(this.pipelineSelector, type, replicationFactor, owner);
            ContainerInfo containerInfo = containerWithPipeline.getContainerInfo();
            byte[] containerIDBytes = Longs.toByteArray((long)containerInfo.getContainerID());
            this.containerStore.put(containerIDBytes, containerInfo.getProtobuf().toByteArray());
        }
        finally {
            this.lock.unlock();
        }
        return containerWithPipeline;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void deleteContainer(long containerID) throws IOException {
        this.lock.lock();
        try {
            byte[] dbKey = Longs.toByteArray((long)containerID);
            byte[] containerBytes = this.containerStore.get(dbKey);
            if (containerBytes == null) {
                throw new SCMException("Failed to delete container " + containerID + ", reason : container doesn't exist.", SCMException.ResultCodes.FAILED_TO_FIND_CONTAINER);
            }
            this.containerStore.delete(dbKey);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public HddsProtos.LifeCycleState updateContainerState(long containerID, HddsProtos.LifeCycleEvent event) throws IOException {
        this.lock.lock();
        try {
            byte[] dbKey = Longs.toByteArray((long)containerID);
            byte[] containerBytes = this.containerStore.get(dbKey);
            if (containerBytes == null) {
                throw new SCMException("Failed to update container state" + containerID + ", reason : container doesn't exist.", SCMException.ResultCodes.FAILED_TO_FIND_CONTAINER);
            }
            ContainerInfo containerInfo = ContainerInfo.fromProtobuf((HddsProtos.SCMContainerInfo)((HddsProtos.SCMContainerInfo)HddsProtos.SCMContainerInfo.PARSER.parseFrom(containerBytes)));
            Preconditions.checkNotNull((Object)containerInfo);
            switch (event) {
                case CREATE: {
                    Lease containerLease = this.containerLeaseManager.acquire((Object)containerInfo);
                    containerLease.registerCallBack(() -> {
                        this.updateContainerState(containerID, HddsProtos.LifeCycleEvent.TIMEOUT);
                        return null;
                    });
                    break;
                }
                case CREATED: {
                    this.containerLeaseManager.release((Object)containerInfo);
                    break;
                }
                case FINALIZE: {
                    break;
                }
                case CLOSE: {
                    break;
                }
                case UPDATE: {
                    break;
                }
                case DELETE: {
                    break;
                }
                case TIMEOUT: {
                    break;
                }
                case CLEANUP: {
                    break;
                }
                default: {
                    throw new SCMException("Unsupported container LifeCycleEvent.", SCMException.ResultCodes.FAILED_TO_CHANGE_CONTAINER_STATE);
                }
            }
            ContainerInfo updatedContainer = this.containerStateManager.updateContainerState(containerInfo, event);
            if (!updatedContainer.isContainerOpen()) {
                this.pipelineSelector.removeContainerFromPipeline(containerInfo.getPipelineID(), containerID);
            }
            this.containerStore.put(dbKey, updatedContainer.getProtobuf().toByteArray());
            HddsProtos.LifeCycleState lifeCycleState = updatedContainer.getState();
            return lifeCycleState;
        }
        catch (LeaseException e) {
            throw new IOException("Lease Exception.", e);
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void updateDeleteTransactionId(Map<Long, Long> deleteTransactionMap) throws IOException {
        if (deleteTransactionMap == null) {
            return;
        }
        this.lock.lock();
        try {
            BatchOperation batch = new BatchOperation();
            for (Map.Entry<Long, Long> entry : deleteTransactionMap.entrySet()) {
                long containerID = entry.getKey();
                byte[] dbKey = Longs.toByteArray((long)containerID);
                byte[] containerBytes = this.containerStore.get(dbKey);
                if (containerBytes == null) {
                    throw new SCMException("Failed to increment number of deleted blocks for container " + containerID + ", reason : container doesn't exist.", SCMException.ResultCodes.FAILED_TO_FIND_CONTAINER);
                }
                ContainerInfo containerInfo = ContainerInfo.fromProtobuf((HddsProtos.SCMContainerInfo)HddsProtos.SCMContainerInfo.parseFrom((byte[])containerBytes));
                containerInfo.updateDeleteTransactionId(entry.getValue().longValue());
                batch.put(dbKey, containerInfo.getProtobuf().toByteArray());
            }
            this.containerStore.writeBatch(batch);
            this.containerStateManager.updateDeleteTransactionId(deleteTransactionMap);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public ContainerStateManager getStateManager() {
        return this.containerStateManager;
    }

    @Override
    public ContainerWithPipeline getMatchingContainerWithPipeline(long sizeRequired, String owner, HddsProtos.ReplicationType type, HddsProtos.ReplicationFactor factor, HddsProtos.LifeCycleState state) throws IOException {
        ContainerInfo containerInfo = this.getStateManager().getMatchingContainer(sizeRequired, owner, type, factor, state);
        if (containerInfo == null) {
            return null;
        }
        Pipeline pipeline = this.pipelineSelector.getPipeline(containerInfo.getPipelineID());
        return new ContainerWithPipeline(containerInfo, pipeline);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void processContainerReports(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.ContainerReportsProto reports, boolean isRegisterCall) throws IOException {
        List containerInfos = reports.getReportsList();
        PendingDeleteStatusList pendingDeleteStatusList = new PendingDeleteStatusList(datanodeDetails);
        BatchOperation batch = new BatchOperation();
        for (StorageContainerDatanodeProtocolProtos.ContainerInfo contInfo : containerInfos) {
            if (isRegisterCall) {
                try {
                    this.getStateManager().addContainerReplica(ContainerID.valueof((long)contInfo.getContainerID()), datanodeDetails);
                }
                catch (Exception ex) {
                    LOG.error("Error while adding replica for containerId {}.", (Object)contInfo.getContainerID(), (Object)ex);
                }
            }
            byte[] dbKey = Longs.toByteArray((long)contInfo.getContainerID());
            this.lock.lock();
            try {
                byte[] containerBytes = this.containerStore.get(dbKey);
                if (containerBytes != null) {
                    HddsProtos.SCMContainerInfo knownState = (HddsProtos.SCMContainerInfo)HddsProtos.SCMContainerInfo.PARSER.parseFrom(containerBytes);
                    if (knownState.getState() == HddsProtos.LifeCycleState.CLOSING && contInfo.getState() == HddsProtos.LifeCycleState.CLOSED) {
                        this.updateContainerState(contInfo.getContainerID(), HddsProtos.LifeCycleEvent.CLOSE);
                        knownState = (HddsProtos.SCMContainerInfo)HddsProtos.SCMContainerInfo.PARSER.parseFrom(this.containerStore.get(dbKey));
                    }
                    HddsProtos.SCMContainerInfo newState = this.reconcileState(contInfo, knownState, datanodeDetails);
                    if (knownState.getDeleteTransactionId() > contInfo.getDeleteTransactionId()) {
                        pendingDeleteStatusList.addPendingDeleteStatus(contInfo.getDeleteTransactionId(), knownState.getDeleteTransactionId(), knownState.getContainerID());
                    }
                    batch.put(dbKey, newState.toByteArray());
                    continue;
                }
                LOG.error("Error while processing container report from datanode : {}, for container: {}, reason: container doesn't exist incontainer database.", (Object)datanodeDetails, (Object)contInfo.getContainerID());
            }
            finally {
                this.lock.unlock();
            }
        }
        this.containerStore.writeBatch(batch);
        if (pendingDeleteStatusList.getNumPendingDeletes() > 0) {
            this.eventPublisher.fireEvent(SCMEvents.PENDING_DELETE_STATUS, (Object)pendingDeleteStatusList);
        }
    }

    private HddsProtos.SCMContainerInfo reconcileState(StorageContainerDatanodeProtocolProtos.ContainerInfo datanodeState, HddsProtos.SCMContainerInfo knownState, DatanodeDetails dnDetails) {
        HddsProtos.SCMContainerInfo.Builder builder = HddsProtos.SCMContainerInfo.newBuilder();
        builder.setContainerID(knownState.getContainerID()).setPipelineID(knownState.getPipelineID()).setReplicationType(knownState.getReplicationType()).setReplicationFactor(knownState.getReplicationFactor());
        long usedSize = datanodeState.getUsed();
        long allocated = knownState.getAllocatedBytes() > usedSize ? knownState.getAllocatedBytes() : usedSize;
        builder.setAllocatedBytes(allocated).setUsedBytes(usedSize).setNumberOfKeys(datanodeState.getKeyCount()).setState(knownState.getState()).setStateEnterTime(knownState.getStateEnterTime()).setContainerID(knownState.getContainerID()).setDeleteTransactionId(knownState.getDeleteTransactionId());
        if (knownState.getOwner() != null) {
            builder.setOwner(knownState.getOwner());
        }
        return builder.build();
    }

    private boolean shouldClose(ContainerInfo info) {
        return info.getState() == HddsProtos.LifeCycleState.OPEN;
    }

    private boolean isClosed(ContainerInfo info) {
        return info.getState() == HddsProtos.LifeCycleState.CLOSED;
    }

    @Override
    public void close() throws IOException {
        if (this.containerLeaseManager != null) {
            this.containerLeaseManager.shutdown();
        }
        if (this.containerStateManager != null) {
            this.flushContainerInfo();
            this.containerStateManager.close();
        }
        if (this.containerStore != null) {
            this.containerStore.close();
        }
        if (this.pipelineSelector != null) {
            this.pipelineSelector.shutdown();
        }
    }

    @VisibleForTesting
    public void flushContainerInfo() throws IOException {
        List<ContainerInfo> containers = this.containerStateManager.getAllContainers();
        ArrayList<Long> failedContainers = new ArrayList<Long>();
        for (ContainerInfo info : containers) {
            try {
                byte[] dbKey = Longs.toByteArray((long)info.getContainerID());
                byte[] containerBytes = this.containerStore.get(dbKey);
                if (containerBytes != null) {
                    this.containerStore.put(dbKey, info.getProtobuf().toByteArray());
                    continue;
                }
                LOG.debug("Container state manager has container {} but not found in container store, a deleted container?", (Object)info.getContainerID());
            }
            catch (IOException ioe) {
                failedContainers.add(info.getContainerID());
            }
        }
        if (!failedContainers.isEmpty()) {
            throw new IOException("Error in flushing container info from container state manager: " + failedContainers);
        }
    }

    @VisibleForTesting
    public MetadataStore getContainerStore() {
        return this.containerStore;
    }

    @Override
    public PipelineSelector getPipelineSelector() {
        return this.pipelineSelector;
    }
}

