/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.client.io;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.Optional;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerLocationProtocolProtos;
import org.apache.hadoop.hdds.scm.XceiverClientManager;
import org.apache.hadoop.hdds.scm.XceiverClientSpi;
import org.apache.hadoop.hdds.scm.container.common.helpers.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.common.helpers.ContainerWithPipeline;
import org.apache.hadoop.hdds.scm.container.common.helpers.Pipeline;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.scm.protocolPB.StorageContainerLocationProtocolClientSideTranslatorPB;
import org.apache.hadoop.hdds.scm.storage.ChunkOutputStream;
import org.apache.hadoop.hdds.scm.storage.ContainerProtocolCalls;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.ozone.om.helpers.OmKeyArgs;
import org.apache.hadoop.ozone.om.helpers.OmKeyInfo;
import org.apache.hadoop.ozone.om.helpers.OmKeyLocationInfo;
import org.apache.hadoop.ozone.om.helpers.OmKeyLocationInfoGroup;
import org.apache.hadoop.ozone.om.helpers.OpenKeySession;
import org.apache.hadoop.ozone.om.protocolPB.OzoneManagerProtocolClientSideTranslatorPB;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChunkGroupOutputStream
extends OutputStream {
    public static final Logger LOG = LoggerFactory.getLogger(ChunkGroupOutputStream.class);
    private final ArrayList<ChunkOutputStreamEntry> streamEntries = new ArrayList();
    private int currentStreamIndex;
    private long byteOffset;
    private final OzoneManagerProtocolClientSideTranslatorPB omClient;
    private final StorageContainerLocationProtocolClientSideTranslatorPB scmClient;
    private final OmKeyArgs keyArgs;
    private final long openID;
    private final XceiverClientManager xceiverClientManager;
    private final int chunkSize;
    private final String requestID;
    private boolean closed;
    private final RetryPolicy retryPolicy;

    @VisibleForTesting
    public ChunkGroupOutputStream() {
        this.omClient = null;
        this.scmClient = null;
        this.keyArgs = null;
        this.openID = -1L;
        this.xceiverClientManager = null;
        this.chunkSize = 0;
        this.requestID = null;
        this.closed = false;
        this.retryPolicy = null;
    }

    @VisibleForTesting
    public void addStream(OutputStream outputStream, long length) {
        this.streamEntries.add(new ChunkOutputStreamEntry(outputStream, length));
    }

    @VisibleForTesting
    public List<ChunkOutputStreamEntry> getStreamEntries() {
        return this.streamEntries;
    }

    @VisibleForTesting
    public XceiverClientManager getXceiverClientManager() {
        return this.xceiverClientManager;
    }

    public List<OmKeyLocationInfo> getLocationInfoList() throws IOException {
        ArrayList<OmKeyLocationInfo> locationInfoList = new ArrayList<OmKeyLocationInfo>();
        for (ChunkOutputStreamEntry streamEntry : this.streamEntries) {
            OmKeyLocationInfo info = new OmKeyLocationInfo.Builder().setBlockID(streamEntry.blockID).setShouldCreateContainer(false).setLength(streamEntry.currentPosition).setOffset(0L).setBlockCommitSequenceId(streamEntry.getBlockCommitSequenceId()).build();
            locationInfoList.add(info);
        }
        return locationInfoList;
    }

    public ChunkGroupOutputStream(OpenKeySession handler, XceiverClientManager xceiverClientManager, StorageContainerLocationProtocolClientSideTranslatorPB scmClient, OzoneManagerProtocolClientSideTranslatorPB omClient, int chunkSize, String requestId, HddsProtos.ReplicationFactor factor, HddsProtos.ReplicationType type, RetryPolicy retryPolicy) throws IOException {
        this.currentStreamIndex = 0;
        this.byteOffset = 0L;
        this.omClient = omClient;
        this.scmClient = scmClient;
        OmKeyInfo info = handler.getKeyInfo();
        this.keyArgs = new OmKeyArgs.Builder().setVolumeName(info.getVolumeName()).setBucketName(info.getBucketName()).setKeyName(info.getKeyName()).setType(type).setFactor(factor).setDataSize(info.getDataSize()).build();
        this.openID = handler.getId();
        this.xceiverClientManager = xceiverClientManager;
        this.chunkSize = chunkSize;
        this.requestID = requestId;
        this.retryPolicy = retryPolicy;
    }

    public void addPreallocateBlocks(OmKeyLocationInfoGroup version, long openVersion) throws IOException {
        for (OmKeyLocationInfo subKeyInfo : version.getLocationList()) {
            if (subKeyInfo.getCreateVersion() != openVersion) continue;
            this.checkKeyLocationInfo(subKeyInfo);
        }
    }

    private void checkKeyLocationInfo(OmKeyLocationInfo subKeyInfo) throws IOException {
        ContainerWithPipeline containerWithPipeline = this.scmClient.getContainerWithPipeline(subKeyInfo.getContainerID());
        ContainerInfo container = containerWithPipeline.getContainerInfo();
        Pipeline pipeline = containerWithPipeline.getPipeline();
        if (pipeline.getMachines().isEmpty()) {
            throw new IOException("No datanodes found in the pipeline " + pipeline.getId());
        }
        XceiverClientSpi xceiverClient = this.xceiverClientManager.acquireClient(pipeline);
        if (subKeyInfo.getShouldCreateContainer()) {
            try {
                ContainerProtocolCalls.createContainer((XceiverClientSpi)xceiverClient, (long)container.getContainerID(), (String)this.requestID);
                this.scmClient.notifyObjectStageChange(StorageContainerLocationProtocolProtos.ObjectStageChangeRequestProto.Type.container, subKeyInfo.getContainerID(), StorageContainerLocationProtocolProtos.ObjectStageChangeRequestProto.Op.create, StorageContainerLocationProtocolProtos.ObjectStageChangeRequestProto.Stage.complete);
            }
            catch (StorageContainerException ex) {
                if (ex.getResult().equals((Object)ContainerProtos.Result.CONTAINER_EXISTS)) {
                    LOG.debug("Container {} already exists.", (Object)container.getContainerID());
                }
                LOG.error("Container creation failed for {}.", (Object)container.getContainerID(), (Object)ex);
                throw ex;
            }
        }
        this.streamEntries.add(new ChunkOutputStreamEntry(subKeyInfo.getBlockID(), this.keyArgs.getKeyName(), this.xceiverClientManager, xceiverClient, this.requestID, this.chunkSize, subKeyInfo.getLength()));
    }

    @VisibleForTesting
    public long getByteOffset() {
        return this.byteOffset;
    }

    @Override
    public void write(int b) throws IOException {
        byte[] buf = new byte[]{(byte)b};
        this.write(buf, 0, 1);
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        this.checkNotClosed();
        this.handleWrite(b, off, len);
    }

    private void handleWrite(byte[] b, int off, int len) throws IOException {
        if (b == null) {
            throw new NullPointerException();
        }
        if (off < 0 || off > b.length || len < 0 || off + len > b.length || off + len < 0) {
            throw new IndexOutOfBoundsException();
        }
        if (len == 0) {
            return;
        }
        int succeededAllocates = 0;
        while (len > 0) {
            if (this.streamEntries.size() <= this.currentStreamIndex) {
                Preconditions.checkNotNull((Object)this.omClient);
                try {
                    this.allocateNewBlock(this.currentStreamIndex);
                    ++succeededAllocates;
                }
                catch (IOException ioe) {
                    LOG.error("Try to allocate more blocks for write failed, already allocated " + succeededAllocates + " blocks for this write.");
                    throw ioe;
                }
            }
            Preconditions.checkArgument((this.currentStreamIndex < this.streamEntries.size() ? 1 : 0) != 0);
            ChunkOutputStreamEntry current = this.streamEntries.get(this.currentStreamIndex);
            int writeLen = Math.min(len, (int)current.getRemaining());
            try {
                current.write(b, off, writeLen);
            }
            catch (IOException ioe) {
                if (this.checkIfContainerIsClosed(ioe)) {
                    this.handleCloseContainerException(current, this.currentStreamIndex);
                    continue;
                }
                throw ioe;
            }
            if (current.getRemaining() <= 0L) {
                this.handleFlushOrClose(true);
                ++this.currentStreamIndex;
            }
            len -= writeLen;
            off += writeLen;
            this.byteOffset += (long)writeLen;
        }
    }

    private long getCommittedBlockLength(ChunkOutputStreamEntry streamEntry) throws IOException {
        int numRetries = 0;
        while (true) {
            try {
                ContainerProtos.GetCommittedBlockLengthResponseProto responseProto = ContainerProtocolCalls.getCommittedBlockLength((XceiverClientSpi)streamEntry.xceiverClient, (BlockID)streamEntry.blockID, (String)this.requestID);
                long blockLength = responseProto.getBlockLength();
                return blockLength;
            }
            catch (StorageContainerException sce) {
                RetryPolicy.RetryAction action;
                try {
                    action = this.retryPolicy.shouldRetry((Exception)((Object)sce), numRetries, 0, true);
                }
                catch (Exception e) {
                    throw e instanceof IOException ? (IOException)e : new IOException(e);
                }
                if (action.action == RetryPolicy.RetryAction.RetryDecision.FAIL) {
                    if (action.reason != null) {
                        LOG.error("GetCommittedBlockLength request failed. " + action.reason, (Throwable)sce);
                    }
                    throw sce;
                }
                if (Thread.currentThread().isInterrupted()) {
                    LOG.warn("Interrupted while trying for connection");
                    throw sce;
                }
                Preconditions.checkArgument((action.action == RetryPolicy.RetryAction.RetryDecision.RETRY ? 1 : 0) != 0);
                try {
                    Thread.sleep(action.delayMillis);
                }
                catch (InterruptedException e) {
                    throw (IOException)new InterruptedIOException("Interrupted: action=" + action + ", retry policy=" + this.retryPolicy).initCause(e);
                }
                LOG.trace("Retrying GetCommittedBlockLength request. Already tried " + ++numRetries + " time(s); retry policy is " + this.retryPolicy);
                continue;
            }
            break;
        }
    }

    private void discardPreallocatedBlocks(long containerID) {
        if (this.currentStreamIndex < this.streamEntries.size()) {
            ListIterator<ChunkOutputStreamEntry> streamEntryIterator = this.streamEntries.listIterator(this.currentStreamIndex);
            while (streamEntryIterator.hasNext()) {
                if (streamEntryIterator.next().blockID.getContainerID() != containerID) continue;
                streamEntryIterator.remove();
            }
        }
    }

    private void removeEmptyBlocks() {
        if (this.currentStreamIndex < this.streamEntries.size()) {
            ListIterator<ChunkOutputStreamEntry> streamEntryIterator = this.streamEntries.listIterator(this.currentStreamIndex);
            while (streamEntryIterator.hasNext()) {
                if (streamEntryIterator.next().currentPosition != 0L) continue;
                streamEntryIterator.remove();
            }
        }
    }

    private void handleCloseContainerException(ChunkOutputStreamEntry streamEntry, int streamIndex) throws IOException {
        long committedLength = 0L;
        ByteBuffer buffer = streamEntry.getBuffer();
        if (buffer == null) {
            return;
        }
        ++this.currentStreamIndex;
        if (streamEntry.currentPosition >= (long)this.chunkSize || streamEntry.currentPosition != (long)buffer.position()) {
            committedLength = this.getCommittedBlockLength(streamEntry);
            streamEntry.currentPosition = committedLength;
        }
        if (buffer.position() > 0) {
            Preconditions.checkState((buffer.position() < this.chunkSize ? 1 : 0) != 0);
            this.byteOffset -= (long)buffer.position();
            this.handleWrite(buffer.array(), 0, buffer.position());
        }
        streamEntry.cleanup();
        if (committedLength == 0L) {
            this.streamEntries.remove(streamIndex);
            Preconditions.checkArgument((this.currentStreamIndex != 0 ? 1 : 0) != 0);
            --this.currentStreamIndex;
        }
        this.discardPreallocatedBlocks(streamEntry.blockID.getContainerID());
    }

    private boolean checkIfContainerIsClosed(IOException ioe) {
        return Optional.of(ioe.getCause()).filter(e -> e instanceof StorageContainerException).map(e -> (StorageContainerException)((Object)e)).filter(sce -> sce.getResult() == ContainerProtos.Result.CLOSED_CONTAINER_IO).isPresent();
    }

    private long getKeyLength() {
        return this.streamEntries.parallelStream().mapToLong(e -> ((ChunkOutputStreamEntry)e).currentPosition).sum();
    }

    private void allocateNewBlock(int index) throws IOException {
        OmKeyLocationInfo subKeyInfo = this.omClient.allocateBlock(this.keyArgs, this.openID);
        this.checkKeyLocationInfo(subKeyInfo);
    }

    @Override
    public void flush() throws IOException {
        this.checkNotClosed();
        this.handleFlushOrClose(false);
    }

    private void handleFlushOrClose(boolean close) throws IOException {
        if (this.streamEntries.size() == 0) {
            return;
        }
        int size = this.streamEntries.size();
        int streamIndex = this.currentStreamIndex >= size ? size - 1 : this.currentStreamIndex;
        ChunkOutputStreamEntry entry = this.streamEntries.get(streamIndex);
        if (entry != null) {
            try {
                if (close) {
                    entry.close();
                } else {
                    entry.flush();
                }
            }
            catch (IOException ioe) {
                if (this.checkIfContainerIsClosed(ioe)) {
                    this.handleCloseContainerException(entry, streamIndex);
                    this.handleFlushOrClose(close);
                }
                throw ioe;
            }
        }
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        this.handleFlushOrClose(true);
        if (this.keyArgs != null) {
            this.removeEmptyBlocks();
            Preconditions.checkState((this.byteOffset == this.getKeyLength() ? 1 : 0) != 0);
            this.keyArgs.setDataSize(this.byteOffset);
            this.keyArgs.setLocationInfoList(this.getLocationInfoList());
            this.omClient.commitKey(this.keyArgs, this.openID);
        } else {
            LOG.warn("Closing ChunkGroupOutputStream, but key args is null");
        }
    }

    private void checkNotClosed() throws IOException {
        if (this.closed) {
            throw new IOException(": Stream is closed! Key: " + this.keyArgs.getKeyName());
        }
    }

    private static class ChunkOutputStreamEntry
    extends OutputStream {
        private OutputStream outputStream;
        private final BlockID blockID;
        private final String key;
        private final XceiverClientManager xceiverClientManager;
        private final XceiverClientSpi xceiverClient;
        private final String requestId;
        private final int chunkSize;
        private final long length;
        private long currentPosition;

        ChunkOutputStreamEntry(BlockID blockID, String key, XceiverClientManager xceiverClientManager, XceiverClientSpi xceiverClient, String requestId, int chunkSize, long length) {
            this.outputStream = null;
            this.blockID = blockID;
            this.key = key;
            this.xceiverClientManager = xceiverClientManager;
            this.xceiverClient = xceiverClient;
            this.requestId = requestId;
            this.chunkSize = chunkSize;
            this.length = length;
            this.currentPosition = 0L;
        }

        ChunkOutputStreamEntry(OutputStream outputStream, long length) {
            this.outputStream = outputStream;
            this.blockID = null;
            this.key = null;
            this.xceiverClientManager = null;
            this.xceiverClient = null;
            this.requestId = null;
            this.chunkSize = -1;
            this.length = length;
            this.currentPosition = 0L;
        }

        long getLength() {
            return this.length;
        }

        long getRemaining() {
            return this.length - this.currentPosition;
        }

        private void checkStream() {
            if (this.outputStream == null) {
                this.outputStream = new ChunkOutputStream(this.blockID, this.key, this.xceiverClientManager, this.xceiverClient, this.requestId, this.chunkSize);
            }
        }

        @Override
        public void write(int b) throws IOException {
            this.checkStream();
            this.outputStream.write(b);
            ++this.currentPosition;
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            this.checkStream();
            this.outputStream.write(b, off, len);
            this.currentPosition += (long)len;
        }

        @Override
        public void flush() throws IOException {
            if (this.outputStream != null) {
                this.outputStream.flush();
            }
        }

        @Override
        public void close() throws IOException {
            if (this.outputStream != null) {
                this.outputStream.close();
            }
        }

        ByteBuffer getBuffer() throws IOException {
            if (this.outputStream instanceof ChunkOutputStream) {
                ChunkOutputStream out = (ChunkOutputStream)this.outputStream;
                return out.getBuffer();
            }
            throw new IOException("Invalid Output Stream for Key: " + this.key);
        }

        long getBlockCommitSequenceId() throws IOException {
            if (this.outputStream instanceof ChunkOutputStream) {
                ChunkOutputStream out = (ChunkOutputStream)this.outputStream;
                return out.getBlockCommitSequenceId();
            }
            if (this.outputStream == null) {
                return 0L;
            }
            throw new IOException("Invalid Output Stream for Key: " + this.key);
        }

        public void cleanup() {
            this.checkStream();
            if (this.outputStream instanceof ChunkOutputStream) {
                ChunkOutputStream out = (ChunkOutputStream)this.outputStream;
                out.cleanup();
            }
        }
    }

    public static class Builder {
        private OpenKeySession openHandler;
        private XceiverClientManager xceiverManager;
        private StorageContainerLocationProtocolClientSideTranslatorPB scmClient;
        private OzoneManagerProtocolClientSideTranslatorPB omClient;
        private int chunkSize;
        private String requestID;
        private HddsProtos.ReplicationType type;
        private HddsProtos.ReplicationFactor factor;
        private RetryPolicy retryPolicy;

        public Builder setHandler(OpenKeySession handler) {
            this.openHandler = handler;
            return this;
        }

        public Builder setXceiverClientManager(XceiverClientManager manager) {
            this.xceiverManager = manager;
            return this;
        }

        public Builder setScmClient(StorageContainerLocationProtocolClientSideTranslatorPB client) {
            this.scmClient = client;
            return this;
        }

        public Builder setOmClient(OzoneManagerProtocolClientSideTranslatorPB client) {
            this.omClient = client;
            return this;
        }

        public Builder setChunkSize(int size) {
            this.chunkSize = size;
            return this;
        }

        public Builder setRequestID(String id) {
            this.requestID = id;
            return this;
        }

        public Builder setType(HddsProtos.ReplicationType replicationType) {
            this.type = replicationType;
            return this;
        }

        public Builder setFactor(HddsProtos.ReplicationFactor replicationFactor) {
            this.factor = replicationFactor;
            return this;
        }

        public ChunkGroupOutputStream build() throws IOException {
            return new ChunkGroupOutputStream(this.openHandler, this.xceiverManager, this.scmClient, this.omClient, this.chunkSize, this.requestID, this.factor, this.type, this.retryPolicy);
        }

        public Builder setRetryPolicy(RetryPolicy rPolicy) {
            this.retryPolicy = rPolicy;
            return this;
        }
    }
}

