/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.AllocatedBlock;
import org.apache.hadoop.hdds.scm.exceptions.SCMException;
import org.apache.hadoop.hdds.scm.protocol.ScmBlockLocationProtocol;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.ozone.common.BlockGroup;
import org.apache.hadoop.ozone.om.KeyDeletingService;
import org.apache.hadoop.ozone.om.KeyManager;
import org.apache.hadoop.ozone.om.OMMetadataManager;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.helpers.OmKeyArgs;
import org.apache.hadoop.ozone.om.helpers.OmKeyInfo;
import org.apache.hadoop.ozone.om.helpers.OmKeyLocationInfo;
import org.apache.hadoop.ozone.om.helpers.OmKeyLocationInfoGroup;
import org.apache.hadoop.ozone.om.helpers.OpenKeySession;
import org.apache.hadoop.ozone.protocol.proto.OzoneManagerProtocolProtos;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.utils.BackgroundService;
import org.rocksdb.RocksDBException;
import org.rocksdb.WriteBatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyManagerImpl
implements KeyManager {
    private static final Logger LOG = LoggerFactory.getLogger(KeyManagerImpl.class);
    private final ScmBlockLocationProtocol scmBlockClient;
    private final OMMetadataManager metadataManager;
    private final long scmBlockSize;
    private final boolean useRatis;
    private final long preallocateMax;
    private final String omId;
    private final BackgroundService keyDeletingService;

    public KeyManagerImpl(ScmBlockLocationProtocol scmBlockClient, OMMetadataManager metadataManager, OzoneConfiguration conf, String omId) {
        this.scmBlockClient = scmBlockClient;
        this.metadataManager = metadataManager;
        this.scmBlockSize = conf.getLong("ozone.scm.block.size.in.mb", 256L) * 0x100000L;
        this.useRatis = conf.getBoolean("dfs.container.ratis.enabled", false);
        this.preallocateMax = conf.getLong("ozone.key.preallocation.maxsize", 0x8000000L);
        long blockDeleteInterval = conf.getTimeDuration("ozone.block.deleting.service.interval", "60s", TimeUnit.MILLISECONDS);
        long serviceTimeout = conf.getTimeDuration("ozone.block.deleting.service.timeout", "300s", TimeUnit.MILLISECONDS);
        this.keyDeletingService = new KeyDeletingService(scmBlockClient, this, blockDeleteInterval, serviceTimeout, (Configuration)conf);
        this.omId = omId;
    }

    @Override
    public void start() {
        this.keyDeletingService.start();
    }

    @Override
    public void stop() throws IOException {
        this.keyDeletingService.shutdown();
    }

    private void validateBucket(String volumeName, String bucketName) throws IOException {
        byte[] volumeKey = this.metadataManager.getVolumeKey(volumeName);
        byte[] bucketKey = this.metadataManager.getBucketKey(volumeName, bucketName);
        if (this.metadataManager.getVolumeTable().get(volumeKey) == null) {
            LOG.error("volume not found: {}", (Object)volumeName);
            throw new OMException("Volume not found", OMException.ResultCodes.FAILED_VOLUME_NOT_FOUND);
        }
        if (this.metadataManager.getBucketTable().get(bucketKey) == null) {
            LOG.error("bucket not found: {}/{} ", (Object)volumeName, (Object)bucketName);
            throw new OMException("Bucket not found", OMException.ResultCodes.FAILED_BUCKET_NOT_FOUND);
        }
    }

    @Override
    public OmKeyLocationInfo allocateBlock(OmKeyArgs args, long clientID) throws IOException {
        AllocatedBlock allocatedBlock;
        Preconditions.checkNotNull((Object)args);
        String volumeName = args.getVolumeName();
        String bucketName = args.getBucketName();
        String keyName = args.getKeyName();
        this.validateBucket(volumeName, bucketName);
        byte[] openKey = this.metadataManager.getOpenKeyBytes(volumeName, bucketName, keyName, clientID);
        byte[] keyData = this.metadataManager.getOpenKeyTable().get(openKey);
        if (keyData == null) {
            LOG.error("Allocate block for a key not in open status in meta store /{}/{}/{} with ID {}", new Object[]{volumeName, bucketName, keyName, clientID});
            throw new OMException("Open Key not found", OMException.ResultCodes.FAILED_KEY_NOT_FOUND);
        }
        OmKeyInfo keyInfo = OmKeyInfo.getFromProtobuf((OzoneManagerProtocolProtos.KeyInfo)OzoneManagerProtocolProtos.KeyInfo.parseFrom((byte[])keyData));
        try {
            allocatedBlock = this.scmBlockClient.allocateBlock(this.scmBlockSize, keyInfo.getType(), keyInfo.getFactor(), this.omId);
        }
        catch (SCMException ex) {
            if (ex.getResult().equals((Object)SCMException.ResultCodes.CHILL_MODE_EXCEPTION)) {
                throw new OMException(ex.getMessage(), OMException.ResultCodes.SCM_IN_CHILL_MODE);
            }
            throw ex;
        }
        OmKeyLocationInfo info = new OmKeyLocationInfo.Builder().setBlockID(allocatedBlock.getBlockID()).setShouldCreateContainer(allocatedBlock.getCreateContainer()).setLength(this.scmBlockSize).setBlockCommitSequenceId(0L).setOffset(0L).build();
        keyInfo.appendNewBlocks(Collections.singletonList(info));
        keyInfo.updateModifcationTime();
        this.metadataManager.getOpenKeyTable().put(openKey, keyInfo.getProtobuf().toByteArray());
        return info;
    }

    @Override
    public OpenKeySession openKey(OmKeyArgs args) throws IOException {
        Preconditions.checkNotNull((Object)args);
        String volumeName = args.getVolumeName();
        String bucketName = args.getBucketName();
        this.validateBucket(volumeName, bucketName);
        this.metadataManager.getLock().acquireBucketLock(volumeName, bucketName);
        String keyName = args.getKeyName();
        HddsProtos.ReplicationFactor factor = args.getFactor();
        HddsProtos.ReplicationType type = args.getType();
        long currentTime = Time.monotonicNowNanos();
        if (factor == null) {
            HddsProtos.ReplicationFactor replicationFactor = factor = this.useRatis ? HddsProtos.ReplicationFactor.THREE : HddsProtos.ReplicationFactor.ONE;
        }
        if (type == null) {
            type = this.useRatis ? HddsProtos.ReplicationType.RATIS : HddsProtos.ReplicationType.STAND_ALONE;
        }
        try {
            long openVersion;
            OmKeyInfo keyInfo;
            long allocateSize;
            ArrayList<OmKeyLocationInfo> locations = new ArrayList<OmKeyLocationInfo>();
            byte[] objectKey = this.metadataManager.getOzoneKeyBytes(volumeName, bucketName, keyName);
            for (long requestedSize = Math.min(this.preallocateMax, args.getDataSize()); requestedSize > 0L; requestedSize -= allocateSize) {
                AllocatedBlock allocatedBlock;
                allocateSize = Math.min(this.scmBlockSize, requestedSize);
                try {
                    allocatedBlock = this.scmBlockClient.allocateBlock(allocateSize, type, factor, this.omId);
                }
                catch (IOException ex) {
                    if (ex instanceof SCMException && ((SCMException)((Object)ex)).getResult().equals((Object)SCMException.ResultCodes.CHILL_MODE_EXCEPTION)) {
                        throw new OMException(ex.getMessage(), OMException.ResultCodes.SCM_IN_CHILL_MODE);
                    }
                    throw ex;
                }
                OmKeyLocationInfo subKeyInfo = new OmKeyLocationInfo.Builder().setBlockID(allocatedBlock.getBlockID()).setShouldCreateContainer(allocatedBlock.getCreateContainer()).setLength(allocateSize).setBlockCommitSequenceId(0L).setOffset(0L).build();
                locations.add(subKeyInfo);
            }
            long size = args.getDataSize() >= 0L ? args.getDataSize() : this.scmBlockSize;
            byte[] value = this.metadataManager.getKeyTable().get(objectKey);
            if (value != null) {
                keyInfo = OmKeyInfo.getFromProtobuf((OzoneManagerProtocolProtos.KeyInfo)OzoneManagerProtocolProtos.KeyInfo.parseFrom((byte[])value));
                openVersion = keyInfo.addNewVersion(locations);
                keyInfo.setDataSize(size + keyInfo.getDataSize());
            } else {
                keyInfo = new OmKeyInfo.Builder().setVolumeName(args.getVolumeName()).setBucketName(args.getBucketName()).setKeyName(args.getKeyName()).setOmKeyLocationInfos(Collections.singletonList(new OmKeyLocationInfoGroup(0L, locations))).setCreationTime(Time.now()).setModificationTime(Time.now()).setDataSize(size).setReplicationType(type).setReplicationFactor(factor).build();
                openVersion = 0L;
            }
            byte[] openKey = this.metadataManager.getOpenKeyBytes(volumeName, bucketName, keyName, currentTime);
            if (this.metadataManager.getOpenKeyTable().get(openKey) != null) {
                LOG.warn("Cannot allocate key. The generated open key id is alreadyused for the same key which is currently being written.");
                throw new OMException("Cannot allocate key. Not able to get a validopen key id.", OMException.ResultCodes.FAILED_KEY_ALLOCATION);
            }
            this.metadataManager.getOpenKeyTable().put(openKey, keyInfo.getProtobuf().toByteArray());
            LOG.debug("Key {} allocated in volume {} bucket {}", new Object[]{keyName, volumeName, bucketName});
            OpenKeySession openKeySession = new OpenKeySession(currentTime, keyInfo, openVersion);
            return openKeySession;
        }
        catch (OMException e) {
            throw e;
        }
        catch (IOException ex) {
            LOG.error("Key open failed for volume:{} bucket:{} key:{}", new Object[]{volumeName, bucketName, keyName, ex});
            throw new OMException(ex.getMessage(), OMException.ResultCodes.FAILED_KEY_ALLOCATION);
        }
        finally {
            this.metadataManager.getLock().releaseBucketLock(volumeName, bucketName);
        }
    }

    @Override
    public void commitKey(OmKeyArgs args, long clientID) throws IOException {
        Preconditions.checkNotNull((Object)args);
        String volumeName = args.getVolumeName();
        String bucketName = args.getBucketName();
        String keyName = args.getKeyName();
        this.metadataManager.getLock().acquireBucketLock(volumeName, bucketName);
        try {
            this.validateBucket(volumeName, bucketName);
            byte[] openKey = this.metadataManager.getOpenKeyBytes(volumeName, bucketName, keyName, clientID);
            byte[] objectKey = this.metadataManager.getOzoneKeyBytes(volumeName, bucketName, keyName);
            byte[] openKeyData = this.metadataManager.getOpenKeyTable().get(openKey);
            if (openKeyData == null) {
                throw new OMException("Commit a key without corresponding entry " + DFSUtil.bytes2String((byte[])objectKey), OMException.ResultCodes.FAILED_KEY_NOT_FOUND);
            }
            OmKeyInfo keyInfo = OmKeyInfo.getFromProtobuf((OzoneManagerProtocolProtos.KeyInfo)OzoneManagerProtocolProtos.KeyInfo.parseFrom((byte[])openKeyData));
            keyInfo.setDataSize(args.getDataSize());
            keyInfo.setModificationTime(Time.now());
            List locationInfoList = args.getLocationInfoList();
            Preconditions.checkNotNull((Object)locationInfoList);
            keyInfo.updateLocationInfoList(locationInfoList);
            this.metadataManager.getStore().move(openKey, objectKey, keyInfo.getProtobuf().toByteArray(), this.metadataManager.getOpenKeyTable(), this.metadataManager.getKeyTable());
        }
        catch (OMException e) {
            throw e;
        }
        catch (IOException ex) {
            LOG.error("Key commit failed for volume:{} bucket:{} key:{}", new Object[]{volumeName, bucketName, keyName, ex});
            throw new OMException(ex.getMessage(), OMException.ResultCodes.FAILED_KEY_ALLOCATION);
        }
        finally {
            this.metadataManager.getLock().releaseBucketLock(volumeName, bucketName);
        }
    }

    @Override
    public OmKeyInfo lookupKey(OmKeyArgs args) throws IOException {
        Preconditions.checkNotNull((Object)args);
        String volumeName = args.getVolumeName();
        String bucketName = args.getBucketName();
        String keyName = args.getKeyName();
        this.metadataManager.getLock().acquireBucketLock(volumeName, bucketName);
        try {
            byte[] keyBytes = this.metadataManager.getOzoneKeyBytes(volumeName, bucketName, keyName);
            byte[] value = this.metadataManager.getKeyTable().get(keyBytes);
            if (value == null) {
                LOG.debug("volume:{} bucket:{} Key:{} not found", new Object[]{volumeName, bucketName, keyName});
                throw new OMException("Key not found", OMException.ResultCodes.FAILED_KEY_NOT_FOUND);
            }
            OmKeyInfo omKeyInfo = OmKeyInfo.getFromProtobuf((OzoneManagerProtocolProtos.KeyInfo)OzoneManagerProtocolProtos.KeyInfo.parseFrom((byte[])value));
            return omKeyInfo;
        }
        catch (IOException ex) {
            LOG.debug("Get key failed for volume:{} bucket:{} key:{}", new Object[]{volumeName, bucketName, keyName, ex});
            throw new OMException(ex.getMessage(), OMException.ResultCodes.FAILED_KEY_NOT_FOUND);
        }
        finally {
            this.metadataManager.getLock().releaseBucketLock(volumeName, bucketName);
        }
    }

    @Override
    public void renameKey(OmKeyArgs args, String toKeyName) throws IOException {
        Preconditions.checkNotNull((Object)args);
        Preconditions.checkNotNull((Object)toKeyName);
        String volumeName = args.getVolumeName();
        String bucketName = args.getBucketName();
        String fromKeyName = args.getKeyName();
        if (toKeyName.length() == 0 || fromKeyName.length() == 0) {
            LOG.error("Rename key failed for volume:{} bucket:{} fromKey:{} toKey:{}", new Object[]{volumeName, bucketName, fromKeyName, toKeyName});
            throw new OMException("Key name is empty", OMException.ResultCodes.FAILED_INVALID_KEY_NAME);
        }
        this.metadataManager.getLock().acquireBucketLock(volumeName, bucketName);
        try {
            byte[] fromKey = this.metadataManager.getOzoneKeyBytes(volumeName, bucketName, fromKeyName);
            byte[] fromKeyValue = this.metadataManager.getKeyTable().get(fromKey);
            if (fromKeyValue == null) {
                LOG.error("Rename key failed for volume:{} bucket:{} fromKey:{} toKey:{}. Key: {} not found.", new Object[]{volumeName, bucketName, fromKeyName, toKeyName, fromKeyName});
                throw new OMException("Key not found", OMException.ResultCodes.FAILED_KEY_NOT_FOUND);
            }
            if (fromKeyName.equals(toKeyName)) {
                return;
            }
            byte[] toKey = this.metadataManager.getOzoneKeyBytes(volumeName, bucketName, toKeyName);
            byte[] toKeyValue = this.metadataManager.getKeyTable().get(toKey);
            if (toKeyValue != null) {
                LOG.error("Rename key failed for volume:{} bucket:{} fromKey:{} toKey:{}. Key: {} already exists.", new Object[]{volumeName, bucketName, fromKeyName, toKeyName, toKeyName});
                throw new OMException("Key not found", OMException.ResultCodes.FAILED_KEY_ALREADY_EXISTS);
            }
            OmKeyInfo newKeyInfo = OmKeyInfo.getFromProtobuf((OzoneManagerProtocolProtos.KeyInfo)OzoneManagerProtocolProtos.KeyInfo.parseFrom((byte[])fromKeyValue));
            newKeyInfo.setKeyName(toKeyName);
            newKeyInfo.updateModifcationTime();
            try (WriteBatch batch = new WriteBatch();){
                batch.delete(this.metadataManager.getKeyTable().getHandle(), fromKey);
                batch.put(this.metadataManager.getKeyTable().getHandle(), toKey, newKeyInfo.getProtobuf().toByteArray());
                this.metadataManager.getStore().write(batch);
            }
        }
        catch (IOException | RocksDBException ex) {
            LOG.error("Rename key failed for volume:{} bucket:{} fromKey:{} toKey:{}", new Object[]{volumeName, bucketName, fromKeyName, toKeyName, ex});
            throw new OMException(ex.getMessage(), OMException.ResultCodes.FAILED_KEY_RENAME);
        }
        finally {
            this.metadataManager.getLock().releaseBucketLock(volumeName, bucketName);
        }
    }

    @Override
    public void deleteKey(OmKeyArgs args) throws IOException {
        Preconditions.checkNotNull((Object)args);
        String volumeName = args.getVolumeName();
        String bucketName = args.getBucketName();
        String keyName = args.getKeyName();
        this.metadataManager.getLock().acquireBucketLock(volumeName, bucketName);
        try {
            byte[] objectKey = this.metadataManager.getOzoneKeyBytes(volumeName, bucketName, keyName);
            byte[] objectValue = this.metadataManager.getKeyTable().get(objectKey);
            if (objectValue == null) {
                throw new OMException("Key not found", OMException.ResultCodes.FAILED_KEY_NOT_FOUND);
            }
            OzoneManagerProtocolProtos.KeyInfo keyInfo = OzoneManagerProtocolProtos.KeyInfo.parseFrom((byte[])objectValue);
            if (this.isKeyEmpty(keyInfo)) {
                this.metadataManager.getKeyTable().delete(objectKey);
                LOG.debug("Key {} deleted from OM DB", (Object)keyName);
                return;
            }
            this.metadataManager.getStore().move(objectKey, this.metadataManager.getKeyTable(), this.metadataManager.getDeletedTable());
        }
        catch (OMException ex) {
            throw ex;
        }
        catch (IOException ex) {
            LOG.error(String.format("Delete key failed for volume:%s bucket:%s key:%s", volumeName, bucketName, keyName), (Throwable)ex);
            throw new OMException(ex.getMessage(), ex, OMException.ResultCodes.FAILED_KEY_DELETION);
        }
        finally {
            this.metadataManager.getLock().releaseBucketLock(volumeName, bucketName);
        }
    }

    private boolean isKeyEmpty(OzoneManagerProtocolProtos.KeyInfo keyInfo) {
        for (OzoneManagerProtocolProtos.KeyLocationList keyLocationList : keyInfo.getKeyLocationListList()) {
            if (keyLocationList.getKeyLocationsCount() == 0) continue;
            return false;
        }
        return true;
    }

    @Override
    public List<OmKeyInfo> listKeys(String volumeName, String bucketName, String startKey, String keyPrefix, int maxKeys) throws IOException {
        Preconditions.checkNotNull((Object)volumeName);
        Preconditions.checkNotNull((Object)bucketName);
        return this.metadataManager.listKeys(volumeName, bucketName, startKey, keyPrefix, maxKeys);
    }

    @Override
    public List<BlockGroup> getPendingDeletionKeys(int count) throws IOException {
        return this.metadataManager.getPendingDeletionKeys(count);
    }

    @Override
    public List<BlockGroup> getExpiredOpenKeys() throws IOException {
        return this.metadataManager.getExpiredOpenKeys();
    }

    @Override
    public void deleteExpiredOpenKey(String objectKeyName) throws IOException {
        Preconditions.checkNotNull((Object)objectKeyName);
    }

    @Override
    public OMMetadataManager getMetadataManager() {
        return this.metadataManager;
    }

    @Override
    public BackgroundService getDeletingService() {
        return this.keyDeletingService;
    }
}

