/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.UUID;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.FollowerInfo;
import org.apache.ratis.server.impl.LeaderState;
import org.apache.ratis.server.impl.RaftServerImpl;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.FileInfo;
import org.apache.ratis.server.storage.RaftLog;
import org.apache.ratis.server.storage.RaftLogIOException;
import org.apache.ratis.statemachine.SnapshotInfo;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.util.Daemon;
import org.apache.ratis.util.IOUtils;
import org.apache.ratis.util.LifeCycle;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.ProtoUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LogAppender {
    public static final Logger LOG = LoggerFactory.getLogger(LogAppender.class);
    protected final RaftServerImpl server;
    private final LeaderState leaderState;
    protected final RaftLog raftLog;
    protected final FollowerInfo follower;
    private final int maxBufferSize;
    private final boolean batchSending;
    private final LogEntryBuffer buffer;
    private final int snapshotChunkMaxSize;
    protected final long halfMinTimeoutMs;
    private final LifeCycle lifeCycle;
    private final Daemon daemon = new Daemon(this::runAppender);

    public LogAppender(RaftServerImpl server, LeaderState leaderState, FollowerInfo f) {
        this.follower = f;
        this.server = server;
        this.leaderState = leaderState;
        this.raftLog = server.getState().getLog();
        RaftProperties properties = server.getProxy().getProperties();
        this.maxBufferSize = RaftServerConfigKeys.Log.Appender.bufferCapacity(properties).getSizeInt();
        this.batchSending = RaftServerConfigKeys.Log.Appender.batchEnabled(properties);
        this.snapshotChunkMaxSize = RaftServerConfigKeys.Log.Appender.snapshotChunkSizeMax(properties).getSizeInt();
        this.halfMinTimeoutMs = server.getMinTimeoutMs() / 2;
        this.buffer = new LogEntryBuffer();
        this.lifeCycle = new LifeCycle((Object)this);
    }

    public String toString() {
        return this.getClass().getSimpleName() + "(" + this.server.getId() + " -> " + this.follower.getPeer().getId() + ")";
    }

    public void startAppender() {
        this.lifeCycle.transition(LifeCycle.State.STARTING);
        this.daemon.start();
    }

    private void runAppender() {
        this.lifeCycle.transition(LifeCycle.State.RUNNING);
        try {
            this.runAppenderImpl();
        }
        catch (InterruptedIOException | InterruptedException e) {
            LOG.info(this + " was interrupted: " + e);
        }
        catch (IOException e) {
            LOG.error(this + " hit IOException while loading raft log", (Throwable)e);
            this.lifeCycle.transition(LifeCycle.State.EXCEPTION);
        }
        catch (Throwable e) {
            LOG.error(this + " unexpected exception", e);
            this.lifeCycle.transition(LifeCycle.State.EXCEPTION);
        }
        finally {
            if (!this.lifeCycle.compareAndTransition(LifeCycle.State.CLOSING, LifeCycle.State.CLOSED)) {
                this.lifeCycle.transitionIfNotEqual(LifeCycle.State.EXCEPTION);
            }
        }
    }

    protected boolean isAppenderRunning() {
        return !this.lifeCycle.getCurrentState().isOneOf(new LifeCycle.State[]{LifeCycle.State.CLOSING, LifeCycle.State.CLOSED, LifeCycle.State.EXCEPTION});
    }

    public void stopAppender() {
        if (this.lifeCycle.compareAndTransition(LifeCycle.State.NEW, LifeCycle.State.CLOSED)) {
            return;
        }
        this.lifeCycle.transition(LifeCycle.State.CLOSING);
        this.daemon.interrupt();
    }

    public FollowerInfo getFollower() {
        return this.follower;
    }

    RaftPeerId getFollowerId() {
        return this.getFollower().getPeer().getId();
    }

    private TermIndex getPrevious() {
        TermIndex previous = this.raftLog.getTermIndex(this.follower.getNextIndex() - 1L);
        if (previous == null) {
            Preconditions.assertTrue((this.follower.getNextIndex() == this.raftLog.getStartIndex() ? 1 : 0) != 0, (String)"%s: follower's next index %s, local log start index %s", (Object[])new Object[]{this, this.follower.getNextIndex(), this.raftLog.getStartIndex()});
            SnapshotInfo snapshot = this.server.getState().getLatestSnapshot();
            previous = snapshot == null ? null : snapshot.getTermIndex();
        }
        return previous;
    }

    protected RaftProtos.AppendEntriesRequestProto createRequest(long callId) throws RaftLogIOException {
        boolean toSend;
        long next;
        TermIndex previous = this.getPrevious();
        long leaderNext = this.raftLog.getNextIndex();
        if (leaderNext == (next = this.follower.getNextIndex() + (long)this.buffer.getPendingEntryNum()) && !this.buffer.isEmpty()) {
            toSend = true;
        } else if (leaderNext > next) {
            boolean hasSpace = true;
            while (hasSpace && leaderNext > next) {
                hasSpace = this.buffer.addEntry(this.raftLog.getEntryWithData(next++));
            }
            toSend = !hasSpace || !this.batchSending;
        } else {
            toSend = false;
        }
        if (toSend || this.shouldHeartbeat()) {
            return this.buffer.getAppendRequest(previous, callId);
        }
        return null;
    }

    private RaftProtos.AppendEntriesReplyProto sendAppendEntriesWithRetries() throws InterruptedException, InterruptedIOException, RaftLogIOException {
        int retry = 0;
        RaftProtos.AppendEntriesRequestProto request = null;
        while (this.isAppenderRunning()) {
            try {
                if (request == null || request.getEntriesCount() == 0) {
                    request = this.createRequest(0L);
                }
                if (request == null) {
                    LOG.trace("{} need not send AppendEntries now. Wait for more entries.", (Object)this.server.getId());
                    return null;
                }
                if (!this.isAppenderRunning()) {
                    LOG.debug("LogAppender {} has been stopped. Skip the request.", (Object)this);
                    return null;
                }
                this.follower.updateLastRpcSendTime();
                RaftProtos.AppendEntriesReplyProto r = this.server.getServerRpc().appendEntries(request);
                this.follower.updateLastRpcResponseTime();
                this.updateCommitIndex(r.getFollowerCommit());
                return r;
            }
            catch (InterruptedIOException | RaftLogIOException e) {
                throw e;
            }
            catch (IOException ioe) {
                if (retry++ % 10 == 0) {
                    LOG.warn("{}: Failed to appendEntries (retry={}): {}", new Object[]{this, retry++, ioe});
                }
                this.handleException(ioe);
                if (!this.isAppenderRunning()) continue;
                this.leaderState.getSyncInterval().sleep();
            }
        }
        return null;
    }

    protected void updateCommitIndex(long commitIndex) {
        if (this.follower.updateCommitIndex(commitIndex)) {
            this.server.commitIndexChanged();
        }
    }

    private RaftProtos.FileChunkProto readFileChunk(FileInfo fileInfo, FileInputStream in, byte[] buf, int length, long offset, int chunkIndex) throws IOException {
        RaftProtos.FileChunkProto.Builder builder = RaftProtos.FileChunkProto.newBuilder().setOffset(offset).setChunkIndex(chunkIndex);
        IOUtils.readFully((InputStream)in, (byte[])buf, (int)0, (int)length);
        Path relativePath = this.server.getState().getStorage().getStorageDir().relativizeToRoot(fileInfo.getPath());
        builder.setFilename(relativePath.toString());
        builder.setDone(offset + (long)length == fileInfo.getFileSize());
        builder.setFileDigest(ByteString.copyFrom((byte[])fileInfo.getFileDigest().getDigest()));
        builder.setData(ByteString.copyFrom((byte[])buf, (int)0, (int)length));
        return builder.build();
    }

    private RaftProtos.InstallSnapshotReplyProto installSnapshot(SnapshotInfo snapshot) throws InterruptedIOException {
        String requestId = UUID.randomUUID().toString();
        RaftProtos.InstallSnapshotReplyProto reply = null;
        try {
            for (RaftProtos.InstallSnapshotRequestProto request : new SnapshotRequestIter(snapshot, requestId)) {
                this.follower.updateLastRpcSendTime();
                reply = this.server.getServerRpc().installSnapshot(request);
                this.follower.updateLastRpcResponseTime();
                if (reply.getServerReply().getSuccess()) continue;
                return reply;
            }
        }
        catch (InterruptedIOException iioe) {
            throw iioe;
        }
        catch (Exception ioe) {
            LOG.warn("{}: Failed to installSnapshot {}: {}", new Object[]{this, snapshot, ioe});
            this.handleException(ioe);
            return null;
        }
        if (reply != null) {
            this.follower.updateMatchIndex(snapshot.getTermIndex().getIndex());
            this.follower.updateNextIndex(snapshot.getTermIndex().getIndex() + 1L);
            LOG.info("{}: install snapshot-{} successfully on follower {}", new Object[]{this.server.getId(), snapshot.getTermIndex().getIndex(), this.follower.getPeer()});
        }
        return reply;
    }

    protected SnapshotInfo shouldInstallSnapshot() {
        long logStartIndex = this.raftLog.getStartIndex();
        if (this.follower.getNextIndex() < this.raftLog.getNextIndex()) {
            SnapshotInfo snapshot = this.server.getState().getLatestSnapshot();
            if (this.follower.getNextIndex() < logStartIndex || logStartIndex == -1L && snapshot != null) {
                return snapshot;
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void runAppenderImpl() throws InterruptedException, IOException {
        while (this.isAppenderRunning()) {
            long waitTime;
            if (this.shouldSendRequest()) {
                RaftProtos.AppendEntriesReplyProto r;
                SnapshotInfo snapshot = this.shouldInstallSnapshot();
                if (snapshot != null) {
                    LOG.info("{}: follower {}'s next index is {}, log's start index is {}, need to install snapshot", new Object[]{this.server.getId(), this.follower.getPeer(), this.follower.getNextIndex(), this.raftLog.getStartIndex()});
                    r = this.installSnapshot(snapshot);
                    if (r != null && r.getResult() == RaftProtos.InstallSnapshotResult.NOT_LEADER) {
                        this.checkResponseTerm(r.getTerm());
                    }
                } else {
                    r = this.sendAppendEntriesWithRetries();
                    if (r != null) {
                        this.handleReply(r);
                    }
                }
            }
            if (this.isAppenderRunning() && !this.shouldAppendEntries(this.follower.getNextIndex() + (long)this.buffer.getPendingEntryNum()) && (waitTime = this.getHeartbeatRemainingTime()) > 0L) {
                LogAppender logAppender = this;
                synchronized (logAppender) {
                    this.wait(waitTime);
                }
            }
            this.checkSlowness();
        }
    }

    private void handleReply(RaftProtos.AppendEntriesReplyProto reply) {
        if (reply != null) {
            switch (reply.getResult()) {
                case SUCCESS: {
                    long oldNextIndex = this.follower.getNextIndex();
                    long nextIndex = reply.getNextIndex();
                    if (nextIndex < oldNextIndex) {
                        throw new IllegalStateException("nextIndex=" + nextIndex + " < oldNextIndex=" + oldNextIndex + ", reply=" + ProtoUtils.toString((RaftProtos.AppendEntriesReplyProto)reply));
                    }
                    if (nextIndex <= oldNextIndex) break;
                    this.follower.updateMatchIndex(nextIndex - 1L);
                    this.follower.updateNextIndex(nextIndex);
                    this.submitEventOnSuccessAppend();
                    break;
                }
                case NOT_LEADER: {
                    this.checkResponseTerm(reply.getTerm());
                    break;
                }
                case INCONSISTENCY: {
                    this.follower.decreaseNextIndex(reply.getNextIndex());
                    break;
                }
                case UNRECOGNIZED: {
                    LOG.warn("{} received UNRECOGNIZED AppendResult from {}", (Object)this.server.getId(), (Object)this.follower.getPeer().getId());
                }
            }
        }
    }

    private void handleException(Exception e) {
        LOG.trace("TRACE", (Throwable)e);
        this.server.getServerRpc().handleException(this.follower.getPeer().getId(), e, false);
    }

    protected void submitEventOnSuccessAppend() {
        if (this.follower.isAttendingVote()) {
            this.leaderState.submitUpdateCommitEvent();
        } else {
            this.leaderState.submitCheckStagingEvent();
        }
    }

    protected void checkSlowness() {
        if (this.follower.isSlow()) {
            this.server.getStateMachine().notifySlowness(this.server.getGroup(), this.server.getRoleInfoProto());
        }
    }

    public synchronized void notifyAppend() {
        this.notify();
    }

    protected boolean shouldSendRequest() {
        return this.shouldAppendEntries(this.follower.getNextIndex()) || this.shouldHeartbeat();
    }

    private boolean shouldAppendEntries(long followerIndex) {
        return followerIndex < this.raftLog.getNextIndex();
    }

    private boolean shouldHeartbeat() {
        return this.getHeartbeatRemainingTime() <= 0L;
    }

    protected long getHeartbeatRemainingTime() {
        return this.halfMinTimeoutMs - this.follower.getLastRpcTime().elapsedTimeMs();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void checkResponseTerm(long responseTerm) {
        RaftServerImpl raftServerImpl = this.server;
        synchronized (raftServerImpl) {
            if (this.isAppenderRunning() && this.follower.isAttendingVote() && responseTerm > this.leaderState.getCurrentTerm()) {
                this.leaderState.submitStepDownEvent(responseTerm);
            }
        }
    }

    protected class SnapshotRequestIter
    implements Iterable<RaftProtos.InstallSnapshotRequestProto> {
        private final SnapshotInfo snapshot;
        private final List<FileInfo> files;
        private FileInputStream in;
        private int fileIndex = 0;
        private FileInfo currentFileInfo;
        private byte[] currentBuf;
        private long currentFileSize;
        private long currentOffset = 0L;
        private int chunkIndex = 0;
        private final String requestId;
        private int requestIndex = 0;

        public SnapshotRequestIter(SnapshotInfo snapshot, String requestId) throws IOException {
            this.snapshot = snapshot;
            this.requestId = requestId;
            this.files = snapshot.getFiles();
            if (this.files.size() > 0) {
                this.startReadFile();
            }
        }

        private void startReadFile() throws IOException {
            this.currentFileInfo = this.files.get(this.fileIndex);
            File snapshotFile = this.currentFileInfo.getPath().toFile();
            this.currentFileSize = snapshotFile.length();
            int bufLength = this.getSnapshotChunkLength(this.currentFileSize);
            this.currentBuf = new byte[bufLength];
            this.currentOffset = 0L;
            this.chunkIndex = 0;
            this.in = new FileInputStream(snapshotFile);
        }

        private int getSnapshotChunkLength(long len) {
            return len < (long)LogAppender.this.snapshotChunkMaxSize ? (int)len : LogAppender.this.snapshotChunkMaxSize;
        }

        @Override
        public Iterator<RaftProtos.InstallSnapshotRequestProto> iterator() {
            return new Iterator<RaftProtos.InstallSnapshotRequestProto>(){

                @Override
                public boolean hasNext() {
                    return SnapshotRequestIter.this.fileIndex < SnapshotRequestIter.this.files.size();
                }

                @Override
                public RaftProtos.InstallSnapshotRequestProto next() {
                    if (SnapshotRequestIter.this.fileIndex >= SnapshotRequestIter.this.files.size()) {
                        throw new NoSuchElementException();
                    }
                    int targetLength = SnapshotRequestIter.this.getSnapshotChunkLength(SnapshotRequestIter.this.currentFileSize - SnapshotRequestIter.this.currentOffset);
                    try {
                        RaftProtos.FileChunkProto chunk = LogAppender.this.readFileChunk(SnapshotRequestIter.this.currentFileInfo, SnapshotRequestIter.this.in, SnapshotRequestIter.this.currentBuf, targetLength, SnapshotRequestIter.this.currentOffset, SnapshotRequestIter.this.chunkIndex);
                        boolean done = SnapshotRequestIter.this.fileIndex == SnapshotRequestIter.this.files.size() - 1 && chunk.getDone();
                        RaftProtos.InstallSnapshotRequestProto request = LogAppender.this.server.createInstallSnapshotRequest(LogAppender.this.follower.getPeer().getId(), SnapshotRequestIter.this.requestId, SnapshotRequestIter.this.requestIndex++, SnapshotRequestIter.this.snapshot, Collections.singletonList(chunk), done);
                        SnapshotRequestIter.this.currentOffset = SnapshotRequestIter.this.currentOffset + (long)targetLength;
                        SnapshotRequestIter.this.chunkIndex++;
                        if (SnapshotRequestIter.this.currentOffset >= SnapshotRequestIter.this.currentFileSize) {
                            SnapshotRequestIter.this.in.close();
                            SnapshotRequestIter.this.fileIndex++;
                            if (SnapshotRequestIter.this.fileIndex < SnapshotRequestIter.this.files.size()) {
                                SnapshotRequestIter.this.startReadFile();
                            }
                        }
                        return request;
                    }
                    catch (IOException e) {
                        if (SnapshotRequestIter.this.in != null) {
                            try {
                                SnapshotRequestIter.this.in.close();
                            }
                            catch (IOException iOException) {
                                // empty catch block
                            }
                        }
                        LOG.warn("Got exception when preparing InstallSnapshot request", (Throwable)e);
                        throw new RuntimeException(e);
                    }
                }
            };
        }
    }

    private class LogEntryBuffer {
        private final List<RaftLog.EntryWithData> buf = new ArrayList<RaftLog.EntryWithData>();
        private int totalSize = 0;

        private LogEntryBuffer() {
        }

        boolean addEntry(RaftLog.EntryWithData entry) {
            int entrySize = entry.getSerializedSize();
            if (this.totalSize + entrySize <= LogAppender.this.maxBufferSize) {
                this.buf.add(entry);
                this.totalSize += entrySize;
                return true;
            }
            return false;
        }

        boolean isEmpty() {
            return this.buf.isEmpty();
        }

        RaftProtos.AppendEntriesRequestProto getAppendRequest(TermIndex previous, long callId) throws RaftLogIOException {
            ArrayList<RaftProtos.LogEntryProto> protos = new ArrayList<RaftProtos.LogEntryProto>();
            for (RaftLog.EntryWithData bufEntry : this.buf) {
                protos.add(bufEntry.getEntry());
            }
            RaftProtos.AppendEntriesRequestProto request = LogAppender.this.leaderState.newAppendEntriesRequestProto(LogAppender.this.getFollowerId(), previous, protos, !LogAppender.this.follower.isAttendingVote(), callId);
            this.buf.clear();
            this.totalSize = 0;
            return request;
        }

        int getPendingEntryNum() {
            return this.buf.size();
        }
    }
}

