/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.scm.cli;

import com.google.common.base.Preconditions;
import com.google.common.primitives.Longs;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Statement;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.ContainerInfo;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.DFSUtilClient;
import org.apache.hadoop.ozone.protocol.proto.OzoneManagerProtocolProtos;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.apache.hadoop.utils.MetadataStore;
import org.apache.hadoop.utils.MetadataStoreBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SQLCLI
extends Configured
implements Tool {
    private Options options;
    private BasicParser parser;
    private final Charset encoding = Charset.forName("UTF-8");
    private final OzoneConfiguration conf;
    private static final String CREATE_CONTAINER_INFO = "CREATE TABLE containerInfo (containerID LONG PRIMARY KEY NOT NULL, replicationType TEXT NOT NULL,replicationFactor TEXT NOT NULL,usedBytes LONG NOT NULL,allocatedBytes LONG NOT NULL,owner TEXT,numberOfKeys LONG)";
    private static final String CREATE_DATANODE_INFO = "CREATE TABLE datanodeInfo (hostName TEXT NOT NULL, datanodeUUId TEXT PRIMARY KEY NOT NULL,ipAddress TEXT, containerPort INTEGER NOT NULL);";
    private static final String INSERT_CONTAINER_INFO = "INSERT INTO containerInfo (containerID, replicationType, replicationFactor, usedBytes, allocatedBytes, owner, numberOfKeys) VALUES (\"%d\", \"%s\", \"%s\", \"%d\", \"%d\", \"%s\", \"%d\")";
    private static final String INSERT_DATANODE_INFO = "INSERT INTO datanodeInfo (hostname, datanodeUUid, ipAddress, containerPort) VALUES (\"%s\", \"%s\", \"%s\", \"%d\")";
    private static final String INSERT_CONTAINER_MEMBERS = "INSERT INTO containerMembers (containerName, datanodeUUID) VALUES (\"%s\", \"%s\")";
    private static final String CREATE_OPEN_CONTAINER = "CREATE TABLE openContainer (containerName TEXT PRIMARY KEY NOT NULL, containerUsed INTEGER NOT NULL)";
    private static final String INSERT_OPEN_CONTAINER = "INSERT INTO openContainer (containerName, containerUsed) VALUES (\"%s\", \"%s\")";
    private static final String CREATE_VOLUME_LIST = "CREATE TABLE volumeList (userName TEXT NOT NULL,volumeName TEXT NOT NULL,PRIMARY KEY (userName, volumeName))";
    private static final String INSERT_VOLUME_LIST = "INSERT INTO volumeList (userName, volumeName) VALUES (\"%s\", \"%s\")";
    private static final String CREATE_VOLUME_INFO = "CREATE TABLE volumeInfo (adminName TEXT NOT NULL,ownerName TEXT NOT NULL,volumeName TEXT NOT NULL,PRIMARY KEY (adminName, ownerName, volumeName))";
    private static final String INSERT_VOLUME_INFO = "INSERT INTO volumeInfo (adminName, ownerName, volumeName) VALUES (\"%s\", \"%s\", \"%s\")";
    private static final String CREATE_ACL_INFO = "CREATE TABLE aclInfo (adminName TEXT NOT NULL,ownerName TEXT NOT NULL,volumeName TEXT NOT NULL,type TEXT NOT NULL,userName TEXT NOT NULL,rights TEXT NOT NULL,FOREIGN KEY (adminName, ownerName, volumeName, userName, type)REFERENCES volumeInfo(adminName, ownerName, volumeName, userName, type)PRIMARY KEY (adminName, ownerName, volumeName, userName, type))";
    private static final String INSERT_ACL_INFO = "INSERT INTO aclInfo (adminName, ownerName, volumeName, type, userName, rights) VALUES (\"%s\", \"%s\", \"%s\", \"%s\", \"%s\", \"%s\")";
    private static final String CREATE_BUCKET_INFO = "CREATE TABLE bucketInfo (volumeName TEXT NOT NULL,bucketName TEXT NOT NULL,versionEnabled BOOLEAN NOT NULL,storageType TEXT,PRIMARY KEY (volumeName, bucketName))";
    private static final String INSERT_BUCKET_INFO = "INSERT INTO bucketInfo(volumeName, bucketName, versionEnabled, storageType)VALUES (\"%s\", \"%s\", \"%s\", \"%s\")";
    private static final String CREATE_KEY_INFO = "CREATE TABLE keyInfo (volumeName TEXT NOT NULL,bucketName TEXT NOT NULL,keyName TEXT NOT NULL,dataSize INTEGER,blockKey TEXT NOT NULL,containerName TEXT NOT NULL,PRIMARY KEY (volumeName, bucketName, keyName))";
    private static final String INSERT_KEY_INFO = "INSERT INTO keyInfo (volumeName, bucketName, keyName, dataSize, blockKey, containerName)VALUES (\"%s\", \"%s\", \"%s\", \"%s\", \"%s\", \"%s\")";
    private static final Logger LOG = LoggerFactory.getLogger(SQLCLI.class);

    public SQLCLI(OzoneConfiguration conf) {
        this.options = this.getOptions();
        this.parser = new BasicParser();
        this.conf = conf;
    }

    private Options getOptions() {
        Options allOptions = new Options();
        OptionBuilder.hasArg((boolean)false);
        OptionBuilder.withLongOpt((String)"help");
        OptionBuilder.withDescription((String)"display help message");
        Option helpOpt = OptionBuilder.create((String)"h");
        allOptions.addOption(helpOpt);
        OptionBuilder.withArgName((String)"DB path");
        OptionBuilder.withLongOpt((String)"dbPath");
        OptionBuilder.hasArgs((int)1);
        OptionBuilder.withDescription((String)"specify DB path");
        Option dbPathOption = OptionBuilder.create((String)"p");
        allOptions.addOption(dbPathOption);
        OptionBuilder.withArgName((String)"output path");
        OptionBuilder.withLongOpt((String)"outPath");
        OptionBuilder.hasArgs((int)1);
        OptionBuilder.withDescription((String)"specify output DB file path");
        Option outPathOption = OptionBuilder.create((String)"o");
        allOptions.addOption(outPathOption);
        return allOptions;
    }

    public void displayHelp() {
        HelpFormatter helpFormatter = new HelpFormatter();
        Options allOpts = this.getOptions();
        helpFormatter.printHelp("hdfs oz_debug -p <DB path> -o <Output DB file path>", allOpts);
    }

    public int run(String[] args) throws Exception {
        CommandLine commandLine = this.parseArgs(args);
        if (commandLine.hasOption("help")) {
            this.displayHelp();
            return 0;
        }
        if (!commandLine.hasOption("p") || !commandLine.hasOption("o")) {
            this.displayHelp();
            return -1;
        }
        String value = commandLine.getOptionValue("p");
        LOG.info("DB path {}", (Object)value);
        Path dbPath = Paths.get(value, new String[0]);
        if (!Files.exists(dbPath, new LinkOption[0])) {
            LOG.error("DB path not exist:{}", (Object)dbPath);
        }
        Path parentPath = dbPath.getParent();
        Path dbName = dbPath.getFileName();
        if (parentPath == null || dbName == null) {
            LOG.error("Error processing db path {}", (Object)dbPath);
            return -1;
        }
        value = commandLine.getOptionValue("o");
        Path outPath = Paths.get(value, new String[0]);
        if (outPath == null || outPath.getParent() == null) {
            LOG.error("Error processing output path {}", (Object)outPath);
            return -1;
        }
        if (outPath.toFile().isDirectory()) {
            LOG.error("The db output path should be a file instead of a directory");
            return -1;
        }
        Path outParentPath = outPath.getParent();
        if (outParentPath != null && !Files.exists(outParentPath, new LinkOption[0])) {
            Files.createDirectories(outParentPath, new FileAttribute[0]);
        }
        LOG.info("Parent path [{}] db name [{}]", (Object)parentPath, (Object)dbName);
        if (dbName.toString().endsWith("container.db")) {
            LOG.info("Converting container DB");
            this.convertContainerDB(dbPath, outPath);
        } else if (dbName.toString().equals("om.db")) {
            LOG.info("Converting om DB");
            this.convertOMDB(dbPath, outPath);
        } else {
            LOG.error("Unrecognized db name {}", (Object)dbName);
        }
        return 0;
    }

    private Connection connectDB(String dbPath) throws Exception {
        Class.forName("org.sqlite.JDBC");
        String connectPath = String.format("jdbc:sqlite:%s", dbPath);
        return DriverManager.getConnection(connectPath);
    }

    private void executeSQL(Connection conn, String sql) throws SQLException {
        try (Statement stmt = conn.createStatement();){
            stmt.executeUpdate(sql);
        }
    }

    private void convertOMDB(Path dbPath, Path outPath) throws Exception {
        LOG.info("Create tables for sql om db.");
        File dbFile = dbPath.toFile();
        try (MetadataStore dbStore = MetadataStoreBuilder.newBuilder().setConf((Configuration)this.conf).setDbFile(dbFile).build();
             Connection conn = this.connectDB(outPath.toString());){
            this.executeSQL(conn, CREATE_VOLUME_LIST);
            this.executeSQL(conn, CREATE_VOLUME_INFO);
            this.executeSQL(conn, CREATE_ACL_INFO);
            this.executeSQL(conn, CREATE_BUCKET_INFO);
            this.executeSQL(conn, CREATE_KEY_INFO);
            dbStore.iterate(null, (key, value) -> {
                String keyString = DFSUtilClient.bytes2String((byte[])key);
                KeyType type = this.getKeyType(keyString);
                try {
                    this.insertOMDB(conn, type, keyString, value);
                }
                catch (IOException | SQLException ex) {
                    LOG.error("Exception inserting key {} type {}", new Object[]{keyString, type, ex});
                }
                return true;
            });
        }
    }

    private void insertOMDB(Connection conn, KeyType type, String keyName, byte[] value) throws IOException, SQLException {
        switch (type) {
            case USER: {
                OzoneManagerProtocolProtos.VolumeList volumeList = OzoneManagerProtocolProtos.VolumeList.parseFrom((byte[])value);
                for (String volumeName : volumeList.getVolumeNamesList()) {
                    String insertVolumeList = String.format(INSERT_VOLUME_LIST, keyName, volumeName);
                    this.executeSQL(conn, insertVolumeList);
                }
                break;
            }
            case VOLUME: {
                OzoneManagerProtocolProtos.VolumeInfo volumeInfo = OzoneManagerProtocolProtos.VolumeInfo.parseFrom((byte[])value);
                String adminName = volumeInfo.getAdminName();
                String ownerName = volumeInfo.getOwnerName();
                String volumeName = volumeInfo.getVolume();
                String insertVolumeInfo = String.format(INSERT_VOLUME_INFO, adminName, ownerName, volumeName);
                this.executeSQL(conn, insertVolumeInfo);
                for (OzoneManagerProtocolProtos.OzoneAclInfo aclInfo : volumeInfo.getVolumeAclsList()) {
                    String insertAclInfo = String.format(INSERT_ACL_INFO, adminName, ownerName, volumeName, aclInfo.getType(), aclInfo.getName(), aclInfo.getRights());
                    this.executeSQL(conn, insertAclInfo);
                }
                break;
            }
            case BUCKET: {
                OzoneManagerProtocolProtos.BucketInfo bucketInfo = OzoneManagerProtocolProtos.BucketInfo.parseFrom((byte[])value);
                String insertBucketInfo = String.format(INSERT_BUCKET_INFO, bucketInfo.getVolumeName(), bucketInfo.getBucketName(), bucketInfo.getIsVersionEnabled(), bucketInfo.getStorageType());
                this.executeSQL(conn, insertBucketInfo);
                break;
            }
            case KEY: {
                OzoneManagerProtocolProtos.KeyInfo keyInfo = OzoneManagerProtocolProtos.KeyInfo.parseFrom((byte[])value);
                String insertKeyInfo = String.format(INSERT_KEY_INFO, keyInfo.getVolumeName(), keyInfo.getBucketName(), keyInfo.getKeyName(), keyInfo.getDataSize(), "EMPTY", "EMPTY");
                this.executeSQL(conn, insertKeyInfo);
                break;
            }
            default: {
                throw new IOException("Unknown key from om.db");
            }
        }
    }

    private KeyType getKeyType(String key) {
        if (key.startsWith("$")) {
            return KeyType.USER;
        }
        if (key.startsWith("/")) {
            return key.replaceFirst("/", "").contains("/") ? KeyType.BUCKET : KeyType.VOLUME;
        }
        return KeyType.KEY;
    }

    private void convertContainerDB(Path dbPath, Path outPath) throws Exception {
        LOG.info("Create tables for sql container db.");
        File dbFile = dbPath.toFile();
        try (MetadataStore dbStore = MetadataStoreBuilder.newBuilder().setConf((Configuration)this.conf).setDbFile(dbFile).build();
             Connection conn = this.connectDB(outPath.toString());){
            this.executeSQL(conn, CREATE_CONTAINER_INFO);
            dbStore.iterate(null, (key, value) -> {
                long containerID = Longs.fromByteArray((byte[])key);
                ContainerInfo containerInfo = null;
                containerInfo = ContainerInfo.fromProtobuf((HddsProtos.SCMContainerInfo)((HddsProtos.SCMContainerInfo)HddsProtos.SCMContainerInfo.PARSER.parseFrom(value)));
                Preconditions.checkNotNull((Object)containerInfo);
                try {
                    this.insertContainerDB(conn, containerInfo, containerID);
                    return true;
                }
                catch (SQLException e) {
                    throw new IOException(e);
                }
            });
        }
    }

    private void insertContainerDB(Connection conn, ContainerInfo containerInfo, long containerID) throws SQLException {
        LOG.info("Insert to sql container db, for container {}", (Object)containerID);
        String insertContainerInfo = String.format(INSERT_CONTAINER_INFO, containerID, containerInfo.getReplicationType(), containerInfo.getReplicationFactor(), containerInfo.getUsedBytes(), containerInfo.getAllocatedBytes(), containerInfo.getOwner(), containerInfo.getNumberOfKeys());
        this.executeSQL(conn, insertContainerInfo);
        LOG.info("Insertion completed.");
    }

    private void convertOpenContainerDB(Path dbPath, Path outPath) throws Exception {
        LOG.info("Create table for open container db.");
        File dbFile = dbPath.toFile();
        try (MetadataStore dbStore = MetadataStoreBuilder.newBuilder().setConf((Configuration)this.conf).setDbFile(dbFile).build();
             Connection conn = this.connectDB(outPath.toString());){
            this.executeSQL(conn, CREATE_OPEN_CONTAINER);
            dbStore.iterate(null, (key, value) -> {
                String containerName = DFSUtil.bytes2String((byte[])key);
                Long containerUsed = Long.parseLong(DFSUtil.bytes2String((byte[])value));
                String insertOpenContainer = String.format(INSERT_OPEN_CONTAINER, containerName, containerUsed);
                try {
                    this.executeSQL(conn, insertOpenContainer);
                    return true;
                }
                catch (SQLException e) {
                    throw new IOException(e);
                }
            });
        }
    }

    private CommandLine parseArgs(String[] argv) throws ParseException {
        return this.parser.parse(this.options, argv);
    }

    public static void main(String[] args) {
        SQLCLI shell = new SQLCLI(new OzoneConfiguration());
        int res = 0;
        try {
            ToolRunner.run((Tool)shell, (String[])args);
        }
        catch (Exception ex) {
            LOG.error(ex.toString());
            if (LOG.isDebugEnabled()) {
                LOG.debug("Command execution failed", (Throwable)ex);
            }
            res = 1;
        }
        System.exit(res);
    }

    private static enum KeyType {
        USER,
        VOLUME,
        BUCKET,
        KEY,
        UNKNOWN;

    }
}

