/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.ozone.om.OMMetadataManager;
import org.apache.hadoop.ozone.om.VolumeManager;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.helpers.OmVolumeArgs;
import org.apache.hadoop.ozone.protocol.proto.OzoneManagerProtocolProtos;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.utils.RocksDBStore;
import org.rocksdb.RocksDBException;
import org.rocksdb.WriteBatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VolumeManagerImpl
implements VolumeManager {
    private static final Logger LOG = LoggerFactory.getLogger(VolumeManagerImpl.class);
    private final OMMetadataManager metadataManager;
    private final int maxUserVolumeCount;

    public VolumeManagerImpl(OMMetadataManager metadataManager, OzoneConfiguration conf) throws IOException {
        this.metadataManager = metadataManager;
        this.maxUserVolumeCount = conf.getInt("ozone.om.user.max.volume", 1024);
    }

    private void addVolumeToOwnerList(String volume, String owner, WriteBatch batchOperation) throws RocksDBException, IOException {
        byte[] dbUserKey = this.metadataManager.getUserKey(owner);
        byte[] volumeList = this.metadataManager.getUserTable().get(dbUserKey);
        LinkedList<String> prevVolList = new LinkedList<String>();
        if (volumeList != null) {
            OzoneManagerProtocolProtos.VolumeList vlist = OzoneManagerProtocolProtos.VolumeList.parseFrom((byte[])volumeList);
            prevVolList.addAll(vlist.getVolumeNamesList());
        }
        if (prevVolList.size() >= this.maxUserVolumeCount) {
            LOG.debug("Too many volumes for user:{}", (Object)owner);
            throw new OMException(OMException.ResultCodes.FAILED_TOO_MANY_USER_VOLUMES);
        }
        prevVolList.add(volume);
        OzoneManagerProtocolProtos.VolumeList newVolList = OzoneManagerProtocolProtos.VolumeList.newBuilder().addAllVolumeNames(prevVolList).build();
        batchOperation.put(this.metadataManager.getUserTable().getHandle(), dbUserKey, newVolList.toByteArray());
    }

    private void delVolumeFromOwnerList(String volume, String owner, WriteBatch batch) throws RocksDBException, IOException {
        byte[] dbUserKey = this.metadataManager.getUserKey(owner);
        byte[] volumeList = this.metadataManager.getUserTable().get(dbUserKey);
        LinkedList prevVolList = new LinkedList();
        if (volumeList == null) {
            LOG.debug("volume:{} not found for user:{}");
            throw new OMException(OMException.ResultCodes.FAILED_USER_NOT_FOUND);
        }
        OzoneManagerProtocolProtos.VolumeList vlist = OzoneManagerProtocolProtos.VolumeList.parseFrom((byte[])volumeList);
        prevVolList.addAll(vlist.getVolumeNamesList());
        prevVolList.remove(volume);
        if (prevVolList.size() == 0) {
            batch.delete(this.metadataManager.getUserTable().getHandle(), dbUserKey);
        } else {
            OzoneManagerProtocolProtos.VolumeList newVolList = OzoneManagerProtocolProtos.VolumeList.newBuilder().addAllVolumeNames(prevVolList).build();
            batch.put(this.metadataManager.getUserTable().getHandle(), dbUserKey, newVolList.toByteArray());
        }
    }

    @Override
    public void createVolume(OmVolumeArgs args) throws IOException {
        Preconditions.checkNotNull((Object)args);
        this.metadataManager.getLock().acquireUserLock(args.getOwnerName());
        this.metadataManager.getLock().acquireVolumeLock(args.getVolume());
        try {
            byte[] dbVolumeKey = this.metadataManager.getVolumeKey(args.getVolume());
            byte[] volumeInfo = this.metadataManager.getVolumeTable().get(dbVolumeKey);
            if (volumeInfo != null) {
                LOG.debug("volume:{} already exists", (Object)args.getVolume());
                throw new OMException(OMException.ResultCodes.FAILED_VOLUME_ALREADY_EXISTS);
            }
            try (WriteBatch batch = new WriteBatch();){
                LinkedList<HddsProtos.KeyValue> metadataList = new LinkedList<HddsProtos.KeyValue>();
                for (Map.Entry entry : args.getKeyValueMap().entrySet()) {
                    metadataList.add(HddsProtos.KeyValue.newBuilder().setKey((String)entry.getKey()).setValue((String)entry.getValue()).build());
                }
                List aclList = args.getAclMap().ozoneAclGetProtobuf();
                OzoneManagerProtocolProtos.VolumeInfo newVolumeInfo = OzoneManagerProtocolProtos.VolumeInfo.newBuilder().setAdminName(args.getAdminName()).setOwnerName(args.getOwnerName()).setVolume(args.getVolume()).setQuotaInBytes(args.getQuotaInBytes()).addAllMetadata(metadataList).addAllVolumeAcls((Iterable)aclList).setCreationTime(Time.now()).build();
                batch.put(this.metadataManager.getVolumeTable().getHandle(), dbVolumeKey, newVolumeInfo.toByteArray());
                this.addVolumeToOwnerList(args.getVolume(), args.getOwnerName(), batch);
                this.metadataManager.getStore().write(batch);
            }
            LOG.debug("created volume:{} user:{}", (Object)args.getVolume(), (Object)args.getOwnerName());
        }
        catch (IOException | RocksDBException ex) {
            if (!(ex instanceof OMException)) {
                LOG.error("Volume creation failed for user:{} volume:{}", new Object[]{args.getOwnerName(), args.getVolume(), ex});
            }
            if (ex instanceof RocksDBException) {
                throw RocksDBStore.toIOException((String)"Volume creation failed.", (RocksDBException)((RocksDBException)ex));
            }
            throw (IOException)ex;
        }
        finally {
            this.metadataManager.getLock().releaseVolumeLock(args.getVolume());
            this.metadataManager.getLock().releaseUserLock(args.getOwnerName());
        }
    }

    @Override
    public void setOwner(String volume, String owner) throws IOException {
        Preconditions.checkNotNull((Object)volume);
        Preconditions.checkNotNull((Object)owner);
        this.metadataManager.getLock().acquireUserLock(owner);
        this.metadataManager.getLock().acquireVolumeLock(volume);
        try {
            byte[] dbVolumeKey = this.metadataManager.getVolumeKey(volume);
            byte[] volInfo = this.metadataManager.getVolumeTable().get(dbVolumeKey);
            if (volInfo == null) {
                LOG.debug("Changing volume ownership failed for user:{} volume:{}", (Object)owner, (Object)volume);
                throw new OMException(OMException.ResultCodes.FAILED_VOLUME_NOT_FOUND);
            }
            OzoneManagerProtocolProtos.VolumeInfo volumeInfo = OzoneManagerProtocolProtos.VolumeInfo.parseFrom((byte[])volInfo);
            OmVolumeArgs volumeArgs = OmVolumeArgs.getFromProtobuf((OzoneManagerProtocolProtos.VolumeInfo)volumeInfo);
            Preconditions.checkState((boolean)volume.equals(volumeInfo.getVolume()));
            try (WriteBatch batch = new WriteBatch();){
                this.delVolumeFromOwnerList(volume, volumeArgs.getOwnerName(), batch);
                this.addVolumeToOwnerList(volume, owner, batch);
                OmVolumeArgs newVolumeArgs = OmVolumeArgs.newBuilder().setVolume(volumeArgs.getVolume()).setAdminName(volumeArgs.getAdminName()).setOwnerName(owner).setQuotaInBytes(volumeArgs.getQuotaInBytes()).setCreationTime(volumeArgs.getCreationTime()).build();
                OzoneManagerProtocolProtos.VolumeInfo newVolumeInfo = newVolumeArgs.getProtobuf();
                batch.put(this.metadataManager.getVolumeTable().getHandle(), dbVolumeKey, newVolumeInfo.toByteArray());
                this.metadataManager.getStore().write(batch);
            }
        }
        catch (IOException | RocksDBException ex) {
            if (!(ex instanceof OMException)) {
                LOG.error("Changing volume ownership failed for user:{} volume:{}", new Object[]{owner, volume, ex});
            }
            if (ex instanceof RocksDBException) {
                throw RocksDBStore.toIOException((String)"Volume creation failed.", (RocksDBException)((RocksDBException)ex));
            }
            throw (IOException)ex;
        }
        finally {
            this.metadataManager.getLock().releaseVolumeLock(volume);
            this.metadataManager.getLock().releaseUserLock(owner);
        }
    }

    @Override
    public void setQuota(String volume, long quota) throws IOException {
        Preconditions.checkNotNull((Object)volume);
        this.metadataManager.getLock().acquireVolumeLock(volume);
        try {
            byte[] dbVolumeKey = this.metadataManager.getVolumeKey(volume);
            byte[] volInfo = this.metadataManager.getVolumeTable().get(dbVolumeKey);
            if (volInfo == null) {
                LOG.debug("volume:{} does not exist", (Object)volume);
                throw new OMException(OMException.ResultCodes.FAILED_VOLUME_NOT_FOUND);
            }
            OzoneManagerProtocolProtos.VolumeInfo volumeInfo = OzoneManagerProtocolProtos.VolumeInfo.parseFrom((byte[])volInfo);
            OmVolumeArgs volumeArgs = OmVolumeArgs.getFromProtobuf((OzoneManagerProtocolProtos.VolumeInfo)volumeInfo);
            Preconditions.checkState((boolean)volume.equals(volumeInfo.getVolume()));
            OmVolumeArgs newVolumeArgs = OmVolumeArgs.newBuilder().setVolume(volumeArgs.getVolume()).setAdminName(volumeArgs.getAdminName()).setOwnerName(volumeArgs.getOwnerName()).setQuotaInBytes(quota).setCreationTime(volumeArgs.getCreationTime()).build();
            OzoneManagerProtocolProtos.VolumeInfo newVolumeInfo = newVolumeArgs.getProtobuf();
            this.metadataManager.getVolumeTable().put(dbVolumeKey, newVolumeInfo.toByteArray());
        }
        catch (IOException ex) {
            if (!(ex instanceof OMException)) {
                LOG.error("Changing volume quota failed for volume:{} quota:{}", new Object[]{volume, quota, ex});
            }
            throw ex;
        }
        finally {
            this.metadataManager.getLock().releaseVolumeLock(volume);
        }
    }

    @Override
    public OmVolumeArgs getVolumeInfo(String volume) throws IOException {
        Preconditions.checkNotNull((Object)volume);
        this.metadataManager.getLock().acquireVolumeLock(volume);
        try {
            byte[] dbVolumeKey = this.metadataManager.getVolumeKey(volume);
            byte[] volInfo = this.metadataManager.getVolumeTable().get(dbVolumeKey);
            if (volInfo == null) {
                LOG.debug("volume:{} does not exist", (Object)volume);
                throw new OMException(OMException.ResultCodes.FAILED_VOLUME_NOT_FOUND);
            }
            OzoneManagerProtocolProtos.VolumeInfo volumeInfo = OzoneManagerProtocolProtos.VolumeInfo.parseFrom((byte[])volInfo);
            OmVolumeArgs volumeArgs = OmVolumeArgs.getFromProtobuf((OzoneManagerProtocolProtos.VolumeInfo)volumeInfo);
            Preconditions.checkState((boolean)volume.equals(volumeInfo.getVolume()));
            OmVolumeArgs omVolumeArgs = volumeArgs;
            return omVolumeArgs;
        }
        catch (IOException ex) {
            if (!(ex instanceof OMException)) {
                LOG.warn("Info volume failed for volume:{}", (Object)volume, (Object)ex);
            }
            throw ex;
        }
        finally {
            this.metadataManager.getLock().releaseVolumeLock(volume);
        }
    }

    @Override
    public void deleteVolume(String volume) throws IOException {
        String owner;
        Preconditions.checkNotNull((Object)volume);
        this.metadataManager.getLock().acquireVolumeLock(volume);
        try {
            owner = this.getVolumeInfo(volume).getOwnerName();
        }
        finally {
            this.metadataManager.getLock().releaseVolumeLock(volume);
        }
        this.metadataManager.getLock().acquireUserLock(owner);
        this.metadataManager.getLock().acquireVolumeLock(volume);
        try {
            byte[] dbVolumeKey = this.metadataManager.getVolumeKey(volume);
            byte[] volInfo = this.metadataManager.getVolumeTable().get(dbVolumeKey);
            if (volInfo == null) {
                LOG.debug("volume:{} does not exist", (Object)volume);
                throw new OMException(OMException.ResultCodes.FAILED_VOLUME_NOT_FOUND);
            }
            if (!this.metadataManager.isVolumeEmpty(volume)) {
                LOG.debug("volume:{} is not empty", (Object)volume);
                throw new OMException(OMException.ResultCodes.FAILED_VOLUME_NOT_EMPTY);
            }
            OzoneManagerProtocolProtos.VolumeInfo volumeInfo = OzoneManagerProtocolProtos.VolumeInfo.parseFrom((byte[])volInfo);
            Preconditions.checkState((boolean)volume.equals(volumeInfo.getVolume()));
            try (WriteBatch batch = new WriteBatch();){
                this.delVolumeFromOwnerList(volume, volumeInfo.getOwnerName(), batch);
                batch.delete(this.metadataManager.getVolumeTable().getHandle(), dbVolumeKey);
                this.metadataManager.getStore().write(batch);
            }
        }
        catch (IOException | RocksDBException ex) {
            if (!(ex instanceof OMException)) {
                LOG.error("Delete volume failed for volume:{}", (Object)volume, (Object)ex);
            }
            if (ex instanceof RocksDBException) {
                throw RocksDBStore.toIOException((String)"Volume creation failed.", (RocksDBException)((RocksDBException)ex));
            }
            throw (IOException)ex;
        }
        finally {
            this.metadataManager.getLock().releaseVolumeLock(volume);
            this.metadataManager.getLock().releaseUserLock(owner);
        }
    }

    @Override
    public boolean checkVolumeAccess(String volume, OzoneManagerProtocolProtos.OzoneAclInfo userAcl) throws IOException {
        Preconditions.checkNotNull((Object)volume);
        Preconditions.checkNotNull((Object)userAcl);
        this.metadataManager.getLock().acquireVolumeLock(volume);
        try {
            byte[] dbVolumeKey = this.metadataManager.getVolumeKey(volume);
            byte[] volInfo = this.metadataManager.getVolumeTable().get(dbVolumeKey);
            if (volInfo == null) {
                LOG.debug("volume:{} does not exist", (Object)volume);
                throw new OMException(OMException.ResultCodes.FAILED_VOLUME_NOT_FOUND);
            }
            OzoneManagerProtocolProtos.VolumeInfo volumeInfo = OzoneManagerProtocolProtos.VolumeInfo.parseFrom((byte[])volInfo);
            OmVolumeArgs volumeArgs = OmVolumeArgs.getFromProtobuf((OzoneManagerProtocolProtos.VolumeInfo)volumeInfo);
            Preconditions.checkState((boolean)volume.equals(volumeInfo.getVolume()));
            boolean bl = volumeArgs.getAclMap().hasAccess(userAcl);
            return bl;
        }
        catch (IOException ex) {
            if (!(ex instanceof OMException)) {
                LOG.error("Check volume access failed for volume:{} user:{} rights:{}", new Object[]{volume, userAcl.getName(), userAcl.getRights(), ex});
            }
            throw ex;
        }
        finally {
            this.metadataManager.getLock().releaseVolumeLock(volume);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<OmVolumeArgs> listVolumes(String userName, String prefix, String startKey, int maxKeys) throws IOException {
        this.metadataManager.getLock().acquireUserLock(userName);
        try {
            List<OmVolumeArgs> list = this.metadataManager.listVolumes(userName, prefix, startKey, maxKeys);
            return list;
        }
        finally {
            this.metadataManager.getLock().releaseUserLock(userName);
        }
    }
}

