/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.server.ServerUtils;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.ozone.common.BlockGroup;
import org.apache.hadoop.ozone.om.OMMetadataManager;
import org.apache.hadoop.ozone.om.OzoneManagerLock;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.helpers.OmBucketInfo;
import org.apache.hadoop.ozone.om.helpers.OmKeyInfo;
import org.apache.hadoop.ozone.om.helpers.OmKeyLocationInfoGroup;
import org.apache.hadoop.ozone.om.helpers.OmVolumeArgs;
import org.apache.hadoop.ozone.protocol.proto.OzoneManagerProtocolProtos;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.utils.db.DBStore;
import org.apache.hadoop.utils.db.DBStoreBuilder;
import org.apache.hadoop.utils.db.Table;
import org.apache.hadoop.utils.db.TableIterator;
import org.eclipse.jetty.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OmMetadataManagerImpl
implements OMMetadataManager {
    private static final Logger LOG = LoggerFactory.getLogger(OmMetadataManagerImpl.class);
    private static final String USER_TABLE = "userTable";
    private static final String VOLUME_TABLE = "volumeTable";
    private static final String BUCKET_TABLE = "bucketTable";
    private static final String KEY_TABLE = "keyTable";
    private static final String DELETED_TABLE = "deletedTable";
    private static final String OPEN_KEY_TABLE = "openKeyTable";
    private final DBStore store;
    private final OzoneManagerLock lock;
    private final long openKeyExpireThresholdMS;
    private final Table userTable;
    private final Table volumeTable;
    private final Table bucketTable;
    private final Table keyTable;
    private final Table deletedTable;
    private final Table openKeyTable;

    public OmMetadataManagerImpl(OzoneConfiguration conf) throws IOException {
        File metaDir = ServerUtils.getOzoneMetaDirPath((Configuration)conf);
        this.lock = new OzoneManagerLock((Configuration)conf);
        this.openKeyExpireThresholdMS = 1000 * conf.getInt("ozone.open.key.expire.threshold", 86400);
        this.store = DBStoreBuilder.newBuilder((Configuration)conf).setName("om.db").setPath(Paths.get(metaDir.getPath(), new String[0])).addTable(USER_TABLE).addTable(VOLUME_TABLE).addTable(BUCKET_TABLE).addTable(KEY_TABLE).addTable(DELETED_TABLE).addTable(OPEN_KEY_TABLE).build();
        this.userTable = this.store.getTable(USER_TABLE);
        this.checkTableStatus(this.userTable, USER_TABLE);
        this.volumeTable = this.store.getTable(VOLUME_TABLE);
        this.checkTableStatus(this.volumeTable, VOLUME_TABLE);
        this.bucketTable = this.store.getTable(BUCKET_TABLE);
        this.checkTableStatus(this.bucketTable, BUCKET_TABLE);
        this.keyTable = this.store.getTable(KEY_TABLE);
        this.checkTableStatus(this.keyTable, KEY_TABLE);
        this.deletedTable = this.store.getTable(DELETED_TABLE);
        this.checkTableStatus(this.deletedTable, DELETED_TABLE);
        this.openKeyTable = this.store.getTable(OPEN_KEY_TABLE);
        this.checkTableStatus(this.openKeyTable, OPEN_KEY_TABLE);
    }

    @Override
    public Table getUserTable() {
        return this.userTable;
    }

    @Override
    public Table getVolumeTable() {
        return this.volumeTable;
    }

    @Override
    public Table getBucketTable() {
        return this.bucketTable;
    }

    @Override
    public Table getKeyTable() {
        return this.keyTable;
    }

    @Override
    public Table getDeletedTable() {
        return this.deletedTable;
    }

    @Override
    public Table getOpenKeyTable() {
        return this.openKeyTable;
    }

    private void checkTableStatus(Table table, String name) throws IOException {
        String logMessage = "Unable to get a reference to %s table. Cannot continue.";
        String errMsg = "Inconsistent DB state, Table - %s. Please check the logsfor more info.";
        if (table == null) {
            LOG.error(String.format(logMessage, name));
            throw new IOException(String.format(errMsg, name));
        }
    }

    @Override
    public void start() {
    }

    @Override
    public void stop() throws Exception {
        if (this.store != null) {
            this.store.close();
        }
    }

    @Override
    @VisibleForTesting
    public DBStore getStore() {
        return this.store;
    }

    @Override
    public byte[] getVolumeKey(String volume) {
        return DFSUtil.string2Bytes((String)("/" + volume));
    }

    @Override
    public byte[] getUserKey(String user) {
        return DFSUtil.string2Bytes((String)user);
    }

    @Override
    public byte[] getBucketKey(String volume, String bucket) {
        StringBuilder builder = new StringBuilder().append("/").append(volume);
        if (StringUtils.isNotBlank((CharSequence)bucket)) {
            builder.append("/").append(bucket);
        }
        return DFSUtil.string2Bytes((String)builder.toString());
    }

    @Override
    public byte[] getOzoneKeyBytes(String volume, String bucket, String key) {
        StringBuilder builder = new StringBuilder().append("/").append(volume);
        builder.append("/").append(bucket);
        if (StringUtil.isNotBlank((String)key)) {
            builder.append("/").append(key);
        }
        return DFSUtil.string2Bytes((String)builder.toString());
    }

    @Override
    public byte[] getOpenKeyBytes(String volume, String bucket, String key, long id) {
        String openKey = "/" + volume + "/" + bucket + "/" + key + "/" + id;
        return DFSUtil.string2Bytes((String)openKey);
    }

    @Override
    public OzoneManagerLock getLock() {
        return this.lock;
    }

    private boolean startsWith(byte[] firstArray, byte[] secondArray) {
        if (firstArray == null) {
            return secondArray == null;
        }
        if (secondArray != null) {
            if (secondArray.length > firstArray.length) {
                return false;
            }
            for (int ndx = 0; ndx < secondArray.length; ++ndx) {
                if (firstArray[ndx] == secondArray[ndx]) continue;
                return false;
            }
            return true;
        }
        return false;
    }

    @Override
    public boolean isVolumeEmpty(String volume) throws IOException {
        byte[] volumePrefix = this.getVolumeKey(volume + "/");
        try (TableIterator bucketIter = this.bucketTable.iterator();){
            Table.KeyValue kv = (Table.KeyValue)bucketIter.seek(volumePrefix);
            if (kv != null && this.startsWith(kv.getKey(), volumePrefix)) {
                boolean bl = false;
                return bl;
            }
        }
        return true;
    }

    @Override
    public boolean isBucketEmpty(String volume, String bucket) throws IOException {
        byte[] keyPrefix = this.getBucketKey(volume, bucket + "/");
        try (TableIterator keyIter = this.keyTable.iterator();){
            Table.KeyValue kv = (Table.KeyValue)keyIter.seek(keyPrefix);
            if (kv != null && this.startsWith(kv.getKey(), keyPrefix)) {
                boolean bl = false;
                return bl;
            }
        }
        return true;
    }

    @Override
    public List<OmBucketInfo> listBuckets(String volumeName, String startBucket, String bucketPrefix, int maxNumOfBuckets) throws IOException {
        byte[] startKey;
        ArrayList<OmBucketInfo> result = new ArrayList<OmBucketInfo>();
        if (Strings.isNullOrEmpty((String)volumeName)) {
            throw new OMException("Volume name is required.", OMException.ResultCodes.FAILED_VOLUME_NOT_FOUND);
        }
        byte[] volumeNameBytes = this.getVolumeKey(volumeName);
        if (this.volumeTable.get(volumeNameBytes) == null) {
            throw new OMException("Volume " + volumeName + " not found.", OMException.ResultCodes.FAILED_VOLUME_NOT_FOUND);
        }
        boolean skipStartKey = false;
        if (StringUtil.isNotBlank((String)startBucket)) {
            startKey = this.getBucketKey(volumeName, startBucket);
            skipStartKey = true;
        } else {
            startKey = this.getBucketKey(volumeName, bucketPrefix);
        }
        byte[] seekPrefix = StringUtil.isNotBlank((String)bucketPrefix) ? this.getBucketKey(volumeName, bucketPrefix) : this.getVolumeKey(volumeName + "/");
        int currentCount = 0;
        try (TableIterator bucketIter = this.bucketTable.iterator();){
            Table.KeyValue kv = (Table.KeyValue)bucketIter.seek(startKey);
            while (currentCount < maxNumOfBuckets && bucketIter.hasNext()) {
                kv = (Table.KeyValue)bucketIter.next();
                if (kv != null && skipStartKey && Arrays.equals(kv.getKey(), startKey)) continue;
                if (kv != null && this.startsWith(kv.getKey(), seekPrefix)) {
                    result.add(OmBucketInfo.getFromProtobuf((OzoneManagerProtocolProtos.BucketInfo)OzoneManagerProtocolProtos.BucketInfo.parseFrom((byte[])kv.getValue())));
                    ++currentCount;
                    continue;
                }
                break;
            }
        }
        return result;
    }

    @Override
    public List<OmKeyInfo> listKeys(String volumeName, String bucketName, String startKey, String keyPrefix, int maxKeys) throws IOException {
        byte[] seekKey;
        ArrayList<OmKeyInfo> result = new ArrayList<OmKeyInfo>();
        if (Strings.isNullOrEmpty((String)volumeName)) {
            throw new OMException("Volume name is required.", OMException.ResultCodes.FAILED_VOLUME_NOT_FOUND);
        }
        if (Strings.isNullOrEmpty((String)bucketName)) {
            throw new OMException("Bucket name is required.", OMException.ResultCodes.FAILED_BUCKET_NOT_FOUND);
        }
        byte[] bucketNameBytes = this.getBucketKey(volumeName, bucketName);
        if (this.getBucketTable().get(bucketNameBytes) == null) {
            throw new OMException("Bucket " + bucketName + " not found.", OMException.ResultCodes.FAILED_BUCKET_NOT_FOUND);
        }
        boolean skipStartKey = false;
        if (StringUtil.isNotBlank((String)startKey)) {
            seekKey = this.getOzoneKeyBytes(volumeName, bucketName, startKey);
            skipStartKey = true;
        } else {
            seekKey = this.getOzoneKeyBytes(volumeName, bucketName, keyPrefix);
        }
        byte[] seekPrefix = StringUtil.isNotBlank((String)keyPrefix) ? this.getOzoneKeyBytes(volumeName, bucketName, keyPrefix) : this.getBucketKey(volumeName, bucketName + "/");
        int currentCount = 0;
        try (TableIterator keyIter = this.getKeyTable().iterator();){
            Table.KeyValue kv = (Table.KeyValue)keyIter.seek(seekKey);
            while (currentCount < maxKeys && keyIter.hasNext()) {
                kv = (Table.KeyValue)keyIter.next();
                if (kv != null && skipStartKey && Arrays.equals(kv.getKey(), seekKey)) continue;
                if (kv != null && this.startsWith(kv.getKey(), seekPrefix)) {
                    result.add(OmKeyInfo.getFromProtobuf((OzoneManagerProtocolProtos.KeyInfo)OzoneManagerProtocolProtos.KeyInfo.parseFrom((byte[])kv.getValue())));
                    ++currentCount;
                    continue;
                }
                break;
            }
        }
        return result;
    }

    @Override
    public List<OmVolumeArgs> listVolumes(String userName, String prefix, String startKey, int maxKeys) throws IOException {
        ArrayList result = Lists.newArrayList();
        if (StringUtil.isBlank((String)userName)) {
            throw new OMException("User name is required to list Volumes.", OMException.ResultCodes.FAILED_USER_NOT_FOUND);
        }
        OzoneManagerProtocolProtos.VolumeList volumes = this.getVolumesByUser(userName);
        if (volumes == null || volumes.getVolumeNamesCount() == 0) {
            return result;
        }
        boolean startKeyFound = Strings.isNullOrEmpty((String)startKey);
        for (String volumeName : volumes.getVolumeNamesList()) {
            if (!Strings.isNullOrEmpty((String)prefix) && !volumeName.startsWith(prefix)) continue;
            if (!startKeyFound && volumeName.equals(startKey)) {
                startKeyFound = true;
                continue;
            }
            if (!startKeyFound || result.size() >= maxKeys) continue;
            byte[] volumeInfo = this.getVolumeTable().get(this.getVolumeKey(volumeName));
            if (volumeInfo == null) {
                throw new OMException("Volume info not found for " + volumeName, OMException.ResultCodes.FAILED_VOLUME_NOT_FOUND);
            }
            OzoneManagerProtocolProtos.VolumeInfo info = OzoneManagerProtocolProtos.VolumeInfo.parseFrom((byte[])volumeInfo);
            OmVolumeArgs volumeArgs = OmVolumeArgs.getFromProtobuf((OzoneManagerProtocolProtos.VolumeInfo)info);
            result.add(volumeArgs);
        }
        return result;
    }

    private OzoneManagerProtocolProtos.VolumeList getVolumesByUser(String userName) throws OMException {
        return this.getVolumesByUser(this.getUserKey(userName));
    }

    private OzoneManagerProtocolProtos.VolumeList getVolumesByUser(byte[] userNameKey) throws OMException {
        OzoneManagerProtocolProtos.VolumeList volumes = null;
        try {
            byte[] volumesInBytes = this.getUserTable().get(userNameKey);
            if (volumesInBytes == null) {
                return OzoneManagerProtocolProtos.VolumeList.newBuilder().build();
            }
            volumes = OzoneManagerProtocolProtos.VolumeList.parseFrom((byte[])volumesInBytes);
        }
        catch (IOException e) {
            throw new OMException("Unable to get volumes info by the given user, metadata might be corrupted", e, OMException.ResultCodes.FAILED_METADATA_ERROR);
        }
        return volumes;
    }

    @Override
    public List<BlockGroup> getPendingDeletionKeys(int keyCount) throws IOException {
        ArrayList keyBlocksList = Lists.newArrayList();
        try (TableIterator keyIter = this.getDeletedTable().iterator();){
            int currentCount = 0;
            while (keyIter.hasNext() && currentCount < keyCount) {
                Table.KeyValue kv = (Table.KeyValue)keyIter.next();
                if (kv == null) continue;
                OmKeyInfo info = OmKeyInfo.getFromProtobuf((OzoneManagerProtocolProtos.KeyInfo)OzoneManagerProtocolProtos.KeyInfo.parseFrom((byte[])kv.getValue()));
                OmKeyLocationInfoGroup latest = info.getLatestVersionLocations();
                List item = latest.getLocationList().stream().map(b -> new BlockID(b.getContainerID(), b.getLocalID())).collect(Collectors.toList());
                BlockGroup keyBlocks = BlockGroup.newBuilder().setKeyName(DFSUtil.bytes2String((byte[])kv.getKey())).addAllBlockIDs(item).build();
                keyBlocksList.add(keyBlocks);
                ++currentCount;
            }
        }
        return keyBlocksList;
    }

    @Override
    public List<BlockGroup> getExpiredOpenKeys() throws IOException {
        ArrayList keyBlocksList = Lists.newArrayList();
        long now = Time.now();
        List<Map.Entry> rangeResult = Collections.emptyList();
        for (Map.Entry entry : rangeResult) {
            OmKeyInfo info = OmKeyInfo.getFromProtobuf((OzoneManagerProtocolProtos.KeyInfo)OzoneManagerProtocolProtos.KeyInfo.parseFrom((byte[])((byte[])entry.getValue())));
            long lastModify = info.getModificationTime();
            if (now - lastModify < this.openKeyExpireThresholdMS) continue;
            List item = info.getLatestVersionLocations().getBlocksLatestVersionOnly().stream().map(b -> new BlockID(b.getContainerID(), b.getLocalID())).collect(Collectors.toList());
            BlockGroup keyBlocks = BlockGroup.newBuilder().setKeyName(DFSUtil.bytes2String((byte[])((byte[])entry.getKey()))).addAllBlockIDs(item).build();
            keyBlocksList.add(keyBlocks);
        }
        return keyBlocksList;
    }
}

