/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.grpc.client;

import java.io.Closeable;
import java.io.IOException;
import java.util.Arrays;
import java.util.Deque;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.ratis.client.impl.ClientProtoUtils;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.grpc.GrpcConfigKeys;
import org.apache.ratis.grpc.RaftGrpcUtil;
import org.apache.ratis.grpc.client.RaftClientProtocolProxy;
import org.apache.ratis.protocol.ClientId;
import org.apache.ratis.protocol.NotLeaderException;
import org.apache.ratis.protocol.RaftClientReply;
import org.apache.ratis.protocol.RaftGroup;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.protocol.RaftPeer;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.shaded.com.google.protobuf.ByteString;
import org.apache.ratis.shaded.proto.RaftProtos;
import org.apache.ratis.util.CollectionUtils;
import org.apache.ratis.util.Daemon;
import org.apache.ratis.util.PeerProxyMap;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.SizeInBytes;
import org.apache.ratis.util.TimeDuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AppendStreamer
implements Closeable {
    public static final Logger LOG = LoggerFactory.getLogger(AppendStreamer.class);
    private final Deque<RaftProtos.RaftClientRequestProto> dataQueue;
    private final Deque<RaftProtos.RaftClientRequestProto> ackQueue;
    private final int maxPendingNum;
    private final SizeInBytes maxMessageSize;
    private final PeerProxyMap<RaftClientProtocolProxy> proxyMap;
    private final Map<RaftPeerId, RaftPeer> peers;
    private RaftPeerId leaderId;
    private volatile RaftClientProtocolProxy leaderProxy;
    private final ClientId clientId;
    private volatile RunningState running = RunningState.RUNNING;
    private final ExceptionAndRetry exceptionAndRetry;
    private final Sender senderThread;
    private final RaftGroupId groupId;

    AppendStreamer(RaftProperties prop, RaftGroup group, RaftPeerId leaderId, ClientId clientId) {
        this.clientId = clientId;
        this.maxPendingNum = GrpcConfigKeys.OutputStream.outstandingAppendsMax(prop);
        this.maxMessageSize = GrpcConfigKeys.messageSizeMax(prop, arg_0 -> ((Logger)LOG).debug(arg_0));
        this.dataQueue = new ConcurrentLinkedDeque<RaftProtos.RaftClientRequestProto>();
        this.ackQueue = new ConcurrentLinkedDeque<RaftProtos.RaftClientRequestProto>();
        this.exceptionAndRetry = new ExceptionAndRetry(prop);
        this.groupId = group.getGroupId();
        this.peers = group.getPeers().stream().collect(Collectors.toMap(RaftPeer::getId, Function.identity()));
        this.proxyMap = new PeerProxyMap(clientId.toString(), raftPeer -> new RaftClientProtocolProxy(clientId, (RaftPeer)raftPeer, x$0 -> new ResponseHandler((RaftPeer)x$0), prop));
        this.proxyMap.addPeers((Iterable)group.getPeers());
        this.refreshLeaderProxy(leaderId, null);
        this.senderThread = new Sender();
        this.senderThread.setName(this.toString() + "-sender");
        this.senderThread.start();
    }

    private synchronized void refreshLeaderProxy(RaftPeerId suggested, RaftPeerId oldLeader) {
        if (suggested != null) {
            this.leaderId = suggested;
        } else if (oldLeader == null) {
            this.leaderId = this.peers.keySet().iterator().next();
        } else {
            this.leaderId = (RaftPeerId)CollectionUtils.random((Object)oldLeader, this.peers.keySet());
            if (this.leaderId == null) {
                this.leaderId = oldLeader;
            }
        }
        LOG.debug("{} switches leader from {} to {}. suggested leader: {}", new Object[]{this, oldLeader, this.leaderId, suggested});
        if (this.leaderProxy != null) {
            this.leaderProxy.closeCurrentSession();
        }
        try {
            this.leaderProxy = (RaftClientProtocolProxy)this.proxyMap.getProxy(this.leaderId);
        }
        catch (IOException e) {
            LOG.error("Should not hit IOException here", (Throwable)e);
            this.refreshLeader(null, this.leaderId);
        }
    }

    private boolean isRunning() {
        return this.running == RunningState.RUNNING || this.running == RunningState.LOOK_FOR_LEADER;
    }

    private void checkState() throws IOException {
        if (!this.isRunning()) {
            this.throwException("The AppendStreamer has been closed");
        }
    }

    synchronized void write(ByteString content, long seqNum) throws IOException {
        this.checkState();
        while (this.isRunning() && this.dataQueue.size() >= this.maxPendingNum) {
            try {
                this.wait();
            }
            catch (InterruptedException interruptedException) {}
        }
        if (this.isRunning()) {
            RaftProtos.RaftClientRequestProto request = ClientProtoUtils.toRaftClientRequestProto((ClientId)this.clientId, (RaftPeerId)this.leaderId, (RaftGroupId)this.groupId, (long)seqNum, (long)seqNum, (ByteString)content);
            if (request.getSerializedSize() > this.maxMessageSize.getSizeInt()) {
                throw new IOException("msg size:" + request.getSerializedSize() + " exceeds maximum:" + this.maxMessageSize.getSizeInt());
            }
            this.dataQueue.offer(request);
            this.notifyAll();
        } else {
            this.throwException(this + " got closed.");
        }
    }

    synchronized void flush() throws IOException {
        this.checkState();
        if (this.dataQueue.isEmpty() && this.ackQueue.isEmpty()) {
            return;
        }
        while (!(!this.isRunning() || this.dataQueue.isEmpty() && this.ackQueue.isEmpty())) {
            try {
                this.wait();
            }
            catch (InterruptedException interruptedException) {}
        }
        if (!(this.isRunning() || this.dataQueue.isEmpty() && this.ackQueue.isEmpty())) {
            this.throwException(this + " got closed before finishing flush");
        }
    }

    @Override
    public void close() throws IOException {
        if (!this.isRunning()) {
            return;
        }
        this.flush();
        this.running = RunningState.CLOSED;
        this.senderThread.interrupt();
        try {
            this.senderThread.join();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        this.proxyMap.close();
    }

    public String toString() {
        return this.getClass().getSimpleName() + "-" + this.clientId;
    }

    private void throwException(String msg) throws IOException {
        if (this.running == RunningState.ERROR) {
            throw this.exceptionAndRetry.getCombinedException();
        }
        throw new IOException(msg);
    }

    private void handleNotLeader(NotLeaderException nle, RaftPeerId oldLeader) {
        Preconditions.assertTrue((boolean)Thread.holdsLock(this));
        this.refreshPeers(nle.getPeers());
        this.refreshLeader(nle.getSuggestedLeader().getId(), oldLeader);
    }

    private void handleError(Throwable t, ResponseHandler handler) {
        Preconditions.assertTrue((boolean)Thread.holdsLock(this));
        IOException e = RaftGrpcUtil.unwrapIOException(t);
        this.exceptionAndRetry.addException(handler.targetId, e);
        LOG.debug("{} got error: {}. Total retry times {}, max retry times {}.", new Object[]{handler, e, this.exceptionAndRetry.retryTimes.get(), this.exceptionAndRetry.maxRetryTimes});
        this.leaderProxy.onError();
        if (this.exceptionAndRetry.shouldRetry()) {
            this.refreshLeader(null, this.leaderId);
        } else {
            this.running = RunningState.ERROR;
        }
    }

    private void refreshLeader(RaftPeerId suggestedLeader, RaftPeerId oldLeader) {
        this.running = RunningState.LOOK_FOR_LEADER;
        this.refreshLeaderProxy(suggestedLeader, oldLeader);
        this.reQueuePendingRequests(this.leaderId);
        RaftProtos.RaftClientRequestProto request = Objects.requireNonNull(this.dataQueue.poll());
        this.ackQueue.offer(request);
        try {
            this.exceptionAndRetry.retryInterval.sleep();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        this.leaderProxy.onNext(request);
    }

    private void reQueuePendingRequests(RaftPeerId newLeader) {
        if (this.isRunning()) {
            while (!this.ackQueue.isEmpty()) {
                RaftProtos.RaftClientRequestProto oldRequest = this.ackQueue.pollLast();
                RaftProtos.RaftRpcRequestProto.Builder newRpc = RaftProtos.RaftRpcRequestProto.newBuilder((RaftProtos.RaftRpcRequestProto)oldRequest.getRpcRequest()).setReplyId(newLeader.toByteString());
                RaftProtos.RaftClientRequestProto newRequest = RaftProtos.RaftClientRequestProto.newBuilder((RaftProtos.RaftClientRequestProto)oldRequest).setRpcRequest(newRpc).build();
                this.dataQueue.offerFirst(newRequest);
            }
        }
    }

    private void refreshPeers(RaftPeer[] newPeers) {
        if (newPeers != null && newPeers.length > 0) {
            Arrays.stream(newPeers).forEach(peer -> {
                this.peers.putIfAbsent(peer.getId(), (RaftPeer)peer);
                this.proxyMap.computeIfAbsent(peer);
            });
            LOG.debug("refreshed peers: {}", this.peers);
        }
    }

    private class ResponseHandler
    implements RaftClientProtocolProxy.CloseableStreamObserver {
        private final RaftPeerId targetId;
        private volatile boolean active = true;

        ResponseHandler(RaftPeer target) {
            this.targetId = target.getId();
        }

        public String toString() {
            return AppendStreamer.this + "-ResponseHandler-" + this.targetId;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void onNext(RaftProtos.RaftClientReplyProto reply) {
            if (!this.active) {
                return;
            }
            AppendStreamer appendStreamer = AppendStreamer.this;
            synchronized (appendStreamer) {
                RaftProtos.RaftClientRequestProto pending = (RaftProtos.RaftClientRequestProto)Objects.requireNonNull(AppendStreamer.this.ackQueue.peek());
                if (reply.getRpcReply().getSuccess()) {
                    Preconditions.assertTrue((pending.getRpcRequest().getCallId() == reply.getRpcReply().getCallId() ? 1 : 0) != 0, () -> "pending=" + ClientProtoUtils.toString((RaftProtos.RaftClientRequestProto)pending) + " but reply=" + ClientProtoUtils.toString((RaftProtos.RaftClientReplyProto)reply));
                    AppendStreamer.this.ackQueue.poll();
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("{} received success ack for {}", (Object)this, (Object)ClientProtoUtils.toString((RaftProtos.RaftClientRequestProto)pending));
                    }
                    if (AppendStreamer.this.running == RunningState.LOOK_FOR_LEADER) {
                        AppendStreamer.this.running = RunningState.RUNNING;
                    }
                } else {
                    RaftClientReply r = ClientProtoUtils.toRaftClientReply((RaftProtos.RaftClientReplyProto)reply);
                    NotLeaderException nle = r.getNotLeaderException();
                    if (nle != null) {
                        LOG.debug("{} received a NotLeaderException from {}", (Object)this, (Object)r.getServerId());
                        AppendStreamer.this.handleNotLeader(nle, this.targetId);
                    }
                }
                AppendStreamer.this.notifyAll();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void onError(Throwable t) {
            LOG.warn(this + " onError", t);
            if (this.active) {
                AppendStreamer appendStreamer = AppendStreamer.this;
                synchronized (appendStreamer) {
                    AppendStreamer.this.handleError(t, this);
                    AppendStreamer.this.notifyAll();
                }
            }
        }

        public void onCompleted() {
            LOG.info("{} onCompleted, pending requests #: {}", (Object)this, (Object)AppendStreamer.this.ackQueue.size());
        }

        @Override
        public void close() throws IOException {
            this.active = false;
        }
    }

    private class Sender
    extends Daemon {
        private Sender() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void run() {
            while (AppendStreamer.this.isRunning()) {
                AppendStreamer appendStreamer = AppendStreamer.this;
                synchronized (appendStreamer) {
                    while (AppendStreamer.this.isRunning() && this.shouldWait()) {
                        try {
                            AppendStreamer.this.wait();
                        }
                        catch (InterruptedException interruptedException) {}
                    }
                    if (AppendStreamer.this.running == RunningState.RUNNING) {
                        Preconditions.assertTrue((!AppendStreamer.this.dataQueue.isEmpty() ? 1 : 0) != 0, (Object)"dataQueue is empty");
                        RaftProtos.RaftClientRequestProto next = (RaftProtos.RaftClientRequestProto)AppendStreamer.this.dataQueue.poll();
                        AppendStreamer.this.leaderProxy.onNext(next);
                        AppendStreamer.this.ackQueue.offer(next);
                    }
                }
            }
        }

        private boolean shouldWait() {
            return AppendStreamer.this.dataQueue.isEmpty() || AppendStreamer.this.ackQueue.size() >= AppendStreamer.this.maxPendingNum || AppendStreamer.this.running == RunningState.LOOK_FOR_LEADER;
        }
    }

    private static class ExceptionAndRetry {
        private final Map<RaftPeerId, IOException> exceptionMap = new HashMap<RaftPeerId, IOException>();
        private final AtomicInteger retryTimes = new AtomicInteger(0);
        private final int maxRetryTimes;
        private final TimeDuration retryInterval;

        ExceptionAndRetry(RaftProperties prop) {
            this.maxRetryTimes = GrpcConfigKeys.OutputStream.retryTimes(prop);
            this.retryInterval = GrpcConfigKeys.OutputStream.retryInterval(prop);
        }

        void addException(RaftPeerId peer, IOException e) {
            this.exceptionMap.put(peer, e);
            this.retryTimes.incrementAndGet();
        }

        IOException getCombinedException() {
            return new IOException("Exceptions: " + this.exceptionMap);
        }

        boolean shouldRetry() {
            return this.retryTimes.get() <= this.maxRetryTimes;
        }
    }

    static enum RunningState {
        RUNNING,
        LOOK_FOR_LEADER,
        CLOSED,
        ERROR;

    }
}

